// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     BinaryStream.cpp
// CREATED:   3 Oct 2001 by Carl Muller
// MODIFIED:  3 Oct 2001 by Carl Muller
//
// This file implements a file class that supports fast reading and
// writing of binary files.
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "filename.h"
#include "binarystream.h"
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>

// ----------------------------------------------------------------------------
// Binary stream
//
// This class extends simple streams
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
long BinaryStream::ReadWord()
{
	long ch = ReadByte();
	return (ch << 8) | (long) (ReadByte());
}


// ----------------------------------------------------------------------------
long BinaryStream::ReadIntelWord()
{
	long ch = (long) ReadByte();
	return ch | (((long) ReadByte()) << 8);
}


// ----------------------------------------------------------------------------
long BinaryStream::ReadLong()
{
	long ch;
	ch = ((long) ReadByte()) << 24;
	ch |= ((long) ReadByte()) << 16;
	ch |= ((long) ReadByte()) << 8;
	ch |= ((long) ReadByte());
	return ch;

}


// ----------------------------------------------------------------------------
long BinaryStream::ReadIntelLong()
{
	long ch;
	ch = ((long) ReadByte());
	ch |= ((long) ReadByte()) << 8;
	ch |= ((long) ReadByte()) << 16;
	ch |= ((long) ReadByte()) << 24;
	return ch;
}

// ----------------------------------------------------------------------------
// Read in big data
void BinaryStream::ReadBytes(BYTE *buffer, long num)
{
	while ((num--) > 0)
		*(buffer++) = ReadByte();
}

// ----------------------------------------------------------------------------
// Read in a UTF-8 string (not very robust)
wstring BinaryStream::ReadString(const string& delimiter, const string& ignored, int* plastch)
{
	int ch, by;
	wstring result;
	for (;;)
	{
		by = ReadByte();
		if (by & 0x80)
		{
			if ((by & 0x40) == 0)
			{
				ch = 0; // Bad position for this char!
			}
			else if (by & 0x20) // 3 byte character
			{
				if (by & 0x10)
				{
					ch = 0; // We only support up to 16 bit chars
				}
				else
				{
					ch = (by & 0x0f) << 12;
					by = ReadByte();
					if ((by & 0xc0) != 0x80)
					{
						ch = 0; // Bad position for this char!
					}
					else
					{
						ch |= (by & 0x3f) << 6;
						by = ReadByte();
						if ((by & 0xc0) != 0x80)
						{
							ch = 0; // Bad position for this char!
						}
						else
						{
							ch |= (by & 0x3f);
						}
					}
				}
			}
			else // 2 byte character
			{
				ch = (by & 0x1f) << 6;
				by = ReadByte();
				if ((by & 0xc0) != 0x80)
				{
					ch = 0; // Bad position for this char!
				}
				else
				{
					ch |= (by & 0x3f);
				}
			}
		}
		else // 1 byte character
		{
			ch = by;
		}

		// Does this mean the string is complete?
		if (ch == 0)
			break;
		else if (delimiter.find_first_of(ch) != delimiter.npos)
			break;
		else if (ignored.find_first_of(ch) == delimiter.npos)
		{
			// Add the character to the string
			result += ch;
		}
	}
	if (plastch)
		*plastch = ch;
	return result;
}

// ----------------------------------------------------------------------------
// Write out a UTF8 string
// 00000000 0aaaaaaa -> 0aaaaaaa
// 00000bbb bbcccccc -> 110bbbbb 10cccccc
// ddddeeee eeffffff -> 1110dddd 10eeeeee 10ffffff
void BinaryStream::WriteString(const wstring& sValue) // Store as UTF8
{
	wstring::const_iterator i;
	for (i = sValue.begin(); i != sValue.end(); ++i)
	{
		int ch = *i;
		if (ch <= 0x007f)
		{
			WriteByte(ch);
		}
		else if (ch <= 0x0fff)
		{
			WriteByte(0xC0 | ((ch >> 6) & 0x1f));
			WriteByte(0x80 | (ch & 0x3f));
		}
		else
		{
			WriteByte(0xE0 | ((ch >> 12) & 0x0f));
			WriteByte(0x80 | ((ch >> 6) & 0x3f));
			WriteByte(0x80 | (ch & 0x3f));
		}
	}
}
