// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     BufferedFile.cpp
// CREATED:  26 Mar 2001 by Carl Muller
// MODIFIED:  3 Oct 2001 by Carl Muller
//
// This file implements a file class that supports fast reading and
// writing of binary files.
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "filename.h"
#include "binarystream.h"
#include "bufferedfile.h"
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>

// ----------------------------------------------------------------------------
// Buffered file
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Create a buffered file variable
BufferedFile::BufferedFile() :
	m_WriteMode(false),
	m_BufferSize(0),
	m_BufferIndex(0),
	m_pFileBuffer(NULL),
	m_FileBaseIndex(0),
	m_FileLength(0),
	m_FileHandle(-1)
{
}


// ----------------------------------------------------------------------------
// Destroy a buffered file variable
BufferedFile::~BufferedFile()
{
	Close();
}


// ----------------------------------------------------------------------------
// Primary file functions
// ----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
// Open an buffered file for writing
HRESULT BufferedFile::OpenWrite(const CFileName& filename, int newbufsize)
{
	if (m_FileHandle >= 0)
		Close();
	m_WriteMode = true;
	m_BufferSize = newbufsize;
	m_BufferIndex = 0;
	m_pFileBuffer = new BYTE[newbufsize];
	m_FileBaseIndex = 0;
	m_FileLength = 0;
	// _wopen is WinNT only, so use _open
	m_FileHandle = ::_open(NarrowString(filename.Get()).c_str(), O_CREAT | O_WRONLY | O_TRUNC | O_BINARY, S_IWRITE | S_IREAD);
	if (m_FileHandle < 0)
		return E_FAIL;
	return S_OK;
}


// ----------------------------------------------------------------------------
// Open an buffered file for reading
HRESULT BufferedFile::OpenRead(const CFileName& filename, int newbufsize)
{
	if (m_FileHandle >= 0)
		Close();
	m_WriteMode = false;
	m_BufferSize = newbufsize;
	m_BufferIndex = 0;
	m_pFileBuffer = new BYTE[newbufsize];
	m_FileBaseIndex = 0;
	m_FileHandle = ::_open(NarrowString(filename.Get()).c_str(), O_RDONLY | O_BINARY);
	if (m_FileHandle < 0)
		return E_FAIL;
	m_FileLength = _filelength(m_FileHandle);
	read(m_FileHandle, m_pFileBuffer, min(m_FileLength, m_BufferSize));
	return S_OK;
}


// ----------------------------------------------------------------------------
// Close a buffered file
HRESULT BufferedFile::Close()
{
	if (m_FileHandle >= 0)
	{
		if (m_WriteMode) // Flush the write buffer
		{
			if (m_BufferIndex > 0)
			{
				if (write(m_FileHandle, m_pFileBuffer, m_BufferIndex) < 0)
					return E_FAIL;
			}
		}
		if (close(m_FileHandle))
			return E_FAIL;
		m_FileHandle = -1;
	}
	if (m_pFileBuffer)
	{
		delete [] m_pFileBuffer;
		m_pFileBuffer = NULL;
	}
	return S_OK;
}


// ----------------------------------------------------------------------------
// Set position in a buffered file
HRESULT BufferedFile::SetPos(long pos)
{
	if (m_WriteMode)
	{
		// Flush the write buffer
		if (m_BufferIndex > 0)
		{
			if (write(m_FileHandle, m_pFileBuffer, m_BufferIndex) < 0)
				return E_FAIL;
		}
		// Goto correct position
		if (lseek(m_FileHandle, pos, SEEK_SET) < 0)
			return E_FAIL;
		m_FileBaseIndex = pos;
		m_BufferIndex = 0;
	}
	else
	{
		// Read mode
		if (pos >= m_FileLength) // End of file
		{
			m_FileBaseIndex = m_FileLength;
			m_BufferIndex = 0;
		}
		else if ((pos >= m_FileBaseIndex) && (pos < m_FileBaseIndex + m_BufferSize))
		{
			m_BufferIndex = pos - m_FileBaseIndex;
		}
		else
		{
			// Read in the new buffer
			if (lseek(m_FileHandle, pos, SEEK_SET) < 0)
				return E_FAIL;
			m_FileBaseIndex = pos;
			m_BufferIndex = 0;
			read(m_FileHandle, m_pFileBuffer, min(m_FileLength - m_FileBaseIndex, m_BufferSize));
		}
	}
	return S_OK;
}


// ----------------------------------------------------------------------------
// Write the buffer to the file
HRESULT BufferedFile::WriteBuffer()
{
	if (m_WriteMode)
	{
		if (m_BufferIndex > 0)
		{
			if (write(m_FileHandle, m_pFileBuffer, m_BufferIndex) < 0)
				return E_FAIL;
		}
	}
	m_FileBaseIndex += m_BufferSize;
	m_BufferIndex = 0;
	return S_OK;
}


// ----------------------------------------------------------------------------
// Read the buffer from the file
void BufferedFile::ReadBuffer()
{
	m_FileBaseIndex += m_BufferSize;
	read(m_FileHandle, m_pFileBuffer, min(m_FileLength - m_FileBaseIndex, m_BufferSize));
	m_BufferIndex = 0;
}


// ----------------------------------------------------------------------------
// Read in big data
void BufferedFile::ReadBytes(BYTE *buffer, long num)
{
	if (num >= 256) // Fast read
	{
		long nl = min(num, m_BufferSize - m_BufferIndex);
		if (nl > 0)
		{
			memcpy(buffer, &m_pFileBuffer[m_BufferIndex], nl);
		}
		if (num > nl)
		{
			read(m_FileHandle, &buffer[nl], num - nl);
			m_FileBaseIndex += num - nl;
			read(m_FileHandle, m_pFileBuffer, min(m_FileLength - m_FileBaseIndex, m_BufferSize));
			m_BufferIndex = 0;
		}
		else
			m_BufferIndex += nl;
	}
	else
	{
		// Small file
		while ((num--) > 0)
			*(buffer++) = ReadByte();
	}
}
