// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkDisplay.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 01 Apr 2001 by Carl Muller
//
// Support drawing operations
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "ChalkObjects.h"
#include "ChalkDisplay.h"

// ----------------------------------------------------------------------------
CString NarrowCString(const wstring& value)
{
	// !!! not mbcs compliant
	CString result;
	int i, n = value.size();
	for (i = 0; i < n; ++i)
	{
		result += value[i];
	}
	return result;
}

// ----------------------------------------------------------------------------
// ChalkDisplay
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkDisplay::ChalkDisplay()
{
	m_pOldBitmap = NULL;
	m_pMemDC = NULL;
	m_pBits = NULL;
	m_BitmapWidth = 0;
	m_BitmapHeight = 0;
	m_pOldFont = NULL;
	m_FontHeight = 0;
	m_FontColour = COLOUR_MAGENTA;
	m_Selected = false;
	m_Zoom = 1.0f;
	m_IsZoomed = false;
	m_ZoomMult = 1;
	m_ZoomDiv = 1;
}


// ----------------------------------------------------------------------------
ChalkDisplay::~ChalkDisplay()
{
	GdiFlush(); // Ensure the system is not writing to the bitmap itself

	// Destroy the attached memory device context
	if (m_pMemDC != NULL)
	{
		if (m_pOldFont)
			m_pMemDC->SelectObject(m_pOldFont);
		delete m_pMemDC;
		m_pMemDC = NULL;
		m_pOldFont = NULL;
	}
	// Destroy the attached font
	if (m_Font.m_hObject != NULL)
		m_Font.DeleteObject();
	// Destroy the attached bitmap
	if (m_Bitmap.m_hObject != NULL)
		m_Bitmap.DeleteObject();
}


// ----------------------------------------------------------------------------
void ChalkDisplay::SetBitmapSize(CSize size, CScrollView* pView)
{
	GdiFlush(); // Ensure the system is not writing to the bitmap itself

	// Delete existing bitmap
	if (m_pMemDC != NULL)
	{
		if (m_pOldFont)
			m_pMemDC->SelectObject(m_pOldFont);
		delete m_pMemDC;
		m_pMemDC = NULL;
		m_pOldFont = NULL;
	}
	if (m_Bitmap.m_hObject != NULL)
		m_Bitmap.DeleteObject();


	// Create a new bitmap
	m_BitmapWidth = size.cx;
	m_BitmapHeight = size.cy;
	if ((m_BitmapWidth < 1) || (m_BitmapHeight < 1))
		return; // No bitmap to view

	LPBITMAPINFO lpbi;

	// Fill in the BITMAPINFOHEADER
	VERIFY(lpbi = (LPBITMAPINFO) new BYTE[sizeof(BITMAPINFOHEADER) + (0 * sizeof(RGBQUAD))]);
	lpbi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	lpbi->bmiHeader.biWidth = size.cx;
	lpbi->bmiHeader.biHeight = -size.cy;
	lpbi->bmiHeader.biPlanes = 1;
	lpbi->bmiHeader.biBitCount = 32;
	lpbi->bmiHeader.biCompression = BI_RGB;
	lpbi->bmiHeader.biSizeImage = m_BitmapWidth * size.cy;
	lpbi->bmiHeader.biXPelsPerMeter = 0;
	lpbi->bmiHeader.biYPelsPerMeter = 0;
	lpbi->bmiHeader.biClrUsed = 0;
	lpbi->bmiHeader.biClrImportant = 0;

	// Create the DIB
	CWindowDC* pWinDC;
	VERIFY(pWinDC = new CWindowDC(pView));
	HBITMAP hBitmap = CreateDIBSection( pWinDC->m_hDC, lpbi,
		DIB_RGB_COLORS, reinterpret_cast<void **>(&m_pBits), NULL, 0 );
	delete [] reinterpret_cast<BYTE *>(lpbi);
	ASSERT(hBitmap != NULL);
	m_Bitmap.Attach( hBitmap );

	// Clear the bitmap to blank value
	for (int y = 0; y < size.cy; ++y )
	{
		PIXEL *pRow = (m_pBits + m_BitmapWidth * y);

		for (int x = 0; x < size.cx; ++x )
			*pRow++ = 0x00808080; // Grey
	}

	// Create the device context
	VERIFY(m_pMemDC = new CDC);
	VERIFY( m_pMemDC->CreateCompatibleDC(pWinDC) );
	delete pWinDC;
}

// ----------------------------------------------------------------------------
void ChalkDisplay::SetZoom(int zoommult, int zoomdiv)
{
	m_IsZoomed = (zoommult != zoomdiv);
	m_Zoom = (float) zoommult / (float) zoomdiv;
	m_ZoomMult = zoommult;
	m_ZoomDiv = zoomdiv;
}

// ----------------------------------------------------------------------------
void ChalkDisplay::StartPaintScene()
{
	m_pOldBitmap = m_pMemDC->SelectObject( &m_Bitmap );
}

// ----------------------------------------------------------------------------
void ChalkDisplay::EndPaintScene(CRect bounds, CDC *pDC)
{
	Flush(); // Ensure the system is not writing to the bitmap itself
	m_pMemDC->SelectObject( m_pOldBitmap );
	m_pOldBitmap = NULL;
	
	// Copy the buffer to the screen
	if (m_Bitmap.m_hObject != NULL)
	{
		int srcx, srcy, dstx, dsty;
		int dstsizex, dstsizey;

		if (pDC != NULL) // Win NT and Win 3.11 method
		{
			CDC* pMemDC;

			VERIFY(pMemDC = new CDC);
			VERIFY( pMemDC->CreateCompatibleDC(pDC) );
			pMemDC->SetMapMode(MM_TEXT);
			pMemDC->SetViewportOrg(0,0);
			pMemDC->SetWindowOrg(0,0);
			pMemDC->SelectClipRgn(NULL);
			pMemDC->IntersectClipRect(CRect(0, 0, bounds.Width(), bounds.Height()));
			CBitmap* pOldBitmap = pMemDC->SelectObject( &m_Bitmap );

			srcx = 0;
			srcy = 0;
			dstx = bounds.left;
			dsty = bounds.top;
			dstsizex = bounds.Width();
			dstsizey = bounds.Height();
			pDC->BitBlt( dstx, dsty, dstsizex, dstsizey, pMemDC,
				srcx, srcy, SRCCOPY );
			pMemDC->SelectObject(pOldBitmap);
			delete pMemDC;
		}
	}
}

// ----------------------------------------------------------------------------
// Return the size of the bitmap
CSize ChalkDisplay::GetBitmapSize()
{
	return CSize(m_BitmapWidth, m_BitmapHeight);
}

// ----------------------------------------------------------------------------
// Remember the bounds of the layer
void ChalkDisplay::SetBounds(CPoint offset, CRect bounds)
{
	m_Offset = offset;
	m_Bounds = bounds;
}

// ----------------------------------------------------------------------------
// Draw a solid box on the bitmap
// Take scrolling and scaling into account
void ChalkDisplay::DrawBox(CRect box, CRect border, PIXEL colour)
{
	int left, right, top, bottom;

	// Handle scaling, in world co-ordinates
	if (m_IsZoomed)
	{
		left = box.left * m_Zoom - m_Bounds.left;
		right = box.right * m_Zoom - m_Bounds.left;
		top = box.top * m_Zoom - m_Bounds.top;
		bottom = box.bottom * m_Zoom - m_Bounds.top;
	}
	else
	{
		left = box.left - m_Bounds.left;
		right = box.right - m_Bounds.left;
		top = box.top - m_Bounds.top;
		bottom = box.bottom - m_Bounds.top;
	}
	// Handle borders, in pixel co-ordinates
	left += border.left;
	top += border.top;
	right += border.right;
	bottom += border.bottom;
	// Clip to borders
	if (left < 0)
		left = 0;
	if (top < 0)
		top = 0;
	if (right > m_Bounds.Width())
		right = m_Bounds.Width();
	if (bottom > m_Bounds.Height())
		bottom = m_Bounds.Height();
	if ((left < right) && (top < bottom)) // Anything to do?
	{
		// Plot the solid box
		PIXEL *pRow = m_pBits + m_BitmapWidth * top + left;
		int h = bottom - top;
		int w = right - left;
		int n;
		PIXEL *pPixel;
		while (h--)
		{
			n = w;
			pPixel = pRow;
			while (n--)
				*pPixel++ = colour;
			pRow += m_BitmapWidth;
		}
	}
}

// ----------------------------------------------------------------------------
// Draw a line across the bitmap
void ChalkDisplay::DrawLine(CPoint pt1, CPoint pt2, PIXEL colour)
{
	int x1 = pt1.x * m_Zoom - m_Bounds.left;
	int x2 = pt2.x * m_Zoom - m_Bounds.left;
	int y1 = pt1.y * m_Zoom - m_Bounds.top;
	int y2 = pt2.y * m_Zoom - m_Bounds.top;
	int clipx1 = 0;
	int clipy1 = 0;
	int clipx2 = m_Bounds.Width();
	int clipy2 = m_Bounds.Height();

	int ix, iy, i, inc, x, y, dx, dy, plotx, ploty;
	bool plot;

	dx = x2 - x1;
	dy = y2 - y1;
	ix = abs(dx);
	iy = abs(dy);
	inc = max(ix, iy);
	if (dx < 0)
		dx = -1;
	else if (dx > 0)
		dx = 1;
	if (dy < 0)
		dy = -1;
	else if (dy > 0)
		dy = 1;

	plotx = x1;
	ploty = y1;
	x = y = inc / 2;

	// Draw appropriate lines
	if ((plotx >= clipx1) && (plotx < clipx2) && (ploty >= clipy1) && (ploty < clipy2))
		m_pBits[ploty * m_BitmapWidth + plotx] = colour;
	for (i = 0; i < inc; ++i)
	{
		x += ix;
		y += iy;
		plot = false;
		if (x > inc)
		{
			plot = true;
			x -= inc;
			plotx += dx;
		}
		if (y > inc)
		{
			plot = true;
			y -= inc;
			ploty += dy;
		}
		if (plot)
			if ((plotx >= clipx1) && (plotx < clipx2) && (ploty >= clipy1) && (ploty < clipy2))
				m_pBits[ploty * m_BitmapWidth + plotx] = colour;
	}
}


// ----------------------------------------------------------------------------
// Draw some text on the bitmap
void ChalkDisplay::DrawText(CPoint pt, int FontHeight, PIXEL colour, const wstring& TextString)
{
	int len = TextString.size();
	if (len > 0)
	{
		if ((m_FontHeight != FontHeight) && (m_FontHeight == 0))
		{
			m_Font.CreatePointFont(FontHeight * 12, _T("Arial"), m_pMemDC);
			m_FontHeight = FontHeight;
		}
		if (!m_Selected)
		{
			m_pMemDC->SetBkMode(TRANSPARENT);
			m_pMemDC->SetTextAlign(TA_BASELINE | TA_LEFT);
			m_pOldFont = m_pMemDC->SelectObject(&m_Font);
			m_Selected = true;
		}
		if (m_FontColour != colour)
		{
			// Convert colour into windows format
			int r = (colour >> 16) & 0xff;
			int g = (colour >> 8) & 0xff;
			int b = (colour) & 0xff;
			COLORREF value = r | (g << 8) | (b << 16);
			m_pMemDC->SetTextColor(value);
			m_FontColour = colour;
		}
		int x = pt.x * m_Zoom - m_Bounds.left;
		int y = pt.y * m_Zoom - m_Bounds.top;
		#ifdef _UNICODE
		m_pMemDC->TextOut(x, y, TextString.c_str(), len);
		#else
		CString str = NarrowCString(TextString);
		m_pMemDC->TextOut(x, y, str, len);
		#endif
	}
}


// ----------------------------------------------------------------------------
// Finish off text display
void ChalkDisplay::Flush()
{
	GdiFlush();
	if (m_Selected && m_pOldFont)
		m_pMemDC->SelectObject(m_pOldFont);
	m_pOldFont = NULL;
	m_Selected = false;
}
