// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkFont.cpp
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Defines the game map objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"
#include "chalkdisplay.h"

// ----------------------------------------------------------------------------
// ChalkFont
// A character set, or a collection of icons
// ----------------------------------------------------------------------------
const int CHARSIZE = 8;


// ----------------------------------------------------------------------------
ChalkFont::ChalkFont() : ChalkMapBase(GUID_VOID, GUID_FONT)
{
	// Override base class - Most character sets are 8*8 pixels
	m_Grid1X = 1;
	m_Grid1Y = 1;
	m_Grid2X = CHARSIZE;
	m_Grid2Y = CHARSIZE;
}


// ----------------------------------------------------------------------------
HRESULT ChalkFont::Write(ChalkStream& os) const
{
	os.WriteElement1("Font");
	os.WriteGUID("systemtype", m_SystemType);
	os.WriteInteger("grid1x", m_Grid1X);
	os.WriteInteger("grid1y", m_Grid1Y);
	os.WriteInteger("grid2x", m_Grid2X);
	os.WriteInteger("grid2y", m_Grid2Y);

	os.WriteElement("Characters", m_Data.size());
	vector<ChalkCharacter>::const_iterator i;
	for (i = m_Data.begin(); i != m_Data.end(); ++i)
	{
		i->Write(os);
	}
	os.WriteEndElement("Characters");
	os.WriteEndElement("Font");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::Read(ChalkStream& is)
{
	is.ReadElement1("Font");
	m_SystemType = is.ReadGUID("systemtype");
	m_Grid1X = is.ReadInteger("grid1x");
	m_Grid1Y = is.ReadInteger("grid1y");
	m_Grid2X = is.ReadInteger("grid2x");
	m_Grid2Y = is.ReadInteger("grid2y");

	int nChars = is.ReadElement("Characters");
	m_Data.clear();
	for (int i = 0; i < nChars; ++i)
	{
		ChalkCharacter c;
		c.Read(is);
		m_Data.push_back(c);
	}
	is.ReadEndElement("Characters");
	is.ReadEndElement("Font");
	return S_OK;
}

// ----------------------------------------------------------------------------
// Does this class support this file extension?
bool ChalkFont::SupportsFile(const CFileName& filename)
{
	if (filename.HasExt(FILEEXTENSION_CHR))
		return true;
	if (filename.HasExt(FILEEXTENSION_SET))
		return true;
	if (filename.HasExt(FILEEXTENSION_GBC))
		return true;
	if (filename.HasExt(FILEEXTENSION_SF2))
		return true;
	if (filename.HasExt(FILEEXTENSION_SF4))
		return true;
	if (filename.HasExt(FILEEXTENSION_MCH))
		return true;
	if (filename.HasExt(FILEEXTENSION_GCH))
		return true;
	if (filename.HasExt(FILEEXTENSION_SCH))
		return true;
	if (filename.HasExt(FILEEXTENSION_SF8))
		return true;
	if (filename.HasExt(FILEEXTENSION_SFB))
		return true;
	return false;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadFile(const CFileName& filename)
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return ReadXML(filename);
	if (filename.HasExt(FILEEXTENSION_CHR))
		return ReadCHR(filename);
	if (filename.HasExt(FILEEXTENSION_SET))
		return ReadSET(filename);
	if (filename.HasExt(FILEEXTENSION_GBC))
		return ReadGBC(filename);
	if (filename.HasExt(FILEEXTENSION_SF2))
		return ReadSF2(filename);
	if (filename.HasExt(FILEEXTENSION_SF4))
		return ReadSF4(filename);
	if (filename.HasExt(FILEEXTENSION_MCH))
		return ReadMCH(filename);
	if (filename.HasExt(FILEEXTENSION_GCH))
		return ReadGCH(filename);
	if (filename.HasExt(FILEEXTENSION_SCH))
		return ReadSCH(filename);
	if (filename.HasExt(FILEEXTENSION_SF8))
		return ReadSF8(filename);
	if (filename.HasExt(FILEEXTENSION_SFB))
		return ReadSFB(filename);
	if (filename.HasExt(FILEEXTENSION_BOX))
		return ReadBOX(filename);
	if (ChalkBitmap::SupportsFile(filename))
		return ReadBitmap(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteFile(const CFileName& filename) const
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return WriteXML(filename);
	if (filename.HasExt(FILEEXTENSION_CHR))
		return WriteCHR(filename);
	if (filename.HasExt(FILEEXTENSION_SET))
		return WriteSET(filename);
	if (filename.HasExt(FILEEXTENSION_GBC))
		return WriteGBC(filename);
	if (filename.HasExt(FILEEXTENSION_SF2))
		return WriteSF2(filename);
	if (filename.HasExt(FILEEXTENSION_SF4))
		return WriteSF4(filename);
	if (filename.HasExt(FILEEXTENSION_MCH))
		return WriteMCH(filename);
	if (filename.HasExt(FILEEXTENSION_GCH))
		return WriteGCH(filename);
	if (filename.HasExt(FILEEXTENSION_SCH))
		return WriteSCH(filename);
	if (filename.HasExt(FILEEXTENSION_SF8))
		return WriteSF8(filename);
	if (filename.HasExt(FILEEXTENSION_SFB))
		return WriteSFB(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
// Read in various file formats
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Load in a boxed bitmap
HRESULT ChalkFont::ReadBOX(const CFileName& filename, ChalkPalette* pPalette, bool merge)
{
	int fontoffset = 0;
	if (merge)
		fontoffset = m_Data.size();

	m_FileName = filename; // Remember my name
	ChalkAnimation boxfile;
	ChalkBitmap bitmapfile;
	boxfile.ReadBOX(filename);
	CFileName kidname = boxfile.m_KidName;
	if (kidname.IsValid() &&
		ChalkBitmap::SupportsFile(kidname))
	{
		bitmapfile.ReadFile(kidname);
		if (pPalette)
			*pPalette = bitmapfile.m_Palette;
		int i, num, x, y;

		// Copy all the sprites to the new font
		ChalkFrame& frame = boxfile.m_Frames[0];
		num = frame.m_Boxes.size();
		int bitmapwidth = bitmapfile.m_Character.m_Width;
		int bitmapheight = bitmapfile.m_Character.m_Height;
		m_Data.resize(fontoffset + num);
		for (i = 0; i < num; ++i)
		{
			ChalkBox& box = frame.m_Boxes[i];
			ChalkCharacter& character = m_Data[fontoffset + i];
			character.SetSize(box.m_Width, box.m_Height,
				bitmapfile.m_Character.m_NumPlanes);
			ChalkColourIndex *pData = character.m_Data;
			int x1 = box.m_Xpos;
			int y1 = box.m_Ypos;
			int x2 = x1 + box.m_Width;
			int y2 = y1 + box.m_Height;
			x1 = max(0, x1);
			y1 = max(0, y1);
			x2 = min(bitmapwidth, x2);
			y2 = min(bitmapheight, y2);
			for (y = y1; y < y2; ++y)
			{
				const ChalkColourIndex *pSrc =
					&bitmapfile.m_Character.m_Data[bitmapwidth * y + x1];
				for (x = x1; x < x2; ++x)
					*pData++ = *pSrc++;
			}
		}
	}

	return S_OK;
}


// ----------------------------------------------------------------------------
// Load in a bitmap
HRESULT ChalkFont::ReadBitmap(const CFileName& filename, ChalkPalette* pPalette, bool merge)
{
	int fontoffset = 0;
	if (merge)
		fontoffset = m_Data.size();

	m_FileName = filename; // Remember my name
	ChalkBitmap bitmap;
	HRESULT hr = bitmap.ReadFile(filename);
	if (FAILED(hr))
		return hr;

	// Copy the palette back out
	if (pPalette)
		*pPalette = bitmap.m_Palette;

	// Create a bitmap from the font
	int width = bitmap.m_Character.m_Width;
	int height = bitmap.m_Character.m_Height;
	int numplanes = bitmap.m_Character.m_NumPlanes;
	int gx = bitmap.m_GridX;
	int gy = bitmap.m_GridY;
	int numblocks = (width / gx);
	int numchars = numblocks * (height / gy);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(fontoffset + numchars);
	m_Grid2X = gx;
	m_Grid2Y = gy;

	// Copy characters into the bitmap
	int i, x, y, xpos, ypos;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[fontoffset + i];
		character.SetSize(gx, gy, numplanes);
		xpos = gx * (i % numblocks);
		ypos = gy * (i / numblocks);
		for (y = 0; y < gy; ++y)
		{
			const ChalkColourIndex *pSrc = bitmap.m_Character.m_Data + width * (ypos + y) + xpos;
			ChalkColourIndex *pDst = character.m_Data + gx * y;
			for (x = 0; x < gx; ++x)
				*pDst++ = *pSrc++;
		}
	}

	return hr;
}


// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Draw a bitmap
void ChalkFont::DrawBlock(ChalkDisplay& display, CRect rt,
	CPoint pt, ChalkMapBlock& mapblock, ChalkPalette& Palette) const
{
	ChalkBlockIndex block = mapblock.m_BlockIndex;
	ChalkColourIndex offset = mapblock.m_PaletteOffset;
	bool xflip = (mapblock.m_Special & SPECIAL_XFLIP) ? true : false;
	bool yflip = (mapblock.m_Special & SPECIAL_YFLIP) ? true : false;
	bool trans = (mapblock.m_Special & SPECIAL_TRANSPARENT) ? true : false;
	if (block >= m_Data.size())
		return; // Safety check

	int bitmapwidth = display.GetBitmapSize().cx;
	int bitmapheight = display.GetBitmapSize().cy;
	int zoommult = display.GetZoomMult();
	int zoomdiv = display.GetZoomDiv();
	const ChalkCharacter& character = m_Data[block];
	const ChalkColourIndex *pData = character.m_Data; // Raw data
	int srcwidth = character.m_Width;
	int srcheight = character.m_Height;
	int dstwidth = srcwidth * zoommult / zoomdiv;
	int dstheight = srcheight * zoommult / zoomdiv;
	pt.x = pt.x * zoommult / zoomdiv;
	pt.y = pt.y * zoommult / zoomdiv;

	// Display the bitmap
	int maxx = rt.Width();
	int maxy = rt.Height();
	int x, y, xpos, ypos, xposbase, yposbase;
	int palettesize = Palette.m_NumColours;
	if (palettesize < 2) // True colour block
	{
		for (y = 0, yposbase = pt.y - rt.top; y < srcheight; y += zoomdiv, yposbase += zoommult)
		{
			for (ypos = yposbase; ypos < yposbase + zoommult; ++ypos)
			{
				if ((ypos >= 0) && (ypos < maxy))
				{
					PIXEL *pRow = display.m_pBits + bitmapwidth * ypos;
					const ChalkColourIndex *pSrc;
					if (yflip)
						pSrc = pData + srcwidth * (srcheight - 1 - y);
					else
						pSrc = pData + srcwidth * y;
					if (xflip)
					{
						xposbase = pt.x - rt.left;
						for (x = 0; x < srcwidth; x += zoomdiv, xposbase += zoommult)
						{
							PIXEL pixel = pSrc[srcwidth - 1 - x];
							for (xpos = xposbase; xpos < xposbase + zoommult; ++xpos)
								if ((xpos >= 0) && (xpos < maxx))
									pRow[xpos] = pixel;
						}
					}
					else
					{
						xposbase = pt.x - rt.left;
						for (x = 0; x < srcwidth; x += zoomdiv, xposbase += zoommult)
						{
							PIXEL pixel = pSrc[x];
							for (xpos = xposbase; xpos < xposbase + zoommult; ++xpos)
								if ((xpos >= 0) && (xpos < maxx))
									pRow[xpos] = pixel;
						}
					}
				}
			}
		}
	}
	else if (trans) // Paletted block (transparent)
	{
		PIXEL *pPalette = Palette.m_Data;
		for (y = 0, yposbase = pt.y - rt.top; y < srcheight; y += zoomdiv, yposbase += zoommult)
		{
			for (ypos = yposbase; ypos < yposbase + zoommult; ++ypos)
			{
				if ((ypos >= 0) && (ypos < maxy))
				{
					PIXEL *pRow = display.m_pBits + bitmapwidth * ypos;
					const ChalkColourIndex *pSrc;
					if (yflip)
						pSrc = pData + srcwidth * (srcheight - 1 - y);
					else
						pSrc = pData + srcwidth * y;
					if (xflip)
					{
						xposbase = pt.x - rt.left;
						for (x = 0; x < srcwidth; x += zoomdiv, xposbase += zoommult)
						{
							ChalkColourIndex colour = pSrc[srcwidth - 1 - x];
							if (colour)
							{
								PIXEL pixel = pPalette[colour + offset];
								for (xpos = xposbase; xpos < xposbase + zoommult; ++xpos)
									if ((xpos >= 0) && (xpos < maxx))
										pRow[xpos] = pixel;
							}
						}
					}
					else
					{
						xposbase = pt.x - rt.left;
						for (x = 0; x < srcwidth; x += zoomdiv, xposbase += zoommult)
						{
							ChalkColourIndex colour = pSrc[x];
							if (colour)
							{
								PIXEL pixel = pPalette[colour + offset];
								for (xpos = xposbase; xpos < xposbase + zoommult; ++xpos)
									if ((xpos >= 0) && (xpos < maxx))
										pRow[xpos] = pixel;
							}
						}
					}
				}
			}
		}
	}

	else // Paletted block (solid)
	{
		PIXEL *pPalette = Palette.m_Data;
		for (y = 0, yposbase = pt.y - rt.top; y < srcheight; y += zoomdiv, yposbase += zoommult)
		{
			for (ypos = yposbase; ypos < yposbase + zoommult; ++ypos)
			{
				if ((ypos >= 0) && (ypos < maxy))
				{
					PIXEL *pRow = display.m_pBits + bitmapwidth * ypos;
					const ChalkColourIndex *pSrc;
					if (yflip)
						pSrc = pData + srcwidth * (srcheight - 1 - y);
					else
						pSrc = pData + srcwidth * y;
					if (xflip)
					{
						xposbase = pt.x - rt.left;
						for (x = 0; x < srcwidth; x += zoomdiv, xposbase += zoommult)
						{
							PIXEL pixel = pPalette[pSrc[srcwidth - 1 - x] + offset];
							for (xpos = xposbase; xpos < xposbase + zoommult; ++xpos)
								if ((xpos >= 0) && (xpos < maxx))
									pRow[xpos] = pixel;
						}
					}
					else
					{
						xposbase = pt.x - rt.left;
						for (x = 0; x < srcwidth; x += zoomdiv, xposbase += zoommult)
						{
							PIXEL pixel = pPalette[pSrc[x] + offset];
							for (xpos = xposbase; xpos < xposbase + zoommult; ++xpos)
								if ((xpos >= 0) && (xpos < maxx))
									pRow[xpos] = pixel;
						}
					}
				}
			}
		}
	}
}

// ----------------------------------------------------------------------------
// The world's smallest font (4*8 pixels)
static const DWORD HexFont[] = 
{
	0x0EAAAAAE, // 0
	0x0C44444E, // 1
	0x0EA2488E, // 2
	0x0EA242AE, // 3
	0x0AAAE222, // 4
	0x0E88E22E, // 5
	0x0E88EAAE, // 6
	0x0E224444, // 7
	0x0EAAEAAE, // 8
	0x0EAAE22E, // 9
	0x0EAAEAAA, // A
	0x0CAACAAC, // B
	0x0EA888AE, // C
	0x0CAAAAAC, // D
	0x0E88E88E, // E
	0x0E88E888, // F
};

// ----------------------------------------------------------------------------
// Generate an 8*8 hexadecimal font
// This is used for attributes when no attribute font is known
void ChalkFont::CreateHexFont()
{
	m_SystemType = GUID_SYSTEM_C64;
	int numchars = 256;
	m_Data.resize(numchars);
	int i, y;
	DWORD digit1, digit2;
	int val1, val2;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 1);
		ChalkColourIndex *pData = character.m_Data;
		if (i == 0) // First hex char is blank
			digit1 = digit2 = 0;
		else
		{
			digit1 = HexFont[(i >> 4) & 15];
			digit2 = HexFont[i & 15];
		}
		for (y = 0; y < CHARSIZE; ++y)
		{
			val1 = digit1 >> (4 * (CHARSIZE - 1 - y));
			val2 = digit2 >> (4 * (CHARSIZE - 1 - y));
			*pData++ = (val1 & 8) ? 1 : 0;
			*pData++ = (val1 & 4) ? 1 : 0;
			*pData++ = (val1 & 2) ? 1 : 0;
			*pData++ = (val1 & 1) ? 1 : 0;
			*pData++ = (val2 & 8) ? 1 : 0;
			*pData++ = (val2 & 4) ? 1 : 0;
			*pData++ = (val2 & 2) ? 1 : 0;
			*pData++ = (val2 & 1) ? 1 : 0;
		}
	}
}
