// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkFrame.cpp
// CREATED:   5 Jul 2000 by Carl Muller
// MODIFIED:  6 Jul 2000 by Carl Muller
//
// Implements miscellaneous game objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"


// ----------------------------------------------------------------------------
// ChalkFrame
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkFrame::ChalkFrame()  : ChalkObject(GUID_VOID, GUID_FRAME)
{
}

// ----------------------------------------------------------------------------
HRESULT ChalkFrame::Write(ChalkStream& os) const
{
	os.WriteElement1("Frame");
	{
		os.WriteElement("Sprites", m_Sprites.size());
		vector<ChalkSprite>::const_iterator i;
		for (i = m_Sprites.begin(); i != m_Sprites.end(); ++i)
		{
			i->Write(os);
		}
		os.WriteEndElement("Sprites");
	}
	{
		os.WriteElement("Boxes", m_Boxes.size());
		vector<ChalkBox>::const_iterator i;
		for (i = m_Boxes.begin(); i != m_Boxes.end(); ++i)
		{
			i->Write(os);
		}
		os.WriteEndElement("Boxes");
	}
	{
		os.WriteElement("Paths", m_Paths.size());
		vector<ChalkPath>::const_iterator i;
		for (i = m_Paths.begin(); i != m_Paths.end(); ++i)
		{
			i->Write(os);
		}
		os.WriteEndElement("Paths");
	}
	os.WriteEndElement("Frame");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFrame::Read(ChalkStream& is)
{
	is.ReadElement1("Frame");
	{
		int n = is.ReadElement("Sprites");
		m_Sprites.clear();
		for (int i = 0; i < n; ++i)
		{
			ChalkSprite item;
			item.Read(is);
			m_Sprites.push_back(item);
		}
		is.ReadEndElement("Sprites");
	}
	{
		int n = is.ReadElement("Boxes");
		m_Boxes.clear();
		for (int i = 0; i < n; ++i)
		{
			ChalkBox item;
			item.Read(is);
			m_Boxes.push_back(item);
		}
		is.ReadEndElement("Boxes");
	}
	{
		int n = is.ReadElement("Paths");
		m_Paths.clear();
		for (int i = 0; i < n; ++i)
		{
			ChalkPath item;
			item.Read(is);
			m_Paths.push_back(item);
		}
		is.ReadEndElement("Paths");
	}
	is.ReadEndElement("Frame");
	return S_OK;
}


// ----------------------------------------------------------------------------
// Select all objects or none
void ChalkFrame::SelectAll(bool bSprites, bool bBoxes, bool bPaths)
{
	int i;
	for (i = 0; i < m_Sprites.size(); ++i)
		m_Sprites[i].m_Selected = bSprites;
	for (i = 0; i < m_Boxes.size(); ++i)
		m_Boxes[i].m_Selected = bBoxes;
	for (i = 0; i < m_Paths.size(); ++i)
		m_Paths[i].m_Selected = bPaths;
}

// ----------------------------------------------------------------------------
// Return the first selected object
ChalkObject* ChalkFrame::GetFirstSelected()
{
	int i;
	for (i = 0; i < m_Sprites.size(); ++i)
		if (m_Sprites[i].m_Selected)
			return &m_Sprites[i];
	for (i = 0; i < m_Boxes.size(); ++i)
		if (m_Boxes[i].m_Selected)
			return &m_Boxes[i];
	for (i = 0; i < m_Paths.size(); ++i)
		if (m_Paths[i].m_Selected)
			return &m_Paths[i];
	return NULL;
}

// ----------------------------------------------------------------------------
// What should be selected next? (shift-tab order)
ChalkObject* ChalkFrame::GetPrevSelected()
{
	ChalkObject* current = GetFirstSelected();
	if (current == NULL)
	{
		if (m_Sprites.size() > 0)
			return &m_Sprites[0];
		if (m_Boxes.size() > 0)
			return &m_Boxes[0];
		if (m_Paths.size() > 0)
			return &m_Paths[0];
		return NULL;
	}
	int i, num;
	if (current->m_TypeId == GUID_SPRITE)
	{
		num = m_Sprites.size();
		for (i = 0; i < num; ++i)
			if (m_Sprites[i].m_Selected)
				return &m_Sprites[(i == 0) ? num - 1 : i - 1];
	}
	else if (current->m_TypeId == GUID_BOX)
	{
		num = m_Boxes.size();
		for (i = 0; i < num; ++i)
			if (m_Boxes[i].m_Selected)
				return &m_Boxes[(i == 0) ? num - 1 : i - 1];
	}
	else if (current->m_TypeId == GUID_PATH)
	{
		num = m_Paths.size();
		for (i = 0; i < num; ++i)
			if (m_Paths[i].m_Selected)
				return &m_Paths[(i == 0) ? num - 1 : i - 1];
	}
	return NULL;
}

// ----------------------------------------------------------------------------
// What should be selected next? (tab order)
ChalkObject* ChalkFrame::GetNextSelected()
{
	ChalkObject* current = GetFirstSelected();
	if (current == NULL)
	{
		if (m_Sprites.size() > 0)
			return &m_Sprites[0];
		if (m_Boxes.size() > 0)
			return &m_Boxes[0];
		if (m_Paths.size() > 0)
			return &m_Paths[0];
		return NULL;
	}
	int i, num;
	if (current->m_TypeId == GUID_SPRITE)
	{
		num = m_Sprites.size();
		for (i = 0; i < num; ++i)
			if (m_Sprites[i].m_Selected)
				return &m_Sprites[(i == num - 1) ? 0 : i + 1];
	}
	else if (current->m_TypeId == GUID_BOX)
	{
		num = m_Boxes.size();
		for (i = 0; i < num; ++i)
			if (m_Boxes[i].m_Selected)
				return &m_Boxes[(i == num - 1) ? 0 : i + 1];
	}
	else if (current->m_TypeId == GUID_PATH)
	{
		num = m_Paths.size();
		for (i = 0; i < num; ++i)
			if (m_Paths[i].m_Selected)
				return &m_Paths[(i == num - 1) ? 0 : i + 1];
	}
	return NULL;
}

// ----------------------------------------------------------------------------
// What should be selected next? (click)
ChalkObject* ChalkFrame::GetNextSelected(ChalkFont& SpriteFont, CPoint pt,
	bool bBoxOnly)
{
	ChalkObject* current = GetFirstSelected();
	ChalkObject* best = NULL;
	int bestscore = 0;
	int score;
	int i;

	// Check selection of boxes
	for (i = 0; i < m_Boxes.size(); ++i)
	{
		ChalkBox& box = m_Boxes[i];
		if ((pt.x >= box.m_Xpos) && (pt.x < box.m_Xpos + box.m_Width) &&
			(pt.y >= box.m_Ypos) && (pt.y < box.m_Ypos + box.m_Height))
		{
			// This click was within the box. Now rank it
			score = 10;
			if (current && (current->m_TypeId == box.m_TypeId))
			{
				score += 50;
				if (&box > current)
					score += 20;
				else if (&box < current)
					score += 10;
			}
			if (score > bestscore)
			{
				best = &box;
				bestscore = score;
			}
		}
	}
	if (bBoxOnly)
		return best;

	// Check selection of sprites
	for (i = 0; i < m_Sprites.size(); ++i)
	{
		ChalkSprite& sprite = m_Sprites[i];
		if (sprite.m_ObjDefn < SpriteFont.m_Data.size())
		{
			const ChalkCharacter& spdata = SpriteFont.m_Data[sprite.m_ObjDefn];
			int x1 = sprite.m_Xpos - spdata.m_OffsetX;
			int y1 = sprite.m_Ypos - spdata.m_OffsetY;
			int x2 = x1 + spdata.m_Width;
			int y2 = y1 + spdata.m_Height;
			if ((pt.x >= x1) && (pt.x < x2) &&
				(pt.y >= y1) && (pt.y < y2))
			{
				int x = pt.x - x1;
				int y = pt.y - y1;
				if (sprite.m_Special & SPECIAL_XFLIP)
					x = (spdata.m_Width - 1) - x;
				if (sprite.m_Special & SPECIAL_YFLIP)
					y = (spdata.m_Height - 1) - y;
				ChalkColourIndex col = spdata.m_Data[x + spdata.m_Width * y];
				if (col) // Not transparent pixel
				{
					// This click was within the sprite. Now rank it
					score = 10;
					if (current && (current->m_TypeId == sprite.m_TypeId))
					{
						score += 50;
						if (&sprite > current)
							score += 20;
						else if (&sprite < current)
							score += 10;
					}
					if (score > bestscore)
					{
						best = &sprite;
						bestscore = score;
					}
				}
			}
		}
	}

	// Check selection of paths
	// ...

	return best;
}

// ----------------------------------------------------------------------------
// What is the extent of selected items? (used for flipping)
CRect ChalkFrame::GetSelectedExtent() const
{
	CRect rt;
	bool bFirst = true;
	int x, y;
	int i, j;

	for (i = 0; i < m_Sprites.size(); ++i)
	{
		const ChalkSprite& sprite = m_Sprites[i];
		if (sprite.m_Selected)
		{
			x = sprite.m_Xpos;
			y = sprite.m_Ypos;
			if (bFirst)
			{
				bFirst = false;
				rt = CRect(x, y, x, y);
			}
			else
			{
				rt.left = min(rt.left, x);
				rt.top = min(rt.top, y);
				rt.right = max(rt.right, x);
				rt.bottom = max(rt.bottom, y);
			}
		}
	}
	for (i = 0; i < m_Boxes.size(); ++i)
	{
		const ChalkBox& box = m_Boxes[i];
		if (box.m_Selected)
		{
			x = box.m_Xpos + box.m_OffsetX;
			y = box.m_Ypos + box.m_OffsetY;
			if (bFirst)
			{
				bFirst = false;
				rt = CRect(x, y, x, y);
			}
			else
			{
				rt.left = min(rt.left, x);
				rt.top = min(rt.top, y);
				rt.right = max(rt.right, x);
				rt.bottom = max(rt.bottom, y);
			}
		}
	}
	for (i = 0; i < m_Paths.size(); ++i)
	{
		const ChalkPath& path = m_Paths[i];
		if (path.m_Selected)
		{
			for (j = 0; j < path.m_Data.size(); ++j)
			{
				const ChalkPoint& point = path.m_Data[j];
				x = point.m_Xpos;
				y = point.m_Ypos;
				if (bFirst)
				{
					bFirst = false;
					rt = CRect(x, y, x, y);
				}
				else
				{
					rt.left = min(rt.left, x);
					rt.top = min(rt.top, y);
					rt.right = max(rt.right, x);
					rt.bottom = max(rt.bottom, y);
				}
			}
		}
	}
	return rt;
}
