// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkLayerAnimation.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 16 Jul 2000 by Carl Muller
//
// Implements the user interface of the animation (FRK/BOX) layer
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkDoc.h"
#include "chalkView.h"
#include "PropSheetBase.h"
#include "resource.h"

const int TEXT_FONTHEIGHT = 8;
const int TEXT_OFFSETY = 9;

// ----------------------------------------------------------------------------
// ChalkLayerAnimation
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Create the layer, customising its attributes
ChalkLayerAnimation::ChalkLayerAnimation(CChalkDoc& Doc) :
	ChalkLayer(Doc)
{
	m_TypeName = _T("Animation");
	m_pBaseData = &m_Data;
	m_Solid = false;
	// What actions can this type of layer take?
	m_DrawModesSupported = DrawMode(drawmode_select | drawmode_sprite |
		drawmode_box | drawmode_offset
		/* ... | drawmode_path | drawmode_replaceall | drawmode_lasso */ );
	m_DrawAttributesSupported = DrawAttributes(drawattr_grid | drawattr_showtext); // What modes do I support?
	m_DrawCommandsSupported = DrawCommands(
		drawcmd_edit_delete | drawcmd_edit_selectall |
		drawcmd_flipx | drawcmd_flipy |
		drawcmd_align | drawcmd_move | drawcmd_purge | drawcmd_sort |
		drawcmd_increment | drawcmd_decrement |
		drawcmd_frame_properties | drawcmd_frame_insert | drawcmd_frame_delete |
		drawcmd_layer_properties);
}

// ----------------------------------------------------------------------------
// Destroy the layer
ChalkLayerAnimation::~ChalkLayerAnimation()
{
}

// ----------------------------------------------------------------------------
// Standard box border colours
const PIXEL BoxColours[8] = {
	COLOUR_CYAN,
	COLOUR_RED,
	COLOUR_WHITE,
	COLOUR_BLUE,
	COLOUR_GREEN,
	COLOUR_CYAN,
	COLOUR_RED,
	COLOUR_WHITE,
};

// Standard box offset colours
const PIXEL OffsetColours[8] = {
	COLOUR_DKCYAN,
	COLOUR_DKRED,
	COLOUR_GREY,
	COLOUR_DKBLUE,
	COLOUR_DKGREEN,
	COLOUR_DKCYAN,
	COLOUR_DKRED,
	COLOUR_GREY,
};

// ----------------------------------------------------------------------------
// Draw the animation layer, which is a set of boxes, paths and sprites
// Generally, this will be on top of a more solid layer such as a map
// or a bitmap
void ChalkLayerAnimation::Draw(ChalkDisplay& display, CRect rt)
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i, num;

	// Draw all the sprites
	ChalkPalette& palette = m_Data.m_Palette;
	ChalkFont& font = m_Data.m_Font;
	ChalkMapBlock tempblock;
	tempblock.m_Attribute = 0;
	num = frame.m_Sprites.size();
	for (i = 0; i < num; ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];
		if (sprite.m_ObjDefn < font.m_Data.size())
		{
			ChalkCharacter& spdata = font.m_Data[sprite.m_ObjDefn];
			tempblock.m_BlockIndex = sprite.m_ObjDefn;
			tempblock.m_PaletteOffset = sprite.m_PaletteOffset;
			tempblock.m_Special = SPECIAL_TRANSPARENT | sprite.m_Special;
			int x1 = sprite.m_Xpos - spdata.m_OffsetX;
			int y1 = sprite.m_Ypos - spdata.m_OffsetY;
			int x2 = x1 + spdata.m_Width;
			int y2 = y1 + spdata.m_Height;
			font.DrawBlock(display, rt, CPoint(x1,y1), tempblock, palette);
			if (sprite.m_Selected)
			{
				display.DrawBox(CRect(x1, y1, x1, y1), CRect(-1, -1, 1, 1), COLOUR_WHITE);
				display.DrawBox(CRect(x1, y2, x1, y2), CRect(-1, -1, 1, 1), COLOUR_WHITE);
				display.DrawBox(CRect(x2, y1, x2, y1), CRect(-1, -1, 1, 1), COLOUR_WHITE);
				display.DrawBox(CRect(x2, y2, x2, y2), CRect(-1, -1, 1, 1), COLOUR_WHITE);
			}
		}
	}

	// Draw all the boxes
	num = frame.m_Boxes.size();
	for (i = 0; i < num; ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		PIXEL boxcolour = BoxColours[box.m_Type & 7];
		PIXEL offsetcolour = OffsetColours[box.m_Type & 7];
		int x1 = box.m_Xpos;
		int y1 = box.m_Ypos;
		int x2 = x1 + box.m_Width;
		int y2 = y1 + box.m_Height;
		if (box.m_Selected)
		{
			display.DrawBox(CRect(x1, y1, x2, y1), CRect(0, 0, 0, 2), boxcolour);
			display.DrawBox(CRect(x1, y2, x2, y2), CRect(0, -1, 0, 1), boxcolour);
			display.DrawBox(CRect(x1, y1, x1, y2), CRect(0, 0, 2, 1), boxcolour);
			display.DrawBox(CRect(x2, y1, x2, y2), CRect(-1, 0, 1, 1), boxcolour);
		}
		else
		{
			display.DrawBox(CRect(x1, y1, x2, y1), CRect(0, 0, 0, 1), boxcolour);
			display.DrawBox(CRect(x1, y2, x2, y2), CRect(0, 0, 0, 1), boxcolour);
			display.DrawBox(CRect(x1, y1, x1, y2), CRect(0, 0, 1, 1), boxcolour);
			display.DrawBox(CRect(x2, y1, x2, y2), CRect(0, 0, 1, 1), boxcolour);
		}
		if ((box.m_OffsetX > 0) && (box.m_OffsetX < box.m_Width))
		{
			display.DrawBox(CRect(x1 + box.m_OffsetX, y1,
				x1 + box.m_OffsetX, y2), CRect(0, 0, 1, 0), offsetcolour);
		}
		if ((box.m_OffsetY > 0) && (box.m_OffsetY < box.m_Height))
		{
			display.DrawBox(CRect(x1, y1 + box.m_OffsetY,
				x2, y1 + box.m_OffsetY), CRect(0, 0, 0, 1), offsetcolour);
		}
	}

	// Draw the text if specified
	if (m_DrawAttributes & drawattr_showtext) // Display text
	{
		// Draw all the sprite text
		num = frame.m_Sprites.size();
		for (i = 0; i < num; ++i)
		{
			ChalkSprite& sprite = frame.m_Sprites[i];
			if (!sprite.m_Script.empty())
			{
				if (sprite.m_ObjDefn < font.m_Data.size())
				{
					ChalkCharacter& spdata = font.m_Data[sprite.m_ObjDefn];
					int x = sprite.m_Xpos - spdata.m_OffsetX;
					int y = sprite.m_Ypos - spdata.m_OffsetY + spdata.m_Height + TEXT_OFFSETY;
					display.DrawText(CPoint(x, y),
						TEXT_FONTHEIGHT, COLOUR_YELLOW, sprite.m_Script.c_str());
				}
			}
		}

		// Draw all the box text
		num = frame.m_Boxes.size();
		for (i = 0; i < num; ++i)
		{
			ChalkBox& box = frame.m_Boxes[i];
			if (!box.m_Script.empty())
				display.DrawText(CPoint(box.m_Xpos, box.m_Ypos + box.m_Height + TEXT_OFFSETY),
					TEXT_FONTHEIGHT, COLOUR_YELLOW, box.m_Script.c_str());
		}
		display.Flush();
	}

	// Draw all the paths
	num = frame.m_Paths.size();
	for (i = 0; i < num; ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		PIXEL pathcolour = BoxColours[path.m_Type & 7];
		PIXEL offsetcolour = OffsetColours[path.m_Type & 7];
		int numpoints = path.m_Data.size();
		int x1, y1, x2, y2, j;

		if (numpoints >= 2)
		{
			for (j = 0; j < numpoints; ++j)
			{
				ChalkPoint& point = path.m_Data[j];
				x2 = point.m_Xpos;
				y2 = point.m_Ypos;
				if (j)
					display.DrawLine(CPoint(x1, y1), CPoint(x2, y2), pathcolour);
				if (point.m_Selected)
					display.DrawBox(CRect(x2 - 1, y2 - 1, x2 + 1, y2 + 1), CRect(-1, -1, 1, 1), offsetcolour);
				x1 = x2;
				y1 = y2;
			}
		}
	}

	// Draw the selection box
	if (!m_SelectionBox.IsRectEmpty())
	{
		int x1 = m_SelectionBox.left;
		int y1 = m_SelectionBox.top;
		int x2 = m_SelectionBox.right;
		int y2 = m_SelectionBox.bottom;
		display.DrawBox(CRect(x1, y1, x2, y1), CRect(0, 0, 0, 1), COLOUR_WHITE);
		display.DrawBox(CRect(x1, y2, x2, y2), CRect(0, -1, 0, 0), COLOUR_WHITE);
		display.DrawBox(CRect(x1, y1, x1, y2), CRect(0, 0, 1, 0), COLOUR_WHITE);
		display.DrawBox(CRect(x2, y1, x2, y2), CRect(-1, 0, 0, 0), COLOUR_WHITE);
	}
}

// ----------------------------------------------------------------------------
CSize ChalkLayerAnimation::GetSize()
{
	int width = 0;
	int height = 0;
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return CSize(0, 0);
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i, j, num;

	// Check all the boxes
	num = frame.m_Boxes.size();
	for (i = 0; i < num; ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		width = max(width, box.m_Xpos + box.m_Width);
		height = max(height, box.m_Ypos + box.m_Height);
	}

	// Check all the paths
	num = frame.m_Paths.size();
	for (i = 0; i < num; ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		int numpoints = path.m_Data.size();

		for (j = 0; j < numpoints; ++j)
		{
			ChalkPoint& point = path.m_Data[j];
			width = max(width, point.m_Xpos);
			height = max(height, point.m_Ypos);
		}
	}
	return CSize(width, height);
}

// ----------------------------------------------------------------------------
// Get number of animation frames in this layer
int ChalkLayerAnimation::GetNumFrames()
{
	return m_Data.m_Frames.size();
}

// ----------------------------------------------------------------------------
// Returns the main selected object
ChalkObject* ChalkLayerAnimation::GetSelectedObject()
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return NULL;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i, j, num;

	// Check all the boxes
	num = frame.m_Boxes.size();
	for (i = 0; i < num; ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		if (box.m_Selected)
			return &box;
	}

	// Check all the paths
	num = frame.m_Paths.size();
	for (i = 0; i < num; ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		int numpoints = path.m_Data.size();

		for (j = 0; j < numpoints; ++j)
		{
			ChalkPoint& point = path.m_Data[j];
			if (point.m_Selected)
				return &point;
		}
		if (path.m_Selected)
			return &path;
	}

	// Check all the sprites
	num = frame.m_Sprites.size();
	for (i = 0; i < num; ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];
		if (sprite.m_Selected)
			return &sprite;
	}

	return NULL;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnLButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point)
{
	switch (m_DrawMode) {

	// Select an object
	case drawmode_select:
		return OnRButtonDblClk(pView, nFlags, point);
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnRButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point)
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return false;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	ChalkObject* pObject = NULL;

	switch (m_DrawMode) {

	// Select an object
	case drawmode_select:
	case drawmode_sprite:
	case drawmode_box:
	case drawmode_path:
		pObject = frame.GetNextSelected(m_Data.m_Font, point);
		if (pObject)
		{
			frame.SelectAll(false, false, false);
			pObject->m_Selected = true; // This object is selected
			m_Doc.UpdateAllViews(NULL); // For safety
			if (pObject->m_TypeId == GUID_SPRITE) // My own RTTI
			{
				CPropertySheetSprite dlg(static_cast<ChalkSprite *>(pObject), &m_Doc);
				dlg.DoModal();
			}
			else if (pObject->m_TypeId == GUID_BOX) // My own RTTI
			{
				CPropertySheetBox dlg(static_cast<ChalkBox *>(pObject), &m_Doc);
				dlg.DoModal();
			}
			else if (pObject->m_TypeId == GUID_POINT) // My own RTTI
			{
				CPropertySheetPoint dlg(static_cast<ChalkPoint *>(pObject), &m_Doc);
				dlg.DoModal();
			}
			else if (pObject->m_TypeId == GUID_PATH) // My own RTTI
			{
				CPropertySheetPath dlg(static_cast<ChalkPath *>(pObject), &m_Doc);
				dlg.DoModal();
			}
		}
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint point)
{
	m_ClickPos = m_LastMovePos = point;

	switch (m_DrawMode) {

	// Select an object
	case drawmode_select:
		// Note we don't call CheckAlignToGrid here because that
		// would make the clicks miss the pixels in collision detection
		return OnRButtonDown(pView, nFlags, point);

	// Create a sprite
	case drawmode_sprite:
		CheckAlignToGrid(point);
		NewSprite(point);
		break;

	// Start drawing a box
	case drawmode_box:
		CheckAlignToGrid(point);
		m_ClickPos = point;
		break;

	// Set box offsets
	case drawmode_offset:
		// Select a box (not a sprite or path)
		OnRButtonDown(pView, nFlags, point);
		// Set the offsets
		return OnMouseMove(pView, nFlags, point);

	// Create a path
	case drawmode_path:
		// ...
		break;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint point)
{
	m_ClickPos = m_LastMovePos = point;

	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return false;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	ChalkObject* pObject = NULL;
	int x1, x2, y1, y2;
	CRect rt;

	switch (m_DrawMode) {

	// Select an object
	case drawmode_select:
	case drawmode_sprite:
	case drawmode_box:
	case drawmode_path:
		pObject = frame.GetNextSelected(m_Data.m_Font, point);
		if (nFlags & (MK_SHIFT | MK_CONTROL)) // Add to selection
		{
		}
		else
		{
			InvalidateSelected();
			frame.SelectAll(false, false, false);
		}
		if (pObject) // Clicked on an object
		{
			pObject->m_Selected = true; // This object is selected
			InvalidateSelected();

			// Remember this object as a brush
			tstringstream S;
			if (pObject->m_TypeId == GUID_SPRITE) // My own RTTI
			{
				m_BrushSprite = *static_cast<ChalkSprite *>(pObject);
				S << "Sprite: " << m_BrushSprite.m_ObjDefn;
			}
			else if (pObject->m_TypeId == GUID_BOX) // My own RTTI
			{
				m_BrushBox = *static_cast<ChalkBox *>(pObject);
				S << "Box: " << m_BrushBox.m_Type;
			}
			else if (pObject->m_TypeId == GUID_PATH) // My own RTTI
			{
				m_BrushPath = *static_cast<ChalkPath *>(pObject);
				S << "Path: " << m_BrushPath.m_Type;
			}
			m_Doc.m_StatusType = S.str();
		}
		else // Select multiple
		{
			m_ClickPos.x = point.x;
			m_ClickPos.y = point.y;
			x1 = min(m_ClickPos.x, point.x);
			x2 = max(m_ClickPos.x, point.x) + 1;
			y1 = min(m_ClickPos.y, point.y);
			y2 = max(m_ClickPos.y, point.y) + 1;
			SetSelectBox(pView, CRect(x1, y1, x2, y2));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
		}
		return true;

	case drawmode_offset:
		// Select an object (box only)
		pObject = frame.GetNextSelected(m_Data.m_Font, point, true);
		InvalidateSelected();
		frame.SelectAll(false, false, false);
		if (pObject) // Clicked on an object
		{
			pObject->m_Selected = true; // This object is selected
			InvalidateSelected();
		}
		return true;

	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnMouseMove(CChalkView *pView, UINT nFlags, CPoint point)
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return false;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	ChalkObject* pObject = frame.GetFirstSelected();
	int smallx = (m_DrawAttributes & drawattr_grid) ? m_Data.m_GridX : 1;
	int smally = (m_DrawAttributes & drawattr_grid) ? m_Data.m_GridY : 1;
	int x1, y1, x2, y2;

	if (nFlags & MK_RBUTTON)
	{
		switch (m_DrawMode) {

		// Move an object
		case drawmode_select:
		case drawmode_sprite:
		case drawmode_box:
		case drawmode_path:
			if (pObject)
			{
				CheckAlignToGrid(point);
				pView->AutoScroll(point);
				MoveSelected(point.x - m_LastMovePos.x, point.y - m_LastMovePos.y);
				m_LastMovePos = point;
			}
			else // We are drawing a multiple selection box
			{
				pView->AutoScroll(point);
				// Show selection box
				x1 = min(m_ClickPos.x, point.x);
				x2 = max(m_ClickPos.x, point.x) + 1;
				y1 = min(m_ClickPos.y, point.y);
				y2 = max(m_ClickPos.y, point.y) + 1;
				SetSelectBox(pView, CRect(x1, y1, x2, y2));
				m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			}
			return true;
		};
	}
	else if (nFlags & MK_LBUTTON)
	{
		switch (m_DrawMode) {

		// Draw a multiple selection box
		case drawmode_select:
			pView->AutoScroll(point);
			// Show selection box
			x1 = min(m_ClickPos.x, point.x);
			x2 = max(m_ClickPos.x, point.x) + 1;
			y1 = min(m_ClickPos.y, point.y);
			y2 = max(m_ClickPos.y, point.y) + 1;
			SetSelectBox(pView, CRect(x1, y1, x2, y2));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			return true;

		// Draw a placeholder box
		case drawmode_box:
			CheckAlignToGrid(point);
			pView->AutoScroll(point);

			// Show selection box
			x1 = min(m_ClickPos.x, point.x);
			x2 = max(m_ClickPos.x, point.x) + smallx;
			y1 = min(m_ClickPos.y, point.y);
			y2 = max(m_ClickPos.y, point.y) + smally;
			SetSelectBox(pView, CRect(x1, y1, x2, y2));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			return true;

		// Set the box offsets
		case drawmode_offset:
			if (pObject && (pObject->m_TypeId == GUID_BOX)) // My own RTTI
			{
				ChalkBox* pBox = static_cast<ChalkBox*>(pObject);
				CheckAlignToGrid(point);
				pView->AutoScroll(point);
				SetBoxOffsets(pBox, point.x - pBox->m_Xpos,
					point.y - pBox->m_Ypos);
			}
			break;

		// Move an object
		case drawmode_sprite:
		case drawmode_path:
			if (pObject)
			{
				CheckAlignToGrid(point);
				pView->AutoScroll(point);
				MoveSelected(point.x - m_LastMovePos.x, point.y - m_LastMovePos.y);
			}
			else
				pView->AutoScroll(point);
			m_LastMovePos = point;
			return true;
		};
	}
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint point)
{
	SetSelectBox(pView, CRect(0,0,0,0));
	int smallx = (m_DrawAttributes & drawattr_grid) ? m_Data.m_GridX : 1;
	int smally = (m_DrawAttributes & drawattr_grid) ? m_Data.m_GridY : 1;
	int x1, y1, x2, y2;

	switch (m_DrawMode) {

	// Select an object
	case drawmode_select:
		return OnRButtonUp(pView, nFlags, point);

	// Create a box
	case drawmode_box:
		CheckAlignToGrid(point);
		pView->AutoScroll(point);
		x1 = min(m_ClickPos.x, point.x);
		x2 = max(m_ClickPos.x, point.x) + smallx;
		y1 = min(m_ClickPos.y, point.y);
		y2 = max(m_ClickPos.y, point.y) + smally;

		// Set solid rectangle
		NewBox(CRect(x1, y1, x2, y2));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint point)
{
	SetSelectBox(pView, CRect(0,0,0,0));
	int x1, y1, x2, y2;

	switch (m_DrawMode) {

	// Select an object
	case drawmode_select:
		// Handle selection box...
		pView->AutoScroll(point);
		x1 = min(m_ClickPos.x, point.x);
		x2 = max(m_ClickPos.x, point.x) + 1;
		y1 = min(m_ClickPos.y, point.y);
		y2 = max(m_ClickPos.y, point.y) + 1;

		// Set solid rectangle
		SelectionBox(CRect(x1, y1, x2, y2));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerAnimation::OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt)
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return false;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	ChalkObject *pObject = NULL;
	int i;

	switch (nChar) {

	// Support layer specific accelerators
	case '+':
		EditCommand(drawcmd_increment);
		break;

	case '-':
		EditCommand(drawcmd_decrement);
		break;

	case 'x':
		EditCommand(drawcmd_flipx);
		break;

	case 'y':
		EditCommand(drawcmd_flipy);
		break;

	// Delete selected objects
	case VK_BACK:
	case VK_DELETE:
		EditCommand(drawcmd_edit_delete);
		break;

	// Tab between objects
	case VK_TAB: 
		InvalidateSelected();
		nRepCnt = 1; // We want to see all the objects tabbed through
		for (i = 0; i < nRepCnt; ++i)
		{
			if (GetKeyState(VK_SHIFT) < 0) // Shift-tab = backwards
				pObject = frame.GetPrevSelected();
			else
				pObject = frame.GetNextSelected(); // Tab = forwards
			if (pObject)
			{
				frame.SelectAll(false, false, false);
				pObject->m_Selected = true;
			}
		}
		if (pObject)
			InvalidateSelected();
		return true;

	default:
		TRACE0("Unknown key\n");
		break;
	};
	return false;
}

// ----------------------------------------------------------------------------
// Perform a drawing command
void ChalkLayerAnimation::EditCommand(DrawCommands cmd)
{
	switch (cmd) {

	// Select all the sprites
	case drawcmd_edit_selectall:
		{
			if (m_CurrentFrame >= m_Data.m_Frames.size())
				;
			else
			{
				ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
				frame.SelectAll(true, true, true);
				InvalidateSelected();
			}
		}
		break;

	case drawcmd_edit_delete:
		DeleteSelected();
		break;

	// Edit layer properties
	case drawcmd_layer_properties:
		{
			CPropertySheetAnimation dlg(this, &m_Doc);
			dlg.DoModal();
		}
		break;

	// Edit the current animation frame properties
	case drawcmd_frame_properties:
		{
			CPropertySheetFrame dlg(&m_Data, &m_Doc, &m_CurrentFrame);
			dlg.DoModal();
			m_Doc.InvalidateViewsAll();
		}
		break;

	case drawcmd_flipx:
		FlipSelectedX();
		m_BrushSprite.m_Special ^= SPECIAL_XFLIP;
		break;

	case drawcmd_flipy:
		FlipSelectedY();
		m_BrushSprite.m_Special ^= SPECIAL_YFLIP;
		break;

	case drawcmd_increment:
		// Add one to the brush
		BrushAdd(1);
		AddToSelected(1);
		break;

	case drawcmd_decrement:
		// Subtract one from the brush
		BrushAdd(-1);
		AddToSelected(-1);
		break;

	case drawcmd_frame_insert:
		// Insert a frame
		m_Data.m_Frames.insert(&m_Data.m_Frames.at(m_CurrentFrame));
		m_Doc.ChangeFrame();
		break;

	case drawcmd_frame_delete:
		// Delete this frame
		m_Data.m_Frames.erase(&m_Data.m_Frames.at(m_CurrentFrame));
		if (m_CurrentFrame > 0 && m_CurrentFrame > m_Data.m_Frames.size())
			m_CurrentFrame--;
		m_Doc.ChangeFrame();
		break;

	default:
		AfxMessageBox(IDS_ERROR_NOTIMPLEMENTED);
		break;
	};
}

// ----------------------------------------------------------------------------
// Clicks are aligned to grid boundaries
void ChalkLayerAnimation::CheckAlignToGrid(CPoint& point)
{
	if (m_DrawAttributes & drawattr_grid)
	{
		int gx = m_Data.m_GridX;
		int gy = m_Data.m_GridY;
		point.x = gx * ((point.x + (gx / 2)) / gx);
		point.y = gy * ((point.y + (gy / 2)) / gy);
	}
}

// ----------------------------------------------------------------------------
// Signal to the document that the currently selected objects
// will all need to be repainted shortly
void ChalkLayerAnimation::InvalidateSelected()
{
	// Show selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i, j;

	// Show selected sprites
	ChalkFont& font = m_Data.m_Font;
	for (i = 0; i < frame.m_Sprites.size(); ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];
		if (sprite.m_Selected)
		{
			if (sprite.m_ObjDefn < font.m_Data.size())
			{
				ChalkCharacter& spdata = font.m_Data[sprite.m_ObjDefn];
				int x1 = sprite.m_Xpos - spdata.m_OffsetX;
				int y1 = sprite.m_Ypos - spdata.m_OffsetY;
				int x2 = x1 + spdata.m_Width;
				int y2 = y1 + spdata.m_Height;
				// Invalidate the sprite area, including the extra pixel
				// in each direction for the selection handles
				m_Doc.InvalidateViewsRect(CRect(x1, y1, x2, y2), CRect(-1, -1, 1, 1));
				// Show text
				if (m_DrawAttributes & drawattr_showtext)
				{
					// ... calculate width of text in pixels
					int width = 200;
					if (width > 0)
						m_Doc.InvalidateViewsRect(CRect(x1, y2, x1, y2),
							CRect(0, 0, width, TEXT_OFFSETY));
				}
			}
		}
	}

	// Show selected boxes
	for (i = 0; i < frame.m_Boxes.size(); ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		if (box.m_Selected)
		{
			int x1 = box.m_Xpos;
			int y1 = box.m_Ypos;
			int x2 = x1 + box.m_Width;
			int y2 = y1 + box.m_Height;
			// Invalidate the sprite area, including the extra pixel
			// to the bottom right for the thick selection border
			m_Doc.InvalidateViewsRect(CRect(x1, y1, x2, y2), CRect(0,0,1,1));
			// Show text
			if (m_DrawAttributes & drawattr_showtext)
			{
				// ... calculate width of text in pixels
				int width = 200;
				if (width > 0)
					m_Doc.InvalidateViewsRect(CRect(x1, y2, x1, y2), CRect(0,0,width,TEXT_OFFSETY));
			}
		}
	}

	// Show selected paths
	for (i = 0; i < frame.m_Paths.size(); ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		if (path.m_Selected && (path.m_Data.size() > 0))
		{
			CRect rt(path.m_Data[0].m_Xpos-2, path.m_Data[0].m_Ypos-2,
				path.m_Data[0].m_Xpos+2, path.m_Data[0].m_Ypos+2);
			for (j = 1; j < path.m_Data.size(); ++j)
			{
				// Include the point within the invalidated area,
				// including the extra two pixels
				// in each direction for the selection handles
				rt.left = min(rt.left, path.m_Data[j].m_Xpos - 2);
				rt.right = max(rt.right, path.m_Data[j].m_Xpos + 2);
				rt.top = min(rt.top, path.m_Data[j].m_Ypos - 2);
				rt.bottom = max(rt.bottom, path.m_Data[j].m_Ypos + 2);
			}
		}
	}
}

// ----------------------------------------------------------------------------
// Which objects are within a selection box?
void ChalkLayerAnimation::SelectionBox(CRect rect)
{
	// Show selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i, j;

	// Select sprites
	ChalkFont& font = m_Data.m_Font;
	for (i = 0; i < frame.m_Sprites.size(); ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];
		if (sprite.m_ObjDefn < font.m_Data.size())
		{
			ChalkCharacter& spdata = font.m_Data[sprite.m_ObjDefn];
			int x1 = sprite.m_Xpos - spdata.m_OffsetX;
			int y1 = sprite.m_Ypos - spdata.m_OffsetY;
			int x2 = x1 + spdata.m_Width;
			int y2 = y1 + spdata.m_Height;
			if (rect.left <= x1 && rect.right >= x2 && rect.top <= y1 && rect.bottom >= y2)
				sprite.m_Selected = true;
		}
	}

	// Select boxes
	for (i = 0; i < frame.m_Boxes.size(); ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		int x1 = box.m_Xpos;
		int y1 = box.m_Ypos;
		int x2 = x1 + box.m_Width;
		int y2 = y1 + box.m_Height;
		if (rect.left <= x1 && rect.right >= x2 && rect.top <= y1 && rect.bottom >= y2)
			box.m_Selected = true;
	}

	// Select paths
	for (i = 0; i < frame.m_Paths.size(); ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		if (path.m_Selected && (path.m_Data.size() > 0))
		{
			CRect rt(path.m_Data[0].m_Xpos, path.m_Data[0].m_Ypos,
				path.m_Data[0].m_Xpos, path.m_Data[0].m_Ypos);
			for (j = 1; j < path.m_Data.size(); ++j)
			{
				// Include the point within the area
				rt.left = min(rt.left, path.m_Data[j].m_Xpos);
				rt.right = max(rt.right, path.m_Data[j].m_Xpos);
				rt.top = min(rt.top, path.m_Data[j].m_Ypos);
				rt.bottom = max(rt.bottom, path.m_Data[j].m_Ypos);
			}
			if (rect.left <= rt.left && rect.right >= rt.right &&
				rect.top <= rt.top && rect.bottom >= rt.bottom)
				path.m_Selected = true;
		}
	}
}


// ----------------------------------------------------------------------------
// Add offset to the brush
void ChalkLayerAnimation::BrushAdd(int offset)
{
	tstringstream S;
	if (m_DrawMode == drawmode_sprite) // My own RTTI
	{
		m_BrushSprite.m_ObjDefn = max(0, m_BrushSprite.m_ObjDefn + offset);
		S << "Sprite: " << m_BrushSprite.m_ObjDefn;
	}
	else if (m_DrawMode == drawmode_box) // My own RTTI
	{
		m_BrushBox.m_Type = max(0, m_BrushBox.m_Type + offset);
		S << "Box: " << m_BrushBox.m_Type;
	}
	else if (m_DrawMode == drawmode_path) // My own RTTI
	{
		m_BrushPath.m_Type = max(0, m_BrushPath.m_Type + offset);
		S << "Path: " << m_BrushPath.m_Type;
	}
	m_Doc.m_StatusType = S.str();
}

// ----------------------------------------------------------------------------
// Commands that actually change the (persistant) data
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Move the currently selected objects by an amount
void ChalkLayerAnimation::MoveSelected(int xoffset, int yoffset)
{
	// Move selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	if ((xoffset == 0) && (yoffset == 0)) // Check for no-op
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i, j;
	bool bChanged = false;

	InvalidateSelected();
	// Move selected sprites
	for (i = 0; i < frame.m_Sprites.size(); ++i)
		if (frame.m_Sprites[i].m_Selected)
		{
			frame.m_Sprites[i].m_Xpos += xoffset;
			frame.m_Sprites[i].m_Ypos += yoffset;
			bChanged = true;
		}
	// Move selected boxes
	for (i = 0; i < frame.m_Boxes.size(); ++i)
		if (frame.m_Boxes[i].m_Selected)
		{
			frame.m_Boxes[i].m_Xpos += xoffset;
			frame.m_Boxes[i].m_Ypos += yoffset;
			bChanged = true;
		}
	// Move selected paths
	for (i = 0; i < frame.m_Paths.size(); ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		if (path.m_Selected)
		{
			for (j = 0; j < path.m_Data.size(); ++j)
			{
				path.m_Data[j].m_Xpos += xoffset;
				path.m_Data[j].m_Ypos += yoffset;
			}
			bChanged = true;
		}
	}
	// Signal that the data has changed
	InvalidateSelected();
	if (bChanged)
		m_Doc.SetModifiedFlag();
}

// ----------------------------------------------------------------------------
// Delete the currently selected objects
void ChalkLayerAnimation::DeleteSelected()
{
	// Delete selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	bool bChanged = false;

	InvalidateSelected();
	// Delete selected sprites
	{
		vector<ChalkSprite>::iterator i = frame.m_Sprites.begin();
		while (i != frame.m_Sprites.end())
		{
			if (i->m_Selected)
			{
				// ... Remember undo
				i = frame.m_Sprites.erase(i);
				bChanged = true;
			}
			else
				++i;
		}
	}

	// Delete selected boxes
	{
		vector<ChalkBox>::iterator i = frame.m_Boxes.begin();
		while (i != frame.m_Boxes.end())
		{
			if (i->m_Selected)
			{
				// ... Remember undo
				frame.m_Boxes.erase(i);
				bChanged = true;
			}
			else
				++i;
		}
	}
	// Delete selected paths
	{
		vector<ChalkPath>::iterator i = frame.m_Paths.begin();
		while (i != frame.m_Paths.end())
		{
			ChalkPath& path = *i;
			if (i->m_Selected)
			{
				// ... Remember undo
				frame.m_Paths.erase(i);
				bChanged = true;
			}
			else
			{
				// Delete selected points within a path
				vector<ChalkPoint>::iterator j = path.m_Data.begin();
				while (j != path.m_Data.end())
				{
					if (j->m_Selected)
					{
						// ... Remember undo
						path.m_Data.erase(j);
						bChanged = true;
					}
					else
						++j;
				}
				++i;
			}
		}
	}
	// Signal that the data has changed
	InvalidateSelected();
	if (bChanged)
		m_Doc.SetModifiedFlag();
}

// ----------------------------------------------------------------------------
// Add offset to selected objects
void ChalkLayerAnimation::AddToSelected(int offset)
{
	// Move selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	if (offset == 0) // Check for no-op
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	int i;
	bool bChanged = false;

	InvalidateSelected();
	// Change type of selected sprites
	for (i = 0; i < frame.m_Sprites.size(); ++i)
		if (frame.m_Sprites[i].m_Selected)
		{
			frame.m_Sprites[i].m_ObjDefn = max(0, frame.m_Sprites[i].m_ObjDefn + offset);
			bChanged = true;
		}
	// Change type of selected boxes
	for (i = 0; i < frame.m_Boxes.size(); ++i)
		if (frame.m_Boxes[i].m_Selected)
		{
			frame.m_Boxes[i].m_Type = max(0, frame.m_Boxes[i].m_Type + offset);
			bChanged = true;
		}
	// Change type of selected paths
	for (i = 0; i < frame.m_Paths.size(); ++i)
		if (frame.m_Paths[i].m_Selected)
		{
			frame.m_Paths[i].m_Type = max(0, frame.m_Paths[i].m_Type + offset);
			bChanged = true;
		}

	// Signal that the data has changed
	InvalidateSelected();
	if (bChanged)
		m_Doc.SetModifiedFlag();
}

// ----------------------------------------------------------------------------
// Flip selected objects horizontally
void ChalkLayerAnimation::FlipSelectedX()
{
	// Flip selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	CRect extent = frame.GetSelectedExtent();
	int cx2 = (extent.left + extent.right); // Pivot point
	int i, j;
	bool bChanged = false;

	InvalidateSelected();
	// Flip selected sprites
	for (i = 0; i < frame.m_Sprites.size(); ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];
		if (sprite.m_Selected)
		{
			sprite.m_Xpos = cx2 - sprite.m_Xpos;
			sprite.m_Special ^= SPECIAL_XFLIP;
			bChanged = true;
		}
	}
	// Flip selected boxes
	for (i = 0; i < frame.m_Boxes.size(); ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		if (box.m_Selected)
		{
			box.m_Xpos = (cx2 - (box.m_Xpos + box.m_OffsetX)) - (box.m_Width - box.m_OffsetX);
			box.m_OffsetX = box.m_Width - box.m_OffsetX;
			bChanged = true;
		}
	}
	// Flip selected paths
	for (i = 0; i < frame.m_Paths.size(); ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		if (path.m_Selected)
		{
			for (j = 0; j < path.m_Data.size(); ++j)
			{
				ChalkPoint &point = path.m_Data[j];
				point.m_Xpos = cx2 - point.m_Xpos;
			}
			bChanged = true;
		}
	}

	// Signal that the data has changed
	InvalidateSelected();
	if (bChanged)
		m_Doc.SetModifiedFlag();
}

// ----------------------------------------------------------------------------
// Flip selected objects vertically
void ChalkLayerAnimation::FlipSelectedY()
{
	// Flip selected objects
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	CRect extent = frame.GetSelectedExtent();
	int cy2 = (extent.top + extent.bottom); // Pivot point
	int i, j;
	bool bChanged = false;

	InvalidateSelected();
	// Flip selected sprites
	for (i = 0; i < frame.m_Sprites.size(); ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];
		if (sprite.m_Selected)
		{
			sprite.m_Ypos = cy2 - sprite.m_Ypos;
			sprite.m_Special ^= SPECIAL_YFLIP;
			bChanged = true;
		}
	}
	// Flip selected boxes
	for (i = 0; i < frame.m_Boxes.size(); ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		if (box.m_Selected)
		{
			box.m_Ypos = (cy2 - (box.m_Ypos + box.m_OffsetY)) - (box.m_Height - box.m_OffsetY);
			box.m_OffsetY = box.m_Height - box.m_OffsetY;
			bChanged = true;
		}
	}
	// Flip selected paths
	for (i = 0; i < frame.m_Paths.size(); ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		if (path.m_Selected)
		{
			for (j = 0; j < path.m_Data.size(); ++j)
			{
				ChalkPoint &point = path.m_Data[j];
				point.m_Ypos = cy2 - point.m_Ypos;
			}
			bChanged = true;
		}
	}

	// Signal that the data has changed
	InvalidateSelected();
	if (bChanged)
		m_Doc.SetModifiedFlag();
}


// ----------------------------------------------------------------------------
// Create a new sprite
void ChalkLayerAnimation::NewSprite(CPoint point)
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	InvalidateSelected();
	frame.SelectAll(false, false, false);

	// Create a new sprite, a clone of the last one
	ChalkSprite sprite;
	sprite.m_Comment = m_BrushSprite.m_Comment;
	sprite.m_Script = m_BrushSprite.m_Script;
	sprite.m_FileName.Clear();
	sprite.m_Dirty = true; // Have I been changed since I was last saved?
	sprite.m_Selected = true; // Now selected

	sprite.m_Xpos = point.x;
	sprite.m_Ypos = point.y;
	sprite.m_Zpos = m_BrushSprite.m_Zpos;
	sprite.m_ObjDefn = m_BrushSprite.m_ObjDefn;
	sprite.m_PaletteOffset = m_BrushSprite.m_PaletteOffset;
	sprite.m_Special = m_BrushSprite.m_PaletteOffset;
	sprite.m_Attribute = m_BrushSprite.m_Attribute;

	// Handle undo ...

	frame.m_Sprites.push_back(sprite);

	// Signal that the data has changed
	InvalidateSelected();
	m_Doc.SetModifiedFlag();
}

// ----------------------------------------------------------------------------
// Create a new box
void ChalkLayerAnimation::NewBox(CRect rect)
{
	if (m_CurrentFrame >= m_Data.m_Frames.size())
		return;
	ChalkFrame& frame = m_Data.m_Frames[m_CurrentFrame];
	InvalidateSelected();
	frame.SelectAll(false, false, false);

	// Create a new box, a clone of the last one
	ChalkBox box;
	box.m_Comment = m_BrushBox.m_Comment;
	box.m_Script = m_BrushBox.m_Script;
	box.m_FileName.Clear();
	box.m_Dirty = true; // Have I been changed since I was last saved?
	box.m_Selected = true; // Now selected

	box.m_Xpos = rect.left;
	box.m_Ypos = rect.top;
	box.m_Zpos = m_BrushBox.m_Zpos;
	box.m_Width = rect.Width();
	box.m_Height = rect.Height();
	box.m_Depth = m_BrushBox.m_Depth;
	box.m_OffsetX = 0;
	box.m_OffsetY = 0;
	box.m_OffsetZ = 0;
	box.m_Type = m_BrushBox.m_Type;
	box.m_Shape = m_BrushBox.m_Shape;

	// Handle undo ...

	frame.m_Boxes.push_back(box);

	// Signal that the data has changed
	InvalidateSelected();
	m_Doc.SetModifiedFlag();
}

// ----------------------------------------------------------------------------
// Set box offsets
void ChalkLayerAnimation::SetBoxOffsets(ChalkBox* pBox, int xoffset, int yoffset)
{
	// Handle undo ...

	pBox->m_OffsetX = xoffset;
	pBox->m_OffsetY = yoffset;

	// Signal that the data has changed
	InvalidateSelected();
	m_Doc.SetModifiedFlag();
}
