// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkMap.h
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Defines the game map objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"
#include "chalkdisplay.h"


// ----------------------------------------------------------------------------
// ChalkMapBlock
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkMapBlock::ChalkMapBlock() :
	m_BlockIndex(0),
	m_PaletteOffset(0),
	m_Special(0),
	m_Attribute(0)
{
}


// ----------------------------------------------------------------------------
// Convert a machine specific block into an independant map block
void ChalkMapBlock::Set(DWORD value, ChalkSystem* systemtype)
{
	if (systemtype == NULL)
		m_BlockIndex = value;
	else
	{
		m_BlockIndex = value & systemtype->m_BlockMask;
		m_PaletteOffset = (value & systemtype->m_PaletteMask) >> systemtype->m_PaletteShift;
		m_Special = ((value & systemtype->m_YflipMask) ? SPECIAL_YFLIP : 0) |
			((value & systemtype->m_XflipMask) ? SPECIAL_XFLIP : 0) | 
			((value & systemtype->m_PriorityMask) ? SPECIAL_PRIORITY : 0);
	}
}

// ----------------------------------------------------------------------------
// Convert an independant map block into a machine specific block
DWORD ChalkMapBlock::Get(ChalkSystem* systemtype) const
{
	if (systemtype == NULL)
		return m_BlockIndex;
	return (m_BlockIndex & systemtype->m_BlockMask) |
		((m_PaletteOffset << systemtype->m_PaletteShift) & systemtype->m_PaletteMask) |
		((m_Special & SPECIAL_YFLIP) ? systemtype->m_YflipMask : 0) |
		((m_Special & SPECIAL_XFLIP) ? systemtype->m_XflipMask : 0) |
		((m_Special & SPECIAL_PRIORITY) ? systemtype->m_PriorityMask : 0);
}


// ----------------------------------------------------------------------------
// ChalkMapBase
// An abstract base class, for things that can be tessalated as part of a map
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkMapBase::ChalkMapBase(GUID id, GUID typeguid) :
	ChalkObject(id, typeguid),
	m_SystemType(GUID_SYSTEM_UNKNOWN),
	m_Grid1X(8),
	m_Grid1Y(8),
	m_Grid2X(16),
	m_Grid2Y(16)
{
}

// ----------------------------------------------------------------------------
void ChalkMapBase::DrawBlock(ChalkDisplay& display, CRect rt,
	CPoint pt, ChalkMapBlock& block, ChalkPalette& Palette) const
{
	PIXEL blockcolour = 0x010101 * ((block.m_BlockIndex * 43) & 255);
	int x1 = pt.x;
	int y1 = pt.y;
	int x2 = x1 + m_Grid2X;
	int y2 = y1 + m_Grid2Y;
	display.DrawBox(CRect(x1, y1, x2, y2), CRect(0, 0, 0, 0), blockcolour);
}

// ----------------------------------------------------------------------------
ChalkPalette* ChalkMapBase::GetPalette()
{
	return NULL;
}


// ----------------------------------------------------------------------------
// ChalkMap
// ----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
ChalkMap::ChalkMap() : ChalkObject(GUID_VOID, GUID_MAP),
	m_Width(0),
	m_Height(0),
	m_Data(NULL)
{
}

// ----------------------------------------------------------------------------
ChalkMap::~ChalkMap()
{
	if (m_Data)
		delete [] m_Data;
}

// ----------------------------------------------------------------------------
// copy constructor
ChalkMap::ChalkMap(const ChalkMap &that)
	: ChalkObject(that.m_Id, GUID_MAP),
	m_Width(that.m_Width),
	m_Height(that.m_Height),
	m_Data(NULL)
{
	if (m_Width * m_Height > 0)
	{
		m_Data = new ChalkMapBlock[m_Width * m_Height];
		// Assumes ChalkMapBlock is Plain Old Data
		memcpy(m_Data, that.m_Data, m_Width * m_Height * sizeof(ChalkMapBlock));
	}
}

// ----------------------------------------------------------------------------
// assignment operator
const ChalkMap& ChalkMap::operator= (const ChalkMap& that)
{
	if (this != &that)
	{
		if (m_Data)
			delete [] m_Data;
		m_Data = NULL;

		ChalkObject::operator= (that);
		m_Width = that.m_Width;
		m_Height = that.m_Height;
		if (m_Width * m_Height > 0)
		{
			m_Data = new ChalkMapBlock[m_Width * m_Height];
			// Assumes ChalkMapBlock is Plain Old Data
			memcpy(m_Data, that.m_Data, m_Width * m_Height * sizeof(ChalkMapBlock));
		}
	}
	return *this;
}

// ----------------------------------------------------------------------------
HRESULT ChalkMap::Write(ChalkStream& os) const
{
	os.WriteElement1("Map");
	os.WriteInteger("sizex", m_Width);
	os.WriteInteger("sizey", m_Height);
	os.WriteInteger("blockbytes", sizeof(ChalkMapBlock));
	os.WriteBlob1("Data", m_Width * m_Height * sizeof(ChalkMapBlock), (BYTE*) m_Data);
	os.WriteEndElement("Map");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkMap::Read(ChalkStream& is)
{
	is.ReadElement1("Map");
	int width = is.ReadInteger("sizex");
	int height = is.ReadInteger("sizey");
	SetSize(width, height);
	int nBytes = is.ReadInteger("blockbytes");
	if (nBytes != sizeof(ChalkMapBlock))
	{
		vector<BYTE> item = is.ReadBlob1("Data");
		// Ignore the data - its format has changed :-(
	}
	else
	{
		vector<BYTE> item = is.ReadBlob1("Data");
		if (item.size() >= m_Width * m_Height * sizeof(ChalkMapBlock))
		{
			memcpy(m_Data, &item[0], m_Width * m_Height * sizeof(ChalkMapBlock));
		}
	}
	is.ReadEndElement("Map");
	return S_OK;
}

// ----------------------------------------------------------------------------
void ChalkMap::SetSize(int width, int height)
{
	if (m_Data)
		delete [] m_Data;
	m_Data = NULL;
	if (width * height > 0)
		m_Data = new ChalkMapBlock[width * height];
	m_Width = width;
	m_Height = height;
}


// ----------------------------------------------------------------------------
// Resize a map preserving the old data
void ChalkMap::Resize(int width, int height)
{
	// Copy old data across
	ChalkMapBlock* pData = NULL;
	if (width * height > 0)
	{
		pData = new ChalkMapBlock[width * height];
		int my = min(height, m_Height);
		int mx = min(width, m_Width);
		for (int y = 0; y < my; ++y)
		{
			int posSrc = y * m_Width;
			int posDst = y * width;
			for (int x = 0; x < mx; ++x)
			{
				pData[posDst++] = m_Data[posSrc++];
			}
		}
	}

	// Use the new data
	m_Width = width;
	m_Height = height;
	if (m_Data)
		delete [] m_Data;
	m_Data = pData;
}


// ----------------------------------------------------------------------------
// ChalkMapAnim
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkMapAnim::ChalkMapAnim() : ChalkMapBase(GUID_VOID, GUID_MAP),
	m_pBlocks(NULL),
	m_pAttrBlocks(NULL),
	m_Speed(15.0)
{
}

// ----------------------------------------------------------------------------
ChalkMapAnim::~ChalkMapAnim()
{
	if (m_pAttrBlocks)
		delete m_pAttrBlocks;
	if (m_pBlocks)
		delete m_pBlocks;
}

// ----------------------------------------------------------------------------
HRESULT ChalkMapAnim::Write(ChalkStream& os) const
{
	os.WriteElement1("MapAnimation");
	os.WriteGUID("systemtype", m_SystemType);
	os.WriteInteger("grid1x", m_Grid1X);
	os.WriteInteger("grid1y", m_Grid1Y);
	os.WriteInteger("grid2x", m_Grid2X);
	os.WriteInteger("grid2y", m_Grid2Y);
	os.WriteReal("speed", m_Speed);
	os.WriteString("blockfile", m_pBlocks ? m_pBlocks->m_FileName.Get().c_str() : L"");
	os.WriteString("attributefile", m_pAttrBlocks ? m_pAttrBlocks->m_FileName.Get().c_str() : L"");

	m_Palette.Write(os);
	os.WriteElement("Maps", m_Data.size());
	for (vector<ChalkMap>::const_iterator i = m_Data.begin(); i != m_Data.end(); ++i)
	{
		i->Write(os);
	}
	os.WriteEndElement("Maps");
	os.WriteEndElement("MapAnimation");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkMapAnim::Read(ChalkStream& is)
{
	is.ReadElement1("MapAnimation");
	m_SystemType = is.ReadGUID("systemtype");
	m_Grid1X = is.ReadInteger("grid1x");
	m_Grid1Y = is.ReadInteger("grid1y");
	m_Grid2X = is.ReadInteger("grid2x");
	m_Grid2Y = is.ReadInteger("grid2y");
	m_Speed = is.ReadReal("speed");
	wstring blockname = is.ReadString("blockfile");
	wstring attrname = is.ReadString("attributefile");
	if (!m_pBlocks)
		LoadBlocks(blockname, attrname, m_Grid2X / m_Grid1X, m_Grid2Y / m_Grid1Y);

	m_Palette.Read(is);
	int n = is.ReadElement("Maps");
	m_Data.clear();
	for (int i = 0; i < n; ++i)
	{
		ChalkMap item;
		item.Read(is);
		m_Data.push_back(item);
	}
	is.ReadEndElement("Maps");
	is.ReadEndElement("MapAnimation");
	return S_OK;
}

// ----------------------------------------------------------------------------
// Does this class support this alternative file format?
bool ChalkMapAnim::SupportsFile(const CFileName& filename)
{
	if (filename.HasExt(FILEEXTENSION_MAP))
		return true;
	if (filename.HasExt(FILEEXTENSION_BLK))
		return true;
	return false;
}

// ----------------------------------------------------------------------------
// Distribute the read alternative file command
HRESULT ChalkMapAnim::ReadFile(const CFileName& filename)
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return ReadXML(filename);
	if (filename.HasExt(FILEEXTENSION_MAP))
		return ReadMAP(filename);
	if (filename.HasExt(FILEEXTENSION_BLK))
		return ReadMAP(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
// Distribute the write alternative file command
HRESULT ChalkMapAnim::WriteFile(const CFileName& filename) const
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return WriteXML(filename);
	if (filename.HasExt(FILEEXTENSION_MAP))
		return WriteMAP(filename);
	if (filename.HasExt(FILEEXTENSION_BLK))
		return WriteMAP(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}


// ----------------------------------------------------------------------------
void ChalkMapAnim::DrawBlock(ChalkDisplay& display, CRect rt,
	CPoint pt, ChalkMapBlock& mapblock, ChalkPalette& Palette) const
{
	int currentframe = 0; // ... Don't allow block set animation yet
	if (currentframe >= m_Data.size())
		return;
	if (!m_pBlocks)
		return;
	const ChalkMap& map = m_Data[currentframe];
	ChalkBlockIndex block = mapblock.m_BlockIndex;
	bool flipx = mapblock.m_Special & SPECIAL_XFLIP ? true : false;
	bool flipy = mapblock.m_Special & SPECIAL_YFLIP ? true : false;

	// Display the bitmap
	int mapwidth = map.m_Width;
	int mapheight = map.m_Height;
	int mx = m_Grid2X / m_Grid1X;
	int my = m_Grid2Y / m_Grid1Y;
	int x, y;
	int ypos = pt.y;

	// Sanity check
	if (my * (block + 1) > mapheight)
	{
		display.DrawBox(CRect(pt.x, pt.y, pt.x + m_Grid2X, pt.y + m_Grid2Y), CRect(0,0,0,0), COLOUR_MAGENTA);
		return;
	}
	ChalkMapBlock tempblock;
	for (y = 0; y < my; ++y, ypos += m_Grid1Y)
	{
		int xpos = pt.x;
		for (x = 0; x < mx; ++x, xpos += m_Grid1X)
		{
			if (flipy)
			{
				if (flipx)
					tempblock = map.m_Data[mapwidth * (block * my + my - y - 1) + (mx - x - 1)];
				else
					tempblock = map.m_Data[mapwidth * (block * my + my - y - 1) + x];
			}
			else
			{
				if (flipx)
					tempblock = map.m_Data[mapwidth * (block * my + y) + (mx - x - 1)];
				else
					tempblock = map.m_Data[mapwidth * (block * my + y) + x];
			}
			tempblock.m_PaletteOffset += mapblock.m_PaletteOffset;
			tempblock.m_Special = tempblock.m_Special ^ (mapblock.m_Special & (SPECIAL_XFLIP | SPECIAL_YFLIP));
			m_pBlocks->DrawBlock(display, rt, CPoint(xpos, ypos),
				tempblock, Palette);
		}
	}
}


// ----------------------------------------------------------------------------
ChalkPalette* ChalkMapAnim::GetPalette()
{
	return &m_Palette;
}
