// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkObj.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 23 Oct 2001 by Carl Muller
//
// This implements the base game object class
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkObjects.h"

// ----------------------------------------------------------------------------
// The one and only GUID object map
ChalkObjectMap g_ChalkObjectMap;

// ----------------------------------------------------------------------------
// Comparison operator for GUIDs
// This is required by the map<> template, since it is implemented as a tree
bool operator< (GUID a, GUID b)
{
	if (a.Data1 < b.Data1)
		return true;
	if (a.Data1 > b.Data1)
		return false;
	if (a.Data2 < b.Data2)
		return true;
	if (a.Data2 > b.Data2)
		return false;
	if (a.Data3 < b.Data3)
		return true;
	if (a.Data3 > b.Data3)
		return false;
	for (int i = 0; i < sizeof(a.Data4); ++i)
	{
		if (a.Data4[i] < b.Data4[i])
			return true;
		if (a.Data4[i] > b.Data4[i])
			return false;
	}
	return false; // GUIDs are equal
}



// ----------------------------------------------------------------------------
// ChalkObject
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Main constructor
ChalkObject::ChalkObject(GUID id, GUID typeguid) :
	m_Id(id),
	m_TypeId(typeguid)
{
	TCHAR buffer[300];
	DWORD length;

	m_DateCreated = COleDateTime::GetCurrentTime();
	m_UserCreated.resize(0); // .clear when STL is supported properly by Microsoft;
	m_ComputerCreated.resize(0); // .clear when STL is supported properly by Microsoft;
	length = sizeof(buffer);
	if (::GetUserName(buffer, &length))
		m_UserCreated = buffer;
	length = sizeof(buffer);
	if (::GetComputerName(buffer, &length))
		m_ComputerCreated = buffer;
	m_DateModified = m_DateCreated;
	m_UserModified = m_UserCreated;
	m_ComputerModified = m_ComputerCreated;

	m_Dirty = false;
	m_Selected = false;
}

// ----------------------------------------------------------------------------
// Assignment operator
const ChalkObject& ChalkObject::operator= (const ChalkObject& that)
{
	if (this != &that)
	{
		m_Id = that.m_Id;   // ID for this particular object
		m_TypeId = that.m_TypeId; // ID for this type of object
		m_DateCreated = that.m_DateCreated;    // Date/time the GUID was allocated
		m_UserCreated = that.m_UserCreated; // Domain\Username of user who created this object
		m_ComputerCreated = that.m_ComputerCreated; // Computer name of user who created this object
		m_DateModified = that.m_DateModified;   // Date/time the object was altered
		m_UserModified = that.m_UserModified; // Domain\Username of user who last modified this object
		m_ComputerModified = that.m_ComputerModified; // Computer name of user who last modified this object

		// Persistant data members
		m_Comment = that.m_Comment;  // User specified comment
		m_Script = that.m_Script;   // User specified script
		m_FileName = that.m_FileName; // Filename of object, if it is stored in a file

		// Current data members
		m_Dirty = that.m_Dirty; // Have I been changed since I was last saved?
		m_Selected = that.m_Selected; // Am I currently selected by the user?
	}
	return *this;
}

// ----------------------------------------------------------------------------
ChalkObject::~ChalkObject()
{
}

// ----------------------------------------------------------------------------
HRESULT ChalkObject::Write(ChalkStream& os) const
{
	os.WriteElement1("Object");
	os.WriteString("comment", m_Comment);
	os.WriteString("script", m_Script);
	os.WriteEndElement("Object");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkObject::Read(ChalkStream& is)
{
	is.ReadElement1("Object");
	m_Comment = is.ReadString("comment");
	m_Script = is.ReadString("script");
	is.ReadEndElement("Object");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkObject::ReadFile(const CFileName& filename)
{
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
HRESULT ChalkObject::WriteFile(const CFileName& filename) const
{
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
// Help derived classes write XML files
HRESULT ChalkObject::WriteXML(const CFileName& filename) const
{
	// Open up a buffered file
	BufferedFile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	// Associate the file with an XML stream
	ChalkStreamText outstream(outfile);

	// Write out the XML header
	outfile.WriteString("<?xml version=\"1.0\" ?>");

	// Write out my data
	Write(outstream);

	// Finish off
	outfile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
// Help derived classes read XML files
HRESULT ChalkObject::ReadXML(const CFileName& filename)
{
	// Open up a buffered file
	BufferedFile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	// Read in the XML header
	infile.ReadString(">", "", NULL);

	// Associate the file with an XML stream
	ChalkStreamText instream(infile);

	// Read in my data
	Read(instream);

	// Finish off
	infile.Close();
	return hr;

}


// ----------------------------------------------------------------------------
// ChalkType
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkType::ChalkType(GUID typeguid, LPCTSTR chalktypename) :
	ChalkObject(typeguid, GUID_TYPE),
	m_TypeName(chalktypename)
{
}

// ----------------------------------------------------------------------------
ChalkType::~ChalkType()
{
}

// ----------------------------------------------------------------------------
HRESULT ChalkType::Write(ChalkStream& os) const
{
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
HRESULT ChalkType::Read(ChalkStream& is)
{
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
// ChalkObjectMap
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkObjectMap::ChalkObjectMap()
{
	// Add standard types
	AddType(GUID_TYPE, TYPENAME_TYPE);
	AddType(GUID_SYSTEM, TYPENAME_SYSTEM);
	AddType(GUID_PALETTE, TYPENAME_PALETTE);
	AddType(GUID_CHARACTER, TYPENAME_CHARACTER);
	AddType(GUID_BITMAP, TYPENAME_BITMAP);
	AddType(GUID_BOX, TYPENAME_BOX);
	AddType(GUID_POINT, TYPENAME_POINT);
	AddType(GUID_PATH, TYPENAME_PATH);
	AddType(GUID_SPRITE, TYPENAME_SPRITE);
	AddType(GUID_FRAME, TYPENAME_FRAME);
	AddType(GUID_ANIMATION, TYPENAME_ANIMATION);
	AddType(GUID_FONT, TYPENAME_FONT);
	AddType(GUID_MAP, TYPENAME_MAP);

	// Add standard computer systems
	AddObject(new ChalkSystem(GUID_SYSTEM_UNKNOWN, NAME_SYSTEM_UNKNOWN,
		8, 0xffff, 0x0000, 0, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_SINCLAIRQL, NAME_SYSTEM_SINCLAIRQL,
		3, 0xffff, 0x0000, 0, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_ATARIST, NAME_SYSTEM_ATARIST,
		3, 0xffff, 0x0000, 0, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_AMIGA, NAME_SYSTEM_AMIGA,
		4, 0xffff, 0x0000, 0, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_PC, NAME_SYSTEM_PC,
		8, 0xffff, 0x0000, 0, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_TARGA, NAME_SYSTEM_TARGA,
		8, 0xffff, 0x0000, 0, 0x0000, 0x0000, 0x0000));

	AddObject(new ChalkSystem(GUID_SYSTEM_C64, NAME_SYSTEM_C64,
		0, 0x00ff, 0x0300, 6, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_NES, NAME_SYSTEM_NES,
		0, 0x00ff, 0x0300, 6, 0x0000, 0x0000, 0x0000));
	AddObject(new ChalkSystem(GUID_SYSTEM_GAMEBOY, NAME_SYSTEM_GAMEBOY,
		0, 0x0fff, 0x1000,10, 0x4000, 0x2000, 0x8000));
	AddObject(new ChalkSystem(GUID_SYSTEM_MASTER, NAME_SYSTEM_MASTER,
		2, 0x01ff, 0x0800, 7, 0x0400, 0x0200, 0x1000));
	AddObject(new ChalkSystem(GUID_SYSTEM_GAMEGEAR, NAME_SYSTEM_GAMEGEAR,
		4, 0x01ff, 0x0800, 7, 0x0400, 0x0200, 0x1000));
	AddObject(new ChalkSystem(GUID_SYSTEM_MEGADRIVE, NAME_SYSTEM_MEGADRIVE,
		3, 0x07ff, 0x6000, 9, 0x1000, 0x0800, 0x8000));
	AddObject(new ChalkSystem(GUID_SYSTEM_SNES, NAME_SYSTEM_SNES,
		5, 0x03ff, 0x1c00, 6, 0x8000, 0x4000, 0x2000));
	AddObject(new ChalkSystem(GUID_SYSTEM_SNESMODE7, NAME_SYSTEM_SNESMODE7,
		5, 0x3fff, 0x0000, 0, 0x8000, 0x4000, 0x0000));
}

// ----------------------------------------------------------------------------
ChalkObjectMap::~ChalkObjectMap()
{
	map<GUID, ChalkObject*>::iterator posn;
	ChalkObject* pObj;

	for (posn = m_Data.begin(); posn != m_Data.end(); ++posn)
	{
		pObj = posn->second;
		delete pObj;
	}
}

// ----------------------------------------------------------------------------
void ChalkObjectMap::AddObject(ChalkObject* object)
{
	ASSERT(m_Data.end() == m_Data.find(object->m_Id));
	m_Data[object->m_Id] = object;
}

// ----------------------------------------------------------------------------
ChalkObject* ChalkObjectMap::GetObject(GUID id) const
{
	ChalkObject* ptr = NULL;
	map<GUID, ChalkObject*>::const_iterator posn = m_Data.find(id);
	if (posn != m_Data.end())
	{
		ptr = posn->second;
	}
	return ptr;
}


// ----------------------------------------------------------------------------
void ChalkObjectMap::AddType(GUID id, LPCTSTR chalktypename)
{
	ChalkType *pType = new ChalkType(id, chalktypename);
	ASSERT(m_Data.end() == m_Data.find(id));
	m_Data[id] = pType;
}

// ----------------------------------------------------------------------------
tstring ChalkObjectMap::GetTypeName(GUID id) const
{
	map<GUID, ChalkObject*>::const_iterator posn = m_Data.find(id);
	if (posn != m_Data.end())
	{
		ChalkObject *pType = posn->second;
		if (pType->m_TypeId == GUID_TYPE) // My own RTTI
			return (static_cast<ChalkType *>(pType))->m_TypeName;
	}
	return tstring(FORMAT_EMPTY);
}

// ----------------------------------------------------------------------------
// ChalkSystem
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkSystem::ChalkSystem(GUID systemid, LPCTSTR systemname, int bitsperpalette,
	int blockmask, int palettemask, int paletteshift,
	int yflipmask, int xflipmask, int prioritymask) :
		ChalkObject(systemid, GUID_SYSTEM),
		m_SystemName(systemname),
		m_BitsPerPalette(bitsperpalette),
		m_BlockMask(blockmask),
		m_PaletteMask(palettemask),
		m_PaletteShift(paletteshift),
		m_YflipMask(yflipmask),
		m_XflipMask(xflipmask),
		m_PriorityMask(prioritymask)
{
}

// ----------------------------------------------------------------------------
ChalkSystem::~ChalkSystem()
{
}

// ----------------------------------------------------------------------------
HRESULT ChalkSystem::Write(ChalkStream& os) const
{
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
HRESULT ChalkSystem::Read(ChalkStream& is)
{
	return E_NOTIMPL;
}
