// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileBMP.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports Microsoft BMP files - standard bitmap file format
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
static int rounddiv(int a, int b)
{
	return ( ((a) + (b) - 1) / (b) );
}

// ----------------------------------------------------------------------------
// Load a picture (Microsoft BMP format)
HRESULT ChalkBitmap::ReadBMP(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	bool flip = true;
	if (infile.ReadByte() != 'B')
		return E_FAIL; // Not a valid bitmap file (note that infile auto-closes)
	if (infile.ReadByte() != 'M')
		return E_FAIL; // Not a valid bitmap file (note that infile auto-closes)
	infile.ReadIntelLong(); // Length of file
	infile.ReadIntelWord(); // 0
	infile.ReadIntelWord(); // 0
	long offset = infile.ReadIntelLong(); // Offset to bitmap data
	infile.ReadIntelLong(); // 40
	int width = infile.ReadIntelLong();
	int height = infile.ReadIntelLong();
	if (height < 0)
	{
		height = -height;
		flip = !flip;
	}
	infile.ReadIntelWord(); // 1 colour plane
	int numbits = infile.ReadIntelWord(); // bits per pixel
	long compression = infile.ReadIntelLong();
	infile.ReadIntelLong(); // Size of picture in bytes
	infile.ReadIntelLong(); // Horizontal resolution
	infile.ReadIntelLong(); // Vertical resolution
	int numcolours = infile.ReadIntelLong(); // Used colours
	infile.ReadIntelLong(); // Important colours
	// Each row is a multiple of 4 bytes
	int chwidth = width;
	int padding = 0;
	if (numbits == 1)
	{
		chwidth = rounddiv(width, 32) * 32;
		padding = (chwidth - width) / 8;
		if (numcolours == 0)
			numcolours = 2;
	}
	else if (numbits == 4)
	{
		chwidth = rounddiv(width, 8) * 8;
		padding = (chwidth - width) / 2;
		if (numcolours == 0)
			numcolours = 16;
	}
	else if (numbits == 8)
	{
		chwidth = rounddiv(width, 4) * 4;
		padding = (chwidth - width);
		if (numcolours == 0)
			numcolours = 256;
	}
	else if (numbits == 24)
	{
		chwidth = width;
		padding = (4*rounddiv(width*3, 4) - width*3);
	}

	// Read in the palette
	if (numcolours > 0)
	{
		int i, r, g, b, a;
		m_Palette.SetSize(numcolours);
		for (i = 0; i < numcolours; ++i)
		{
			b = infile.ReadByte();
			g = infile.ReadByte();
			r = infile.ReadByte();
			a = infile.ReadByte();
			m_Palette.m_Data[i] = (r << 16) | (g << 8) | (b);
		}
	}
	else if (numbits <= 8) // Lazy microsoft didn't set a palette
		m_Palette.SetFalseColour();

	m_Character.SetSize(chwidth, height, numbits);
	infile.SetPos(offset);

	// Read in the data
	int x, y;
	if (numbits == 1)
	{
		if (compression == BI_RGB)
		{
			for (y = 0; y < height; ++y)
			{
				ChalkColourIndex *pRow;
				if (flip)
					pRow = m_Character.m_Data + chwidth * (height - 1 - y);
				else
					pRow = m_Character.m_Data + chwidth * y;
				for (x = 0; x < width; x += 8)
				{
					int val = infile.ReadByte();
					*pRow++ = (val & 128) ? 1 : 0;
					*pRow++ = (val & 64) ? 1 : 0;
					*pRow++ = (val & 32) ? 1 : 0;
					*pRow++ = (val & 16) ? 1 : 0;
					*pRow++ = (val & 8) ? 1 : 0;
					*pRow++ = (val & 4) ? 1 : 0;
					*pRow++ = (val & 2) ? 1 : 0;
					*pRow++ = (val & 1) ? 1 : 0;
				}
				for (x = 0; x < padding; ++x)
					infile.ReadByte();
			}
		}
	}
	else if (numbits == 4)
	{
		if (compression == BI_RGB)
		{
			for (y = 0; y < height; ++y)
			{
				ChalkColourIndex *pRow;
				if (flip)
					pRow = m_Character.m_Data + chwidth * (height - 1 - y);
				else
					pRow = m_Character.m_Data + chwidth * y;
				for (x = 0; x < width; x += 2)
				{
					int val = infile.ReadByte();
					*pRow++ = val >> 4;
					*pRow++ = val & 15;
				}
				for (x = 0; x < padding; ++x)
					infile.ReadByte();
			}
		}
		else if (compression == BI_RLE4)
		{
			// TBD ...
		}
	}

	else if (numbits == 8)
	{
		if (compression == BI_RGB)
		{
			for (y = 0; y < height; ++y)
			{
				ChalkColourIndex *pRow;
				if (flip)
					pRow = m_Character.m_Data + chwidth * (height - 1 - y);
				else
					pRow = m_Character.m_Data + chwidth * y;
				for (x = 0; x < width; ++x)
					*pRow++ = infile.ReadByte();
				for (x = 0; x < padding; ++x)
					infile.ReadByte();
			}
		}
		else if (compression == BI_RLE8)
		{
			bool rollover = false;
			for (y = 0; y < height; ++y)
			{
				ChalkColourIndex *pRow;
				if (flip)
					pRow = m_Character.m_Data + chwidth * (height - 1 - y);
				else
					pRow = m_Character.m_Data + chwidth * y;
				ChalkColourIndex *ptr;
				int ch, rep;

				// Load in a scanline
				x = 0;
				while (x < width)
				{
					ch = infile.ReadByte();
					if (ch == EOF)
						break;
					else if (ch > 0)
					{
						rollover = false;
						rep = infile.ReadByte();
						ptr = pRow + x;
						while (ch--)
						{
							*ptr++ = rep; // Set pixels to the repeated colour
							++x;
							if (x >= width)  // Ignore big bits 
							{
								rollover = true;
								break;
							}
						}
					}
					else
					{
						ch = infile.ReadByte();
						if (ch < 2)
						{
							if (ch == 0)
							{
								if (!rollover)
									x = width; // End of line
							}
							else if (ch == 1)
								return hr; // End of bitmap (infile autocloses)
							else // Delta record
							{
								int xoffset = infile.ReadByte();
								int yoffset = infile.ReadByte();
								x += xoffset;
								y += yoffset;
								if (flip)
									pRow = m_Character.m_Data + chwidth * (height - 1 - y);
								else
									pRow = m_Character.m_Data + chwidth * y;
								ptr = pRow + x;
							}
							rollover = false;
						}
						else
						{
							bool padding = (ch & 1) != 0;
							ptr = pRow + x;
							while (ch--)
							{
								// Set pixels to this colour 
								rep = infile.ReadByte();
								*ptr++ = rep; // Set pixels to the repeated colour 
								++x;
								if (x >= width)  // Ignore big bits
								{
									while (ch)
									{
										infile.ReadByte();
										--ch;
									}
									rollover = true;
								}
							}
							if (padding)
								infile.ReadByte();
						}
					}
				}
			}
		}
	}

	else if (numbits == 24)
	{
		if (compression == BI_RGB)
		{
			for (y = 0; y < height; ++y)
			{
				ChalkColourIndex *pRow;
				if (flip)
					pRow = m_Character.m_Data + chwidth * (height - 1 - y);
				else
					pRow = m_Character.m_Data + chwidth * y;
				int r, g, b;
				for (x = 0; x < width; ++x)
				{
					b = infile.ReadByte();
					g = infile.ReadByte();
					r = infile.ReadByte();
					*pRow++ = (r << 16) | (g << 8) | (b);
				}
				for (x = 0; x < padding; ++x)
					infile.ReadByte();
			}
		}
	}

	infile.Close();
	return hr;
}


// ----------------------------------------------------------------------------
// Save a bitmap (Microsoft BMP format)
HRESULT ChalkBitmap::WriteBMP(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int width = m_Character.m_Width;
	int height = m_Character.m_Height;
	int numbits = (m_Character.m_NumPlanes > 8) ? 24 : 8; // 24 bit or 8 bit
	int numcolours = m_Palette.m_NumColours;
	int picsize = (numbits / 8) * width * height;
	int offset = numcolours * 4 + 50;
	int filelength = offset + picsize;

	outfile.WriteByte('B');
	outfile.WriteByte('M');
	outfile.WriteIntelLong(filelength);
	outfile.WriteIntelWord(0);
	outfile.WriteIntelWord(0);
	outfile.WriteIntelLong(offset);

	outfile.WriteIntelLong(40);
	outfile.WriteIntelLong(width);
	outfile.WriteIntelLong(height);
	outfile.WriteIntelWord(1);
	outfile.WriteIntelWord(numbits);
	outfile.WriteIntelLong(BI_RGB);
	outfile.WriteIntelLong(picsize); // Size of picture in bytes
	outfile.WriteIntelLong(1024); // Horizontal resolution
	outfile.WriteIntelLong(1024); // Vertical resolution
	outfile.WriteIntelLong(numcolours);

	// Read in the palette
	{
		int i;
		for (i = 0; i < numcolours; ++i)
		{
			ChalkColourIndex val = m_Palette.m_Data[i];
			outfile.WriteByte(val & 0xff);
			outfile.WriteByte((val >> 8) & 0xff);
			outfile.WriteByte((val >> 16) & 0xff);
			outfile.WriteByte(0);
		}
	}

	// Write out the data
	int x, y;
	if (numbits == 8)
	{
		for (y = 0; y < height; ++y)
		{
			const ChalkColourIndex *pRow = m_Character.m_Data + width * y;
			for (x = 0; x < width; ++x)
				outfile.WriteByte(*pRow++);
		}
	}
	else if (numbits == 24)
	{
		for (y = 0; y < height; ++y)
		{
			const ChalkColourIndex *pRow = m_Character.m_Data + width * y;
			for (x = 0; x < width; ++x)
			{
				ChalkColourIndex val = *pRow++;
				outfile.WriteByte(val & 0xff);
				outfile.WriteByte((val >> 8) & 0xff);
				outfile.WriteByte((val >> 16) & 0xff);
			}
		}
	}

	outfile.Close();
	return hr;
}
