// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileBOXFRK.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2000 by Carl Muller
//
// Supports BOX and FRK file formats (from legacy graphics editors)
// using their documented file formats.
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
// Read in a BOXS chunk from a FRK file, or a POSN chunk from a BOX file
static void ReadChunkBOXS(ChalkFrame& frame, IFFfile& infile)
{
	int sbox, shdr, nboxes;
	int i, j;

	nboxes = infile.ReadWord();   // Number of boxes
	sbox = infile.ReadWord();     // Bytes per record (8..16)
	shdr = infile.ReadWord();     // Bytes of header (0)
	for (i = 0; i < shdr; ++i)
		infile.ReadByte();

	frame.m_Boxes.resize(nboxes);
	for (i = 0; i < nboxes; ++i)
	{
		ChalkBox& box = frame.m_Boxes[i];
		box.m_Xpos = 0;
		box.m_Ypos = 0;
		box.m_Width = 16;
		box.m_Height = 16;
		box.m_Type = 0;
		box.m_OffsetX = 0;
		box.m_OffsetY = 0;
		int extrasize = 0;

		if (sbox >= 2)
			box.m_Xpos = infile.ReadWord();
		if (sbox >= 4)
			box.m_Ypos = infile.ReadWord();
		if (sbox >= 6)
			box.m_Width = infile.ReadWord() - box.m_Xpos;
		if (sbox >= 8)
			box.m_Height = infile.ReadWord() - box.m_Ypos;
		if (sbox >= 10)
			box.m_Type = infile.ReadWord();
		if (sbox >= 12)
			box.m_OffsetX = (short) infile.ReadWord();
		if (sbox >= 14)
			box.m_OffsetY = (short) infile.ReadWord();
		if (sbox >= 16)
			extrasize = infile.ReadWord();
		for (j = 16; j < sbox; ++j)
			infile.ReadByte();
		box.m_Script.empty();
		if (extrasize)
		{
			for (j = 0; j < extrasize; ++j)
				box.m_Script += infile.ReadByte();
		}
	}
}

// ----------------------------------------------------------------------------
// Read in a POSN chunk from a FRK file
static void ReadChunkFRAK(ChalkFrame& frame, IFFfile& infile)
{
	int hdrsize, recsize, numsprites;
	int i, j;

	numsprites = infile.ReadWord();
	recsize = infile.ReadWord(); // Bytes per record
	hdrsize = infile.ReadWord(); // Header size
	for (i = 0; i < hdrsize; ++i)
		infile.ReadByte();

	frame.m_Sprites.resize(numsprites);
	for (i = 0; i < numsprites; ++i)
	{
		ChalkSprite& sprite = frame.m_Sprites[i];

		sprite.m_Script.resize(0); // .clear when STL is supported properly by Microsoft;
		sprite.m_Selected = false;
		sprite.m_ObjDefn = 0;
		sprite.m_Xpos = 0;
		sprite.m_Ypos= 0;
		sprite.m_Zpos = 0;
		int extrasize = 0;

		if (recsize >= 2)
			sprite.m_ObjDefn = infile.ReadWord();
		if (recsize >= 4)
			sprite.m_Xpos = infile.ReadWord();
		if (recsize >= 6)
			sprite.m_Ypos = infile.ReadWord();
		if (recsize >= 8)
			extrasize = infile.ReadWord();
		if (recsize >= 10)
			sprite.m_Zpos = infile.ReadWord();
		for (j = 10; j < recsize; ++j)
			infile.ReadByte();
		sprite.m_Script.empty();
		if (extrasize > 0)
		{
			for (j = 0; j < extrasize; ++j)
				sprite.m_Script += infile.ReadByte();
		}
	}
}


// ----------------------------------------------------------------------------
// Read in a PATH chunk from a FRK file or BOX file
static void ReadChunkPATH(ChalkFrame& frame, IFFfile& infile)
{
	int hdrsize, recsize, numpaths;
	int i, j;

	numpaths = infile.ReadWord();
	recsize = infile.ReadWord(); // Bytes per record
	hdrsize = infile.ReadWord(); // Header size
	for (i = 0; i < hdrsize; ++i)
		infile.ReadByte();

	frame.m_Paths.resize(numpaths);
	for (i = 0; i < numpaths; ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		path.m_Script.resize(0); // .clear when STL is supported properly by Microsoft;
		path.m_Type = 0;
		path.m_ObjDefn = 0;
		path.m_Speed = 60.0;
		path.m_Selected = false;
		int zpos = 0;
		int numpoints = 0;
		int extrasize = 0;

		if (recsize >= 2)
			numpoints = infile.ReadWord();
		if (recsize >= 4)
			path.m_Type = infile.ReadWord();
		if (recsize >= 6)
			extrasize = infile.ReadWord();
		if (recsize >= 8)
			zpos = infile.ReadWord();
		if (recsize >= 10)
			path.m_ObjDefn = infile.ReadWord();
		if (recsize >= 12)
			path.m_Speed = infile.ReadWord() / 40.0;
		for (j = 12; j < recsize; ++j)
			infile.ReadByte();

		if (numpoints > 0)
		{
			path.m_Data.resize(numpoints);
			for (j = 0; j < numpoints; ++j)
			{
				ChalkPoint& point = path.m_Data[j];
				point.m_Xpos = (short) infile.ReadWord();
				point.m_Ypos = (short) infile.ReadWord();
				point.m_Zpos = zpos;
			}
		}
		path.m_Script.empty();
		if (extrasize > 0)
		{
			for (j = 0; j < extrasize; ++j)
				path.m_Script += infile.ReadByte();
		}
	}
}


// ----------------------------------------------------------------------------
// Read in a LINE chunk from a FRK file or BOX file
// This has a subset of the PATH chunk functionality
// Note that it should come AFTER the PATH chunk of the appropriate frame
static void ReadChunkLINE(ChalkFrame& frame, IFFfile& infile)
{
	int srecord, sheader, numlines;
	int i, j;

	numlines = infile.ReadWord();     // Number of lines
	srecord = infile.ReadWord();    // Bytes per record (8)
	sheader = infile.ReadWord();     // Bytes of header (0)
	for (i = 0; i < sheader; ++i)
		infile.ReadByte();

	int numpaths = frame.m_Paths.size();
	frame.m_Paths.resize(numpaths + numlines);
	for (i = numpaths; i < numpaths + numlines; ++i)
	{
		ChalkPath& path = frame.m_Paths[i];
		path.m_Script.resize(0); // .clear when STL is supported properly by Microsoft;
		path.m_Type = 0;
		path.m_ObjDefn = 0;
		path.m_Speed = 6.0;
		path.m_Selected = false;
		path.m_Data.resize(2);
		int extrasize = 0;

		if (srecord >= 2)
			path.m_Data[0].m_Xpos = infile.ReadWord();
		if (srecord >= 4)
			path.m_Data[0].m_Ypos = infile.ReadWord();
		if (srecord >= 6)
			path.m_Data[1].m_Xpos = infile.ReadWord();
		if (srecord >= 8)
			path.m_Data[1].m_Ypos = infile.ReadWord();
		if (srecord >= 10)
			path.m_Type = infile.ReadWord();
		if (srecord >= 12)
			extrasize = infile.ReadWord();
		for (j = 12; j < srecord; ++j)
			infile.ReadByte();
		path.m_Script.empty();
		if (extrasize > 0)
		{
			for (j = 0; j < extrasize; ++j)
				path.m_Script += infile.ReadByte();
		}
	}
}

// ----------------------------------------------------------------------------
HRESULT ChalkAnimation::ReadBOX(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int currentframe = -1;
	CFileName auxblockname;

	// Parse IFF file
	long type, len, pos;
	type = infile.ReadLong();
	len = infile.ReadLong();
	if (type == CHUNK_FORM)
	{
		type = infile.ReadLong();
		if (type == CHUNK_BOXS)
		{
			while (!infile.EndOfFile())
			{
				type = infile.ReadLong();
				len = infile.ReadLong();
				pos = infile.GetPos();

				if (type == CHUNK_KIDS) // Read in the palette
				{
					tstring temp1;
					int ch;

					while ((ch = infile.ReadByte()) >= ' ')
						temp1 += ch;
					m_KidName = CFileName::MakeAbsolute(filename, temp1);
					if (ch > 0)
					{
						while ((ch = infile.ReadByte()) >= ' ')
							temp1 += ch;
						auxblockname = CFileName::MakeAbsolute(filename, temp1);
					}
				}

				else if (type == CHUNK_POSN) // Read in the data
				{
					// Create a frame and some boxes
					currentframe++;
					m_Frames.resize(currentframe + 1);
					ChalkFrame& frame = m_Frames[currentframe];
					ReadChunkBOXS(frame, infile);
				}

				else if (type == CHUNK_PATH) // Read in the data
				{
					if (currentframe >= 0)
					{
						ChalkFrame& frame = m_Frames[currentframe];
						ReadChunkPATH(frame, infile);
					}
				}

				else if (type == CHUNK_LINE) // Read in the data
				{
					if (currentframe >= 0)
					{
						ChalkFrame& frame = m_Frames[currentframe];
						ReadChunkLINE(frame, infile);
					}
				}

				else if (type == CHUNK_GRID) // Read in the data
				{
					infile.ReadWord();
					if (infile.ReadWord() >= 2) // At least two grids specified
					{
						infile.ReadWord();
						infile.ReadWord();
						m_GridX = infile.ReadWord();
						m_GridY = infile.ReadWord();
					}
				}

				// Find next chunk (if possible)
				if (len & 1)
					++len;
				infile.SetPos(pos + len);
			}
		}
	}
	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
// Load block sets
HRESULT ChalkAnimation::LoadBlocks()
{
	m_Font.Clear();

	int defnindex;
	for (defnindex = 0; defnindex < m_DefnNames.size(); ++defnindex)
	{
		CFileName fname = m_DefnNames[defnindex];

		if (fname.HasExt(FILEEXTENSION_BOX))
			m_Font.ReadBOX(fname.Get(), &m_Palette, defnindex != 0);
		else if (fname.HasExt(FILEEXTENSION_PXP))
			m_Font.ReadPXP(fname.Get(), &m_Palette, defnindex != 0);
		else if (ChalkBitmap::SupportsFile(fname.Get()) && defnindex == 0)
			m_Font.ReadFile(fname.Get());
		//TBD... Support MAT file
	}
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkAnimation::ReadFRK(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int currentframe = -1;
	CFileName auxblockname;

	// Parse IFF file
	long type, len, pos;
	type = infile.ReadLong();
	len = infile.ReadLong();
	if (type == CHUNK_FORM)
	{
		type = infile.ReadLong();
		if (type == CHUNK_FRAK)
		{
			while (!infile.EndOfFile())
			{
				type = infile.ReadLong();
				len = infile.ReadLong();
				pos = infile.GetPos();

				if (type == CHUNK_KIDS) // Read in the background layer name
				{
					tstring temp1;
					int ch;

					while ((ch = infile.ReadByte()) >= ' ')
						temp1 += ch;
					m_KidName = CFileName::MakeAbsolute(filename, temp1);
					if (ch > 0)
					{
						temp1.empty();
						while ((ch = infile.ReadByte()) >= ' ')
							temp1 += ch;
						auxblockname = CFileName::MakeAbsolute(filename, temp1);
					}
				}

				else if (type == CHUNK_DKID)
				{
					wstring temp;
					int ch = ' ';
					m_DefnNames.clear();

					while (ch)
					{
						temp = infile.ReadString("\r\n", "", &ch);
						m_DefnNames.push_back(CFileName::MakeAbsolute(filename, temp));
					}

					// Load block sets
					LoadBlocks();
				}

				else if (type == CHUNK_POSN) // Read in the sprite data
				{
					// Create a frame
					currentframe++;
					m_Frames.resize(currentframe + 1);
					// Read in sprite position data
					ChalkFrame& frame = m_Frames[currentframe];
					ReadChunkFRAK(frame, infile);
				}

				else if (type == CHUNK_BOXS) // Read in the box data
				{
					if (currentframe >= 0)
					{
						ChalkFrame& frame = m_Frames[currentframe];
						ReadChunkBOXS(frame, infile);
					}
				}

				else if (type == CHUNK_PATH) // Read in the data
				{
					if (currentframe >= 0)
					{
						ChalkFrame& frame = m_Frames[currentframe];
						ReadChunkPATH(frame, infile);
					}
				}

				else if (type == CHUNK_LINE) // Read in the data
				{
					if (currentframe >= 0)
					{
						ChalkFrame& frame = m_Frames[currentframe];
						ReadChunkLINE(frame, infile);
					}
				}

				else if (type == CHUNK_GRID) // Read in the data
				{
					infile.ReadWord();
					if (infile.ReadWord() >= 2) // At least two grids specified
					{
						infile.ReadWord();
						infile.ReadWord();
						m_GridX = infile.ReadWord();
						m_GridY = infile.ReadWord();
					}
				}

				else if (type == CHUNK_ANIM) // Read in animation info
				{
					infile.ReadWord(); // Number of animation records
					infile.ReadWord(); // Bytes per record
					if (infile.ReadWord() >= 2) // Bytes of additional header
					{
						m_Speed = infile.ReadWord() / 40.0;
					}
					// Ignore the animation data itself...
				}

				// Find next chunk (if possible)
				if (len & 1)
					++len;
				infile.SetPos(pos + len);
			}
		}
	}
	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkAnimation::WriteBOX(const CFileName& filename) const
{
	IFFfile outfile;
	wstring temp;
	int i, j;

	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	CFileName auxblockname;

	// Write out header of box file
	outfile.Form(CHUNK_BOXS);

	// Write out info about blocks
	outfile.Chunk(CHUNK_KIDS);
	temp = CFileName::MakeRelative(filename, m_KidName);
	outfile.WriteString(temp);
	if (auxblockname.IsValid())
	{
		outfile.WriteByte('\r');
		temp = CFileName::MakeRelative(filename, auxblockname);
		outfile.WriteString(temp);
	}
	outfile.WriteByte(0);
	outfile.EndChunk();

	// Write out the GRID chunk
	outfile.Chunk(CHUNK_GRID);
	outfile.WriteWord(0); // selected grid
	outfile.WriteWord(3); // number of grids
	outfile.WriteWord(1); // gridx
	outfile.WriteWord(1); // gridy
	outfile.WriteWord(8); // grid2x
	outfile.WriteWord(8); // grid2y
	outfile.WriteWord(0); // grid2xoff
	outfile.WriteWord(0); // grid2yoff
	outfile.EndChunk();

	for (int currentframe = 0; currentframe < m_Frames.size(); ++currentframe)
	{
		const ChalkFrame& frame = m_Frames[currentframe];
		int numboxes = frame.m_Boxes.size();
				
		// Write out body of file
		outfile.Chunk(CHUNK_POSN);
		outfile.WriteWord(numboxes); // Number of boxes
		outfile.WriteWord(14);       // Bytes per record
		outfile.WriteWord(0);        // Bytes of header
		for (i = 0; i < numboxes; ++i)
		{
			const ChalkBox& box = frame.m_Boxes[i];

			outfile.WriteWord(box.m_Xpos);
			outfile.WriteWord(box.m_Ypos);
			outfile.WriteWord(box.m_Xpos + box.m_Width);
			outfile.WriteWord(box.m_Ypos + box.m_Height);
			outfile.WriteWord(box.m_Type);
			outfile.WriteWord(box.m_OffsetX);
			outfile.WriteWord(box.m_OffsetY);
		}
		outfile.EndChunk();

		// LINE chunk not needed (subsumed into PATH)

		// Output all the paths
		outfile.Chunk(CHUNK_PATH);
		int numpaths = frame.m_Paths.size();
		outfile.WriteWord(numpaths);    // Number of paths in this frame
		outfile.WriteWord(12);    // Bytes per record
		outfile.WriteWord(0);     // Bytes of header
		for (i = 0; i < numpaths; ++i)
		{
			const ChalkPath& path = frame.m_Paths[i];
			int sizepath = path.m_Data.size();
			outfile.WriteWord(sizepath);
			outfile.WriteWord(path.m_Type);
			outfile.WriteWord(path.m_Script.length() + 1);
			outfile.WriteWord(path.m_Data[0].m_Zpos);
			outfile.WriteWord(path.m_ObjDefn);
			outfile.WriteWord(int(path.m_Speed * 40.0));
			for (j = 0; j < sizepath; ++j)
			{
				const ChalkPoint& point = path.m_Data[j];
				outfile.WriteWord(point.m_Xpos);
				outfile.WriteWord(point.m_Ypos);
			}
			outfile.WriteString(path.m_Script);
			outfile.WriteByte(0);
		}
		outfile.EndChunk();
	}

	// Finish off file
	outfile.EndForm();
	outfile.Close();
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkAnimation::WriteFRK(const CFileName& filename) const
{
	IFFfile outfile;
	wstring temp;
	int i, j;

	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	// Write out header of frak file
	outfile.Form(CHUNK_FRAK);

	// Write out info about background layer
	outfile.Chunk(CHUNK_KIDS);
	temp = CFileName::MakeRelative(filename, m_KidName.Get());
	outfile.WriteString(temp);
	outfile.WriteByte(0);
	outfile.EndChunk();

	// Write out info about defn files
	if (m_DefnNames.size() > 0)
	{
		outfile.Chunk(CHUNK_DKID);
		// Write out info about defn files
		for (i = 0; i < m_DefnNames.size(); ++i)
		{
			temp = CFileName::MakeRelative(filename, m_DefnNames[i]);
			outfile.WriteString(temp);
			if (i < m_DefnNames.size() - 1)
				outfile.WriteByte('\r');
		}
		outfile.WriteByte(0);
		outfile.EndChunk();
	}

	// Dummy ANIM chunk
	outfile.Chunk(CHUNK_ANIM);
	outfile.WriteWord(0);			// Number of records
	outfile.WriteWord(6);			// Bytes per record
	outfile.WriteWord(2);			// Bytes of header
	outfile.WriteWord(int(m_Speed * 40.0));	// Animation speed
	outfile.EndChunk();

	// Write CMAP / CRNG chunks
	m_Palette.WritePalette(outfile);

	// Write out the GRID chunk
	outfile.Chunk(CHUNK_GRID);
	outfile.WriteWord(0); // selected grid
	outfile.WriteWord(3); // number of grids
	outfile.WriteWord(1); // gridx
	outfile.WriteWord(1); // gridy
	outfile.WriteWord(8); // grid2x
	outfile.WriteWord(8); // grid2y
	outfile.WriteWord(0); // grid2xoff
	outfile.WriteWord(0); // grid2yoff
	outfile.EndChunk();

	for (int currentframe = 0; currentframe < m_Frames.size(); ++currentframe)
	{
		const ChalkFrame& frame = m_Frames[currentframe];
		
		// Output all the sprite positions
		int numsprites = frame.m_Sprites.size();
		outfile.Chunk(CHUNK_POSN);
		outfile.WriteWord(numsprites); // Number of records
		outfile.WriteWord(10);         // Bytes per record
		outfile.WriteWord(0);          // Bytes of header
		// Body
		for (i = 0; i < numsprites; ++i)
		{
			const ChalkSprite& sprite = frame.m_Sprites[i];
			outfile.WriteWord(sprite.m_ObjDefn);
			outfile.WriteWord(sprite.m_Xpos);
			outfile.WriteWord(sprite.m_Ypos);
			outfile.WriteWord(sprite.m_Script.length() + 1);
			outfile.WriteWord(sprite.m_Zpos);
			outfile.WriteString(sprite.m_Script);
			outfile.WriteByte(0);
		}
		outfile.EndChunk();

		// Output all the boxes
		int numboxes = frame.m_Boxes.size();
		outfile.Chunk(CHUNK_BOXS);
		outfile.WriteWord(numboxes); // Number of boxes
		outfile.WriteWord(16);       // Bytes per record
		outfile.WriteWord(0);        // Bytes of header
		for (i = 0; i < numboxes; ++i)
		{
			const ChalkBox& box = frame.m_Boxes[i];

			outfile.WriteWord(box.m_Xpos);
			outfile.WriteWord(box.m_Ypos);
			outfile.WriteWord(box.m_Xpos + box.m_Width);
			outfile.WriteWord(box.m_Ypos + box.m_Height);
			outfile.WriteWord(box.m_Type);
			outfile.WriteWord(box.m_OffsetX);
			outfile.WriteWord(box.m_OffsetY);
			if (box.m_Script.empty())
				outfile.WriteWord(0);
			else
			{
				outfile.WriteWord(box.m_Script.length() + 1);
				outfile.WriteString(box.m_Script);
				outfile.WriteByte(0);
			}
		}
		outfile.EndChunk();

		// LINE chunk not needed (subsumed into PATH)

		// Output all the paths
		outfile.Chunk(CHUNK_PATH);
		int numpaths = frame.m_Paths.size();
		outfile.WriteWord(numpaths);    // Number of paths in this frame
		outfile.WriteWord(12);    // Bytes per record
		outfile.WriteWord(0);     // Bytes of header
		for (i = 0; i < numpaths; ++i)
		{
			const ChalkPath& path = frame.m_Paths[i];
			int sizepath = path.m_Data.size();
			outfile.WriteWord(sizepath);
			outfile.WriteWord(path.m_Type);
			outfile.WriteWord(path.m_Script.length() + 1);
			outfile.WriteWord(path.m_Data[0].m_Zpos);
			outfile.WriteWord(path.m_ObjDefn);
			outfile.WriteWord(int(path.m_Speed * 40.0));
			for (j = 0; j < sizepath; ++j)
			{
				const ChalkPoint& point = path.m_Data[j];
				outfile.WriteWord(point.m_Xpos);
				outfile.WriteWord(point.m_Ypos);
			}
			outfile.WriteString(path.m_Script);
			outfile.WriteByte(0);
		}
		outfile.EndChunk();
	}


	// Finish off file
	outfile.EndForm();
	outfile.Close();
	return S_OK;
}
