// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileCHR12.cpp
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 16 Jul 2000 by Carl Muller
//
// Supports character set file format (1 bit and 2 bits)
// These are used by various machines
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
// ChalkFont
// A character set, or a collection of icons
// ----------------------------------------------------------------------------
const int CHARSIZE = 8;

// ----------------------------------------------------------------------------
// Read in 1 and 2 bit character formats
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadCHR(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_C64;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 1 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);
	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 1);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val = infile.ReadByte();
			*pData++ = (val & 128) ? 1 : 0;
			*pData++ = (val & 64) ? 1 : 0;
			*pData++ = (val & 32) ? 1 : 0;
			*pData++ = (val & 16) ? 1 : 0;
			*pData++ = (val & 8) ? 1 : 0;
			*pData++ = (val & 4) ? 1 : 0;
			*pData++ = (val & 2) ? 1 : 0;
			*pData++ = (val & 1) ? 1 : 0;
		}
	}
	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadSET(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_NES;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 2 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);
	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 2);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val = infile.ReadByte();
			*pData++ = (val & 128) ? 1 : 0;
			*pData++ = (val & 64) ? 1 : 0;
			*pData++ = (val & 32) ? 1 : 0;
			*pData++ = (val & 16) ? 1 : 0;
			*pData++ = (val & 8) ? 1 : 0;
			*pData++ = (val & 4) ? 1 : 0;
			*pData++ = (val & 2) ? 1 : 0;
			*pData++ = (val & 1) ? 1 : 0;
		}
		pData -= CHARSIZE * CHARSIZE;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val = infile.ReadByte();
			*pData++ |= (val & 128) ? 2 : 0;
			*pData++ |= (val & 64) ? 2 : 0;
			*pData++ |= (val & 32) ? 2 : 0;
			*pData++ |= (val & 16) ? 2 : 0;
			*pData++ |= (val & 8) ? 2 : 0;
			*pData++ |= (val & 4) ? 2 : 0;
			*pData++ |= (val & 2) ? 2 : 0;
			*pData++ |= (val & 1) ? 2 : 0;
		}
	}
	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadGBC(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_GAMEBOY;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 2 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);
	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 2);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ = ((val1 & 128) ? 1 : 0) | ((val2 & 128) ? 2 : 0);
			*pData++ = ((val1 & 64) ? 1 : 0) | ((val2 & 64) ? 2 : 0);
			*pData++ = ((val1 & 32) ? 1 : 0) | ((val2 & 32) ? 2 : 0);
			*pData++ = ((val1 & 16) ? 1 : 0) | ((val2 & 16) ? 2 : 0);
			*pData++ = ((val1 & 8) ? 1 : 0) | ((val2 & 8) ? 2 : 0);
			*pData++ = ((val1 & 4) ? 1 : 0) | ((val2 & 4) ? 2 : 0);
			*pData++ = ((val1 & 2) ? 1 : 0) | ((val2 & 2) ? 2 : 0);
			*pData++ = ((val1 & 1) ? 1 : 0) | ((val2 & 1) ? 2 : 0);
		}
	}
	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadSF2(const CFileName& filename)
{
	HRESULT hr = ReadGBC(filename);
	m_SystemType = GUID_SYSTEM_SNES;
	return hr;
}


// ----------------------------------------------------------------------------
// Write 1 and 2 bit character formats
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteCHR(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int numchars = m_Data.size();
	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		const ChalkCharacter& character = m_Data[i];
		const ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val = 0;
			if ((*pData++) & 1) val |= 128;
			if ((*pData++) & 1) val |= 64;
			if ((*pData++) & 1) val |= 32;
			if ((*pData++) & 1) val |= 16;
			if ((*pData++) & 1) val |= 8;
			if ((*pData++) & 1) val |= 4;
			if ((*pData++) & 1) val |= 2;
			if ((*pData++) & 1) val |= 1;
			outfile.WriteByte(val);
		}
	}
	outfile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteSET(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int numchars = m_Data.size();
	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		const ChalkCharacter& character = m_Data[i];
		const ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val = 0;
			if ((*pData++) & 1) val |= 128;
			if ((*pData++) & 1) val |= 64;
			if ((*pData++) & 1) val |= 32;
			if ((*pData++) & 1) val |= 16;
			if ((*pData++) & 1) val |= 8;
			if ((*pData++) & 1) val |= 4;
			if ((*pData++) & 1) val |= 2;
			if ((*pData++) & 1) val |= 1;
			outfile.WriteByte(val);
		}
		pData -= CHARSIZE * CHARSIZE;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val = 0;
			if ((*pData++) & 2) val |= 128;
			if ((*pData++) & 2) val |= 64;
			if ((*pData++) & 2) val |= 32;
			if ((*pData++) & 2) val |= 16;
			if ((*pData++) & 2) val |= 8;
			if ((*pData++) & 2) val |= 4;
			if ((*pData++) & 2) val |= 2;
			if ((*pData++) & 2) val |= 1;
			outfile.WriteByte(val);
		}
	}
	outfile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteGBC(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int numchars = m_Data.size();
	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		const ChalkCharacter& character = m_Data[i];
		const ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = 0;
			int val2 = 0;
			if ((*pData++) & 1) val1 |= 128;
			if ((*pData++) & 1) val1 |= 64;
			if ((*pData++) & 1) val1 |= 32;
			if ((*pData++) & 1) val1 |= 16;
			if ((*pData++) & 1) val1 |= 8;
			if ((*pData++) & 1) val1 |= 4;
			if ((*pData++) & 1) val1 |= 2;
			if ((*pData++) & 1) val1 |= 1;
			pData -= CHARSIZE;
			if ((*pData++) & 2) val2 |= 128;
			if ((*pData++) & 2) val2 |= 64;
			if ((*pData++) & 2) val2 |= 32;
			if ((*pData++) & 2) val2 |= 16;
			if ((*pData++) & 2) val2 |= 8;
			if ((*pData++) & 2) val2 |= 4;
			if ((*pData++) & 2) val2 |= 2;
			if ((*pData++) & 2) val2 |= 1;
			outfile.WriteByte(val1);
			outfile.WriteByte(val2);
		}
	}
	outfile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteSF2(const CFileName& filename) const
{
	return WriteGBC(filename);
}

