// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileCHR48.cpp
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports character set file format (4 bits and 8 bits)
// These are used by various machines
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
// ChalkFont
// A character set, or a collection of icons
// ----------------------------------------------------------------------------
const int CHARSIZE = 8;


// ----------------------------------------------------------------------------
// Read in various file formats
// ----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadSF4(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_SNES;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 4 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);

	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 4);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ = ((val1 & 128) ? 1 : 0) | ((val2 & 128) ? 2 : 0);
			*pData++ = ((val1 & 64) ? 1 : 0) | ((val2 & 64) ? 2 : 0);
			*pData++ = ((val1 & 32) ? 1 : 0) | ((val2 & 32) ? 2 : 0);
			*pData++ = ((val1 & 16) ? 1 : 0) | ((val2 & 16) ? 2 : 0);
			*pData++ = ((val1 & 8) ? 1 : 0) | ((val2 & 8) ? 2 : 0);
			*pData++ = ((val1 & 4) ? 1 : 0) | ((val2 & 4) ? 2 : 0);
			*pData++ = ((val1 & 2) ? 1 : 0) | ((val2 & 2) ? 2 : 0);
			*pData++ = ((val1 & 1) ? 1 : 0) | ((val2 & 1) ? 2 : 0);
		}
		pData -= CHARSIZE * CHARSIZE;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ |= ((val1 & 128) ? 4 : 0) | ((val2 & 128) ? 8 : 0);
			*pData++ |= ((val1 & 64) ? 4 : 0) | ((val2 & 64) ? 8 : 0);
			*pData++ |= ((val1 & 32) ? 4 : 0) | ((val2 & 32) ? 8 : 0);
			*pData++ |= ((val1 & 16) ? 4 : 0) | ((val2 & 16) ? 8 : 0);
			*pData++ |= ((val1 & 8) ? 4 : 0) | ((val2 & 8) ? 8 : 0);
			*pData++ |= ((val1 & 4) ? 4 : 0) | ((val2 & 4) ? 8 : 0);
			*pData++ |= ((val1 & 2) ? 4 : 0) | ((val2 & 2) ? 8 : 0);
			*pData++ |= ((val1 & 1) ? 4 : 0) | ((val2 & 1) ? 8 : 0);
		}
	}

	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadMCH(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_MASTER;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 4 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);

	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 4);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			int val3 = infile.ReadByte();
			int val4 = infile.ReadByte();
			*pData++ = ((val1 & 128) ? 1 : 0) | ((val2 & 128) ? 2 : 0)
				| ((val3 & 128) ? 4 : 0) | ((val4 & 128) ? 8 : 0);
			*pData++ = ((val1 & 64) ? 1 : 0) | ((val2 & 64) ? 2 : 0)
				| ((val3 & 64) ? 4 : 0) | ((val4 & 64) ? 8 : 0);
			*pData++ = ((val1 & 32) ? 1 : 0) | ((val2 & 32) ? 2 : 0)
				| ((val3 & 32) ? 4 : 0) | ((val4 & 32) ? 8 : 0);
			*pData++ = ((val1 & 16) ? 1 : 0) | ((val2 & 16) ? 2 : 0)
				| ((val3 & 16) ? 4 : 0) | ((val4 & 16) ? 8 : 0);
			*pData++ = ((val1 & 8) ? 1 : 0) | ((val2 & 8) ? 2 : 0)
				| ((val3 & 8) ? 4 : 0) | ((val4 & 8) ? 8 : 0);
			*pData++ = ((val1 & 4) ? 1 : 0) | ((val2 & 4) ? 2 : 0)
				| ((val3 & 4) ? 4 : 0) | ((val4 & 4) ? 8 : 0);
			*pData++ = ((val1 & 2) ? 1 : 0) | ((val2 & 2) ? 2 : 0)
				| ((val3 & 2) ? 4 : 0) | ((val4 & 2) ? 8 : 0);
			*pData++ = ((val1 & 1) ? 1 : 0) | ((val2 & 1) ? 2 : 0)
				| ((val3 & 1) ? 4 : 0) | ((val4 & 1) ? 8 : 0);
		}
	}

	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadGCH(const CFileName& filename)
{
	HRESULT hr = ReadMCH(filename);
	m_SystemType = GUID_SYSTEM_GAMEGEAR;
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadSCH(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_MEGADRIVE;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 4 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);
	int i, x, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 4);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
			for (x = 0; x < CHARSIZE; x += 2)
			{
				int val = infile.ReadByte();
				*pData++ = val >> 4;
				*pData++ = val & 15;
			}
	}
	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadSF8(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_SNES;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 8 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);

	int i, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 8);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ = ((val1 & 128) ? 1 : 0) | ((val2 & 128) ? 2 : 0);
			*pData++ = ((val1 & 64) ? 1 : 0) | ((val2 & 64) ? 2 : 0);
			*pData++ = ((val1 & 32) ? 1 : 0) | ((val2 & 32) ? 2 : 0);
			*pData++ = ((val1 & 16) ? 1 : 0) | ((val2 & 16) ? 2 : 0);
			*pData++ = ((val1 & 8) ? 1 : 0) | ((val2 & 8) ? 2 : 0);
			*pData++ = ((val1 & 4) ? 1 : 0) | ((val2 & 4) ? 2 : 0);
			*pData++ = ((val1 & 2) ? 1 : 0) | ((val2 & 2) ? 2 : 0);
			*pData++ = ((val1 & 1) ? 1 : 0) | ((val2 & 1) ? 2 : 0);
		}
		pData -= CHARSIZE * CHARSIZE;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ |= ((val1 & 128) ? 4 : 0) | ((val2 & 128) ? 8 : 0);
			*pData++ |= ((val1 & 64) ? 4 : 0) | ((val2 & 64) ? 8 : 0);
			*pData++ |= ((val1 & 32) ? 4 : 0) | ((val2 & 32) ? 8 : 0);
			*pData++ |= ((val1 & 16) ? 4 : 0) | ((val2 & 16) ? 8 : 0);
			*pData++ |= ((val1 & 8) ? 4 : 0) | ((val2 & 8) ? 8 : 0);
			*pData++ |= ((val1 & 4) ? 4 : 0) | ((val2 & 4) ? 8 : 0);
			*pData++ |= ((val1 & 2) ? 4 : 0) | ((val2 & 2) ? 8 : 0);
			*pData++ |= ((val1 & 1) ? 4 : 0) | ((val2 & 1) ? 8 : 0);
		}
		pData -= CHARSIZE * CHARSIZE;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ |= ((val1 & 128) ? 16 : 0) | ((val2 & 128) ? 32 : 0);
			*pData++ |= ((val1 & 64) ? 16 : 0) | ((val2 & 64) ? 32 : 0);
			*pData++ |= ((val1 & 32) ? 16 : 0) | ((val2 & 32) ? 32 : 0);
			*pData++ |= ((val1 & 16) ? 16 : 0) | ((val2 & 16) ? 32 : 0);
			*pData++ |= ((val1 & 8) ? 16 : 0) | ((val2 & 8) ? 32 : 0);
			*pData++ |= ((val1 & 4) ? 16 : 0) | ((val2 & 4) ? 32 : 0);
			*pData++ |= ((val1 & 2) ? 16 : 0) | ((val2 & 2) ? 32 : 0);
			*pData++ |= ((val1 & 1) ? 16 : 0) | ((val2 & 1) ? 32 : 0);
		}
		pData -= CHARSIZE * CHARSIZE;
		for (y = 0; y < CHARSIZE; ++y)
		{
			int val1 = infile.ReadByte();
			int val2 = infile.ReadByte();
			*pData++ |= ((val1 & 128) ? 64 : 0) | ((val2 & 128) ? 128 : 0);
			*pData++ |= ((val1 & 64) ? 64 : 0) | ((val2 & 64) ? 128 : 0);
			*pData++ |= ((val1 & 32) ? 64 : 0) | ((val2 & 32) ? 128 : 0);
			*pData++ |= ((val1 & 16) ? 64 : 0) | ((val2 & 16) ? 128 : 0);
			*pData++ |= ((val1 & 8) ? 64 : 0) | ((val2 & 8) ? 128 : 0);
			*pData++ |= ((val1 & 4) ? 64 : 0) | ((val2 & 4) ? 128 : 0);
			*pData++ |= ((val1 & 2) ? 64 : 0) | ((val2 & 2) ? 128 : 0);
			*pData++ |= ((val1 & 1) ? 64 : 0) | ((val2 & 1) ? 128 : 0);
		}
	}

	infile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::ReadSFB(const CFileName& filename)
{
	m_SystemType = GUID_SYSTEM_SNESMODE7;

	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int numchars = infile.GetLength() / (CHARSIZE * CHARSIZE * 8 / 8);
	if (numchars <= 0)
		return E_FAIL;
	m_Data.resize(numchars);
	int i, x, y;
	for (i = 0; i < numchars; ++i)
	{
		ChalkCharacter& character = m_Data[i];
		character.SetSize(CHARSIZE, CHARSIZE, 1);
		ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
			for (x = 0; x < CHARSIZE; ++x)
				*pData++ = infile.ReadByte();
	}
	infile.Close();
	return hr;
}



// ----------------------------------------------------------------------------
// Write files in various formats
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteSF4(const CFileName& filename) const
{
	// TBD
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteMCH(const CFileName& filename) const
{
	// TBD
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteGCH(const CFileName& filename) const
{
	return WriteMCH(filename);
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteSCH(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int numchars = m_Data.size();
	int i, x, y;
	for (i = 0; i < numchars; ++i)
	{
		const ChalkCharacter& character = m_Data[i];
		const ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
		{
			for (x = 0; x < CHARSIZE; x += 2)
			{
				int val1 = (*pData++) & 15;
				int val2 = (*pData++) & 15;
				outfile.WriteByte((val1 << 4) | val2);
			}
		}
	}
	outfile.Close();
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteSF8(const CFileName& filename) const
{
	// ...
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkFont::WriteSFB(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int numchars = m_Data.size();
	int i, x, y;
	for (i = 0; i < numchars; ++i)
	{
		const ChalkCharacter& character = m_Data[i];
		const ChalkColourIndex *pData = character.m_Data;
		for (y = 0; y < CHARSIZE; ++y)
			for (x = 0; x < CHARSIZE; ++x)
				outfile.WriteByte(*pData++);
	}
	outfile.Close();
	return hr;
}

