// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileLBM.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports LBM and PBM file formats - standard bitmap file format
// (includes documented extensions from legacy graphics editors)
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"


// ----------------------------------------------------------------------------
// Static functions
// ----------------------------------------------------------------------------

const int CHARSIZE = 8;

// ----------------------------------------------------------------------------
static int rounddiv(int a, int b)
{
	return ( ((a) + (b) - 1) / (b) );
}

// ----------------------------------------------------------------------------
// Load in a PBM body chunk
static void LoadPBM(IFFfile *infile, ChalkColourIndex* data, int width, int height, IffCompression compression)
{
	int ch;
	ChalkColourIndex *ptr;
	BYTE rep;
	int i, x;

	for (i = 0; i < height; ++i)
	{
		ptr = data + width * i;
		memset(ptr, 0, width * sizeof(ChalkColourIndex));

		if (compression == cmpNone) // Uncompressed format 
		{
			ptr = data + width * i;
			for (x = 0; x < width; ++x)
				*ptr++ = infile->ReadByte();
			if (width & 1)
				infile->ReadByte();
		}

		else if (compression == cmpByteRun1) // Compressed format
		{
			// Load in a scanline
			x = 0;
			while (x < width)
			{
				ch = infile->ReadByte();
				if (ch == EOF)
					break;
				else if (ch == 128)
					; // nothing
				else if (ch > 128)
				{
					rep = infile->ReadByte();
					ch -= 257;
					ptr = data + width * i + x;
					while (ch++)
					{
						*ptr++ = rep; // Set pixels to the repeated colour
						++x;
						if (x >= width)  // Ignore big bits 
							break;
					}
				}
				else
				{
					++ch;
					ptr = data + width * i + x;
					while (ch--)
					{
						// Set pixels to this colour 
						rep = infile->ReadByte();
						*ptr++ = rep; // Set pixels to the repeated colour 
						++x;
						if (x >= width)  // Ignore big bits
						{
							while (ch)
							{
								infile->ReadByte();
								--ch;
							}
						}
					}
				}
			}
		}
	}
}



// ----------------------------------------------------------------------------
// Load in a picture (ILBM)
static void LoadILBM(IFFfile *infile, ChalkColourIndex *data, int width, int height,
	IffCompression compression, int numplanes, IffMasking masking)
{
	ChalkColourIndex *ptr;
	BYTE rep;
	DWORD bit;
	int ch;
	int i, pl, x, npl;
	int thewid;

	npl = numplanes;
	thewid = 16 * rounddiv(width, 16);

	if (masking == mskHasMask)
		++npl;

	for (i = 0; i < height; ++i)
	{
		ptr = data + width * i;
		memset(ptr, 0, width * sizeof(ChalkColourIndex));

		// Load in each plane 
		bit = 1;
		for (pl = 0; pl < npl; ++pl)
		{
			// Load in a planar scanline 
			if (compression == cmpNone) // Uncompressed format 
			{
				for (x = 0; x < thewid; x += CHARSIZE)
				{
					rep = infile->ReadByte();
					ptr = data + width * i + x;
					if (rep)
					{
						if (rep & 128)
							ptr[0] |= bit;
						if (rep & 64)
							ptr[1] |= bit;
						if (rep & 32)
							ptr[2] |= bit;
						if (rep & 16)
							ptr[3] |= bit;
						if (rep & 8)
							ptr[4] |= bit;
						if (rep & 4)
							ptr[5] |= bit;
						if (rep & 2)
							ptr[6] |= bit;
						if (rep & 1)
							ptr[7] |= bit;
					}
				}
			}

			else if (compression == cmpByteRun1) // Compressed format 
			{
				x = 0;
				while (x < thewid)
				{
					ch = infile->ReadByte();
					if (ch == EOF)
						break;
					else if (ch == 128)
						; // Nothing
					else if (ch > 128)
					{
						rep = infile->ReadByte();
						ch -= 257;
						while (ch++)
						{
							if (rep)
							{
								ptr = data + width * i + x;
								if (rep & 128)
									ptr[0] |= bit;
								if (rep & 64)
									ptr[1] |= bit;
								if (rep & 32)
									ptr[2] |= bit;
								if (rep & 16)
									ptr[3] |= bit;
								if (rep & 8)
									ptr[4] |= bit;
								if (rep & 4)
									ptr[5] |= bit;
								if (rep & 2)
									ptr[6] |= bit;
								if (rep & 1)
									ptr[7] |= bit;
							}
							x += CHARSIZE;
							if (x >= thewid)
								break;
						}
					}
					else
					{
						++ch;
						while (ch--)
						{
							rep = infile->ReadByte();
							if (rep)
							{
								ptr = data + width * i + x;
								if (rep & 128)
									ptr[0] |= bit;
								if (rep & 64)
									ptr[1] |= bit;
								if (rep & 32)
									ptr[2] |= bit;
								if (rep & 16)
									ptr[3] |= bit;
								if (rep & 8)
									ptr[4] |= bit;
								if (rep & 4)
									ptr[5] |= bit;
								if (rep & 2)
									ptr[6] |= bit;
								if (rep & 1)
									ptr[7] |= bit;
							}
							x += CHARSIZE;
							if (x >= thewid)
								while (ch)
								{
									infile->ReadByte();
									ch--;
								}
						}
					}
				}
			}
			bit <<= 1;
		}
	}
}


// ----------------------------------------------------------------------------
// ChalkBitmap 
// ----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
// Read in an IFF file (extended)
HRESULT ChalkBitmap::ReadLBM(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	IffCompression compression = cmpNone;
	IffMasking masking = mskNone;
	int wid = 320;
	int hei = 200;
	int formtype = 0;
	int numplanes = 4;

	m_GridX = CHARSIZE;
	m_GridY = CHARSIZE;
	m_Character.m_OffsetX = 0;
	m_Character.m_OffsetY = 0;

	// Is the file IFF?
	long type, len, pos;

	// Parse IFF file
	type = infile.ReadLong();
	len = infile.ReadLong();
	if (type == CHUNK_FORM)
	{
		type = infile.ReadLong();
		if (type == CHUNK_ILBM)
			formtype = 1;
		else if (type == CHUNK_PBM)
			formtype = 2;
		if (formtype)
		{
			while (!infile.EndOfFile())
			{
				type = infile.ReadLong();
				len = infile.ReadLong();
				pos = infile.GetPos();

				// Standard chunk
				if (type == CHUNK_BMHD) // Read in the header
				{
					// Read in header information
					wid = infile.ReadWord();
					hei = infile.ReadWord();
					m_Character.m_OffsetX = infile.ReadWord(); // xpos 
					m_Character.m_OffsetY = infile.ReadWord(); // ypos 
					numplanes = infile.ReadByte();
					masking = (IffMasking) infile.ReadByte();
					compression = (IffCompression) infile.ReadByte();
					infile.ReadByte(); // Pad value
					infile.ReadWord(); // Transparent colour
					infile.ReadByte(); // xAspect
					infile.ReadByte(); // yAspect
					infile.ReadWord(); // Page width
					infile.ReadWord(); // Page height
					if (formtype == 1)
						wid = 16 * rounddiv(wid, 16);
					else if (compression == cmpNone)
						wid = 8 * rounddiv(wid, 8);

					// Allocate memory for bitmap
					m_Character.SetSize(wid, hei, numplanes);
				}

				// Standard chunk
				else if (type == CHUNK_CMAP) // Read in the palette
				{
					int i, r, g, b;
					m_Palette.SetSize(len / 3);
					for (i = 0; i < len / 3; ++i)
					{
						r = infile.ReadByte();
						g = infile.ReadByte();
						b = infile.ReadByte();
						m_Palette.m_Data[i] = (r << 16) | (g << 8) | (b);
					}
				}

				// Standard chunk
				else if (type == CHUNK_CRNG) // Read in a colour range
				{
					ColourRange range;
				
					infile.ReadWord();
					range.rate = infile.ReadWord();
					range.active = infile.ReadWord();
					range.first = infile.ReadByte();
					range.last = infile.ReadByte();
					m_Palette.m_ColourRanges.push_back(range);
				}


				// Standard chunk
				else if (type == CHUNK_BODY) // Read in the data 
				{
					// Load in each row 
					if (m_Character.m_Data)
					{
						if (formtype == 1) // ILBM form 
							LoadILBM(&infile, m_Character.m_Data, wid, hei, compression, numplanes, masking);
						else if (formtype == 2) // PBM form 
							LoadPBM(&infile, m_Character.m_Data, wid, hei, compression);
					}
				}

				// Extended chunk
				else if (type == CHUNK_KIDS) // Read in attfont name
				{
					int ch;
					// Ignore parallax picture name
					while ((ch = infile.ReadByte()) >= ' ')
						; // ...
					// Ignore attribute font name
					while ((ch = infile.ReadByte()) >= ' ')
						; // ...
				}

				// Extended chunk
				else if (type == CHUNK_GRID) // Read in grid info 
				{
					infile.ReadWord();
					if (infile.ReadWord() >= 2) // At least two grids specified
					{
						infile.ReadWord();
						infile.ReadWord();
						m_GridX = infile.ReadWord();
						m_GridY = infile.ReadWord();
					}
				}

				// Extended chunk
				else if (type == CHUNK_SYST) // Read in system type
				{
					int nhdr, i;
					char name[40];

					nhdr = infile.ReadWord();
					// Ignore header
					for (i = 0; i < nhdr; ++i)
						infile.ReadByte();
					// Read in system name
					i = 0;
					while (i < 38)
					{
						name[i] = infile.ReadByte();
						if (name[i])
						{
							++i;
						}
						else
						{
							++i;
							break;
						}
					}
					name[i] = 0;
//TBD				thesys = FindSystemName(name);
				}

				// Extended chunk
				else if (type == CHUNK_OFST) // Read in offset info 
				{
					if (len >= 2)
						infile.ReadWord(); // wid
					if (len >= 4)
						infile.ReadWord(); // hei
					if (len >= 6)
						m_Character.m_OffsetX = infile.ReadWord();
					if (len >= 8)
						m_Character.m_OffsetY = infile.ReadWord();
				}

				// Find next chunk (if possible)
				if (len & 1)
					++len;
				infile.SetPos(pos + len);
			}
		}
	}
	infile.Close();
	return hr;
}


// ----------------------------------------------------------------------------
// Save a picture (LBM)
HRESULT ChalkBitmap::WriteLBM(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int i, x, pl, ch;
	int width = m_Character.m_Width;
	int height = m_Character.m_Height;
	int numplanes = (m_Character.m_NumPlanes > 8) ? 24 : 8; // 24 bit or 8 bit

	outfile.Form(CHUNK_ILBM);
	outfile.Chunk(CHUNK_BMHD);
	outfile.WriteWord(width);
	outfile.WriteWord(height);
	outfile.WriteWord(m_Character.m_OffsetX);  // xpos
	outfile.WriteWord(m_Character.m_OffsetY);  // ypos
	outfile.WriteByte(numplanes); // num-planes
	outfile.WriteByte(mskNone); // masking
	outfile.WriteByte(cmpNone); // compression
	outfile.WriteByte(0); // Pad byte
	outfile.WriteWord(0); // Transparent colour
	outfile.WriteByte(1); // xAspect
	outfile.WriteByte(1); // yAspect
	outfile.WriteWord(320); // Screen width
	outfile.WriteWord(200); // Screen height
	outfile.EndChunk();

	// Write CMAP / CRNG chunks
	m_Palette.WritePalette(outfile);

	outfile.Chunk(CHUNK_OFST);
	outfile.WriteWord(m_Character.m_Width);
	outfile.WriteWord(m_Character.m_Height);
	outfile.WriteWord(m_Character.m_OffsetX);
	outfile.WriteWord(m_Character.m_OffsetY);
	outfile.EndChunk();

	outfile.Chunk(CHUNK_GRID);
	outfile.WriteWord(0);
	outfile.WriteWord(2);
	outfile.WriteWord(1);
	outfile.WriteWord(1);
	outfile.WriteWord(m_GridX);
	outfile.WriteWord(m_GridY);
	outfile.EndChunk();

	// No FIXD chunk
	// No ATTR chunk
	// No SYST chunk

	outfile.Chunk(CHUNK_BODY);
	for (i = 0; i < height; ++i)
	{
		const ChalkColourIndex* ptr = m_Character.m_Data + width * i;
		for (pl = 1; pl < (1 << numplanes); pl += pl)
		{
			for (x = 0; x + 7 < width; x += 8)
			{
				ch = (ptr[x] & pl ? 128 : 0) |
					(ptr[x+1] & pl ?  64 : 0) |
					(ptr[x+2] & pl ?  32 : 0) |
					(ptr[x+3] & pl ?  16 : 0) |
					(ptr[x+4] & pl ?   8 : 0) |
					(ptr[x+5] & pl ?   4 : 0) |
					(ptr[x+6] & pl ?   2 : 0) |
					(ptr[x+7] & pl ?   1 : 0);
				outfile.WriteByte(ch);
			}
			if (width & 7)
			{
				ch = 0;
				switch (width & 7) {
					case 7: ch |= (ptr[x+6] & pl ?   2 : 0);
					case 6: ch |= (ptr[x+5] & pl ?   4 : 0);
					case 5: ch |= (ptr[x+4] & pl ?   8 : 0);
					case 4: ch |= (ptr[x+3] & pl ?  16 : 0);
					case 3: ch |= (ptr[x+2] & pl ?  32 : 0);
					case 2: ch |= (ptr[x+1] & pl ?  64 : 0);
					case 1: ch |= (ptr[x+0] & pl ? 128 : 0); break;
				}
				outfile.WriteByte(ch);
			}
			if (((width & 15) > 0) && ((width & 15) <= 8))
				outfile.WriteByte(0xff);
		}
	}
	outfile.EndChunk();

	outfile.EndForm();
	outfile.Close();
	return hr;
}



// ----------------------------------------------------------------------------
// Save a picture (PBM)
HRESULT ChalkBitmap::WritePBM(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int i, x;
	int width = m_Character.m_Width;
	int height = m_Character.m_Height;
	outfile.Form(CHUNK_PBM);
	outfile.Chunk(CHUNK_BMHD);
	outfile.WriteWord(width);
	outfile.WriteWord(height);
	outfile.WriteWord(m_Character.m_OffsetX);  // xpos
	outfile.WriteWord(m_Character.m_OffsetY);  // ypos
	outfile.WriteByte(8); // num-planes
	outfile.WriteByte(mskNone); // masking
	outfile.WriteByte(cmpNone); // compression
	outfile.WriteByte(0); // Pad byte
	outfile.WriteWord(0); // Transparent colour
	outfile.WriteByte(1); // xAspect
	outfile.WriteByte(1); // yAspect
	outfile.WriteWord(320); // Screen width
	outfile.WriteWord(200); // Screen height
	outfile.EndChunk();

	// Write CMAP / CRNG chunks
	m_Palette.WritePalette(outfile);

	outfile.Chunk(CHUNK_OFST);
	outfile.WriteWord(m_Character.m_Width);
	outfile.WriteWord(m_Character.m_Height);
	outfile.WriteWord(m_Character.m_OffsetX);
	outfile.WriteWord(m_Character.m_OffsetY);
	outfile.EndChunk();

	outfile.Chunk(CHUNK_GRID);
	outfile.WriteWord(0);
	outfile.WriteWord(2);
	outfile.WriteWord(1);
	outfile.WriteWord(1);
	outfile.WriteWord(m_GridX);
	outfile.WriteWord(m_GridY);
	outfile.EndChunk();

	outfile.Chunk(CHUNK_BODY);
	for (i = 0; i < height; ++i)
	{
		const ChalkColourIndex* ptr = m_Character.m_Data + i * width;
		for (x = 0; x < width; ++x)
			outfile.WriteByte(*ptr++);
		if (width & 1)
			outfile.WriteByte(0xff);
	}
	outfile.EndChunk();

	outfile.EndForm();
	outfile.Close();
	return hr;
}
