// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileMAP.cpp
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports BLK and MAP file formats (from legacy graphics editors)
// using their documented file formats.
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
HRESULT ChalkMapAnim::LoadBlocks(const CFileName& blockname, const CFileName& attfontname,
	int matrixx, int matrixy)
{
	// Load block set
	// For MAP files this must be done before the main map data is loaded,
	// since the character set defines the format
	if (blockname.IsValid())
	{
		if (ChalkMapAnim::SupportsFile(blockname))
		{
			ChalkMapAnim *pBlocks = new ChalkMapAnim();
			m_pBlocks = pBlocks;
			pBlocks->ReadFile(blockname);
			m_SystemType = pBlocks->m_SystemType;
		}
		else if (ChalkFont::SupportsFile(blockname) || ChalkBitmap::SupportsFile(blockname))
		{
			ChalkFont* pBlocks = new ChalkFont();
			m_pBlocks = pBlocks;
			pBlocks->ReadFile(blockname);
			m_SystemType = pBlocks->m_SystemType;
		}
		if (m_pBlocks)
		{
			m_Grid1X = m_pBlocks->m_Grid2X;
			m_Grid1Y = m_pBlocks->m_Grid2Y;
			m_Grid2X = matrixx * m_Grid1X;
			m_Grid2Y = matrixy * m_Grid1Y;
		}
	}

	// Load attribute block set
	if (attfontname.IsValid())
	{
		if (ChalkMapAnim::SupportsFile(attfontname))
		{
			ChalkMapAnim *pBlocks = new ChalkMapAnim();
			m_pAttrBlocks = pBlocks;
			pBlocks->ReadFile(attfontname);
			m_AttrPalette = pBlocks->m_Palette;
		}
		else if (ChalkBitmap::SupportsFile(attfontname))
		{
			ChalkFont* pBlocks = new ChalkFont();
			m_pAttrBlocks = pBlocks;
			pBlocks->ReadBitmap(attfontname, &m_AttrPalette);
		}
		else if (ChalkFont::SupportsFile(attfontname))
		{
			ChalkFont* pBlocks = new ChalkFont();
			m_pAttrBlocks = pBlocks;
			pBlocks->ReadFile(attfontname);
			m_AttrPalette = m_Palette;
		}
	}
	// Use Hex font if all else fails
	if (!m_pAttrBlocks)
	{
		ChalkFont* pBlocks = new ChalkFont();
		m_pAttrBlocks = pBlocks;
		pBlocks->CreateHexFont();
	}
	// Use C64 palette if all else fails
	if (m_AttrPalette.m_NumColours == 0)
	{
		m_AttrPalette.Set(sizeof(g_PaletteC64) / sizeof(ChalkColour),
			&g_PaletteC64[0]);
	}
	return S_OK;
}


// ----------------------------------------------------------------------------
HRESULT ChalkMapAnim::ReadMAP(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int currentmap = -1;
	CFileName blockname;
	CFileName attfontname;
	CFileName auxblockname;
	int width = 0;
	int height = 0;
	int datasize = 1;
	int matrixx = 1;
	int matrixy = 1;
	m_SystemType = GUID_SYSTEM_UNKNOWN;

	// Parse IFF file
	long type, len, pos;
	type = infile.ReadLong();
	len = infile.ReadLong();
	if (type == CHUNK_FORM)
	{
		type = infile.ReadLong();
		if (type == CHUNK_TMAP)
		{
			while (!infile.EndOfFile())
			{
				type = infile.ReadLong();
				len = infile.ReadLong();
				pos = infile.GetPos();

				if (type == CHUNK_THDR)
				{
					// Read in header information
					width = infile.ReadWord();
					height = infile.ReadWord();
					matrixx = infile.ReadWord();
					matrixy = infile.ReadWord();
					datasize = infile.ReadByte();
					infile.ReadByte(); // Compression
					if (len >= 14)
					{
						infile.ReadWord(); // Number of BODY chunks
						m_Speed = infile.ReadWord() / 40.0;
					}
				}

				else if (type == CHUNK_KIDS) // Read in the palette
				{
					wstring temp1;
					int ch;

					temp1 = infile.ReadString("\r\n", "", &ch);
					blockname = CFileName::MakeAbsolute(filename, temp1);
					if (ch > 0)
					{
						temp1 = infile.ReadString("\r\n", "", &ch);
						m_KidName = CFileName::MakeAbsolute(filename, temp1);
					}
					if (ch > 0)
					{
						temp1 = infile.ReadString("\r\n", "", &ch);
						attfontname = CFileName::MakeAbsolute(filename, temp1);
					}
					if (ch > 0)
					{
						temp1 = infile.ReadString("\r\n", "", &ch);
						auxblockname = CFileName::MakeAbsolute(filename, temp1);
					}

					LoadBlocks(blockname, attfontname, matrixx, matrixy);
				}

				else if (type == CHUNK_CMAP) // Read in the palette
				{
					int i, r, g, b;
					m_Palette.SetSize(len / 3);
					for (i = 0; i < len / 3; ++i)
					{
						r = infile.ReadByte();
						g = infile.ReadByte();
						b = infile.ReadByte();
						m_Palette.m_Data[i] = (r << 16) | (g << 8) | (b);
					}
				}

				else if (type == CHUNK_CRNG) // Read in a colour range
				{
					ColourRange range;
				
					infile.ReadWord();
					range.rate = infile.ReadWord();
					range.active = infile.ReadWord();
					range.first = infile.ReadByte();
					range.last = infile.ReadByte();
					m_Palette.m_ColourRanges.push_back(range);
				}

				else if (type == CHUNK_GRID) // Read in grid info 
				{
					infile.ReadWord();
					if (infile.ReadWord() >= 2) // At least two grids specified
					{
						infile.ReadWord(); // not m_Grid1X
						infile.ReadWord(); // not m_Grid1Y
						infile.ReadWord(); // not m_Grid2X
						infile.ReadWord(); // not m_Grid2Y
					}
				}

				else if (type == CHUNK_OFST) // Read in offset info 
				{
					// ...
					// Width
					// Height
					// OffsetX
					// OffsetY
				}

				else if (type == CHUNK_BODY) // Read in the data
				{
					int i, j;

					// Create a map
					++currentmap;
					m_Data.resize(currentmap + 1);
					ChalkMap& map = m_Data[currentmap];
					map.SetSize(width, height);
					ChalkSystem* systemobject = static_cast<ChalkSystem *>(g_ChalkObjectMap.GetObject(m_SystemType));

					// Add the blocks
					if (datasize == 2)
					{
						for (i = 0; i < height; ++i)
						{
							ChalkMapBlock* pBlocks = &map.m_Data[i * width];
							for (j = 0; j < width; ++j)
								(pBlocks++)->Set(infile.ReadWord(), systemobject);
						}
					}
					else
					{
						for (i = 0; i < height; ++i)
						{
							ChalkMapBlock* pBlocks = &map.m_Data[i * width];
							for (j = 0; j < width; ++j)
								(pBlocks++)->Set(infile.ReadByte() & 0xff, systemobject);
						}
					}
				}

				else if (type == CHUNK_COLR) // Read in the data
				{
					if (currentmap >= 0)
					{
						ChalkMap& map = m_Data[currentmap];
						int i, j;

						if (datasize == 2) // Extra data
						{
							for (i = 0; i < height; ++i)
							{
								ChalkMapBlock* pBlocks = &map.m_Data[i * width];
								for (j = 0; j < width; ++j)
									(pBlocks++)->m_BlockIndex |= (DWORD(infile.ReadByte())) << 16;
							}
						}
						else
						{
							for (i = 0; i < height; ++i)
							{
								ChalkMapBlock* pBlocks = &map.m_Data[i * width];
								for (j = 0; j < width; ++j)
								{
									pBlocks->m_BlockIndex  &= 0xffff00ff;
									(pBlocks++)->m_BlockIndex |= (DWORD(infile.ReadByte())) << 8;
								}
							}
						}
					}
				}

				else if (type == CHUNK_ATTR) // Read in the data
				{
					if (currentmap >= 0)
					{
						ChalkMap& map = m_Data[currentmap];
						int i, j;

						for (i = 0; i < height; ++i)
						{
							ChalkMapBlock* pAttributes = &map.m_Data[i * width];
							for (j = 0; j < width; ++j)
								(pAttributes++)->m_Attribute = infile.ReadByte();
						}
					}
				}

				// FIXD Chunk is irrelevant
				// PRLX Chunk is irrelevant

				// Find next chunk (if possible)
				if (len & 1)
					++len;
				infile.SetPos(pos + len);
			}
		}
	}
	infile.Close();

	return hr;
}


// ----------------------------------------------------------------------------
HRESULT ChalkMapAnim::WriteMAP(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;
	int i, x, y;

	// Find maximum size
	int nummaps = m_Data.size();
	int maxwidth = 0;
	int maxheight = 0;
	for (i = 0; i < m_Data.size(); ++i)
	{
		const ChalkMap& map = m_Data[i];
		maxwidth = max(maxwidth, map.m_Width);
		maxheight = max(maxheight, map.m_Height);
	}
	ChalkSystem* systemobject = static_cast<ChalkSystem *>(g_ChalkObjectMap.GetObject(m_SystemType));

	// Write out header of MAP file 
	outfile.Form(CHUNK_TMAP);
	outfile.Chunk(CHUNK_THDR);
	outfile.WriteWord(maxwidth);				// Wwidth 
	outfile.WriteWord(maxheight);				// Height 
	outfile.WriteWord(m_Grid2X / m_Grid1X);     // Block width 
	outfile.WriteWord(m_Grid2Y / m_Grid1Y);     // Block height 
	outfile.WriteByte(2);						// Bytes per datum 
	outfile.WriteByte(cmpNone);					// Compression algorithm 
	if (nummaps > 1)
	{
		outfile.WriteWord(nummaps);
		outfile.WriteWord(int(m_Speed * 40.0));
	}
	outfile.EndChunk();

	// Write CMAP / CRNG chunks
	m_Palette.WritePalette(outfile);

	// Write out KIDS info
	{
		wstring S;

		outfile.Chunk(CHUNK_KIDS);
		if (m_pBlocks && m_pBlocks->m_FileName.IsValid())
		{
			S = CFileName::MakeRelative(filename, m_pBlocks->m_FileName);
			outfile.WriteString(S);
		}
		outfile.WriteByte('\r');
		if (m_KidName.IsValid())
		{
			S = CFileName::MakeRelative(filename, m_KidName);
			outfile.WriteString(S);
		}
		outfile.WriteByte('\r');
		if (m_pAttrBlocks && m_pAttrBlocks->m_FileName.IsValid())
		{
			S = CFileName::MakeRelative(filename, m_pAttrBlocks->m_FileName);
			outfile.WriteString(S);
		}
		outfile.WriteByte('\r');
			// auxblocksname
		outfile.WriteByte(0);
		outfile.EndChunk();
	}

	// Write out GRID info
	outfile.Chunk(CHUNK_GRID);
	outfile.WriteWord(0);
	outfile.WriteWord(2);
	outfile.WriteWord(m_Grid1X);
	outfile.WriteWord(m_Grid1Y);
	outfile.WriteWord(m_Grid2X);
	outfile.WriteWord(m_Grid2Y);
	outfile.EndChunk();

	// No PRLX chunk
	// No FIXD chunk

	// Write out each map
	for (i = 0; i < nummaps; ++i)
	{
		const ChalkMap& map = m_Data[i];
		const ChalkMapBlock* pBlock;

		// Write out offsets 
		outfile.Chunk(CHUNK_OFST);
		outfile.WriteWord(m_Grid1X * map.m_Width); // Width in pixels
		outfile.WriteWord(m_Grid1X * map.m_Height); // Height in pixels
		outfile.WriteWord(0); // X offset in pixels
		outfile.WriteWord(0); // Y offset in pixels
		outfile.EndChunk();

		// Write out info of file 
		outfile.Chunk(CHUNK_BODY);
		for (y = 0; y < maxheight; ++y)
		{
			pBlock = &map.m_Data[y * map.m_Width];
			for (x = 0; x < maxwidth; ++x)
				outfile.WriteWord((pBlock++)->Get(systemobject) & 0xffff);
		}
		outfile.EndChunk();

		outfile.Chunk(CHUNK_ATTR);
		for (y = 0; y < maxheight; ++y)
		{
			pBlock = &map.m_Data[y * map.m_Width];
			for (x = 0; x < maxwidth; ++x)
				outfile.WriteByte((pBlock++)->m_Attribute);
		}
		outfile.EndChunk();

		outfile.Chunk(CHUNK_COLR);
		for (y = 0; y < maxheight; ++y)
		{
			pBlock = &map.m_Data[y * map.m_Width];
			for (x = 0; x < maxwidth; ++x)
				outfile.WriteByte(((pBlock++)->Get(systemobject) >> 16) & 0xff);
		}
		outfile.EndChunk();
	}

	// Finish off file
	outfile.EndForm();
	outfile.Close();
	return hr;
}
