// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FilePAL.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports LBM file format for palettes - standard bitmap file format
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"


// ----------------------------------------------------------------------------
HRESULT ChalkPalette::ReadLBM(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (SUCCEEDED(hr))
	{
		// Is the file IFF?
		long type, len, pos;

		// Parse IFF file
		type = infile.ReadLong();
		len = infile.ReadLong();
		if (type == CHUNK_FORM)
		{
			type = infile.ReadLong();
			SetSize(0);

			while (!infile.EndOfFile())
			{
				type = infile.ReadLong();
				len = infile.ReadLong();
				pos = infile.GetPos();

				if (type == CHUNK_CMAP) // Read in the palette
				{
					int i, r, g, b;
					SetSize(len / 3);
					for (i = 0; i < len / 3; ++i)
					{
						r = infile.ReadByte();
						g = infile.ReadByte();
						b = infile.ReadByte();
						m_Data[i] = (r << 16) | (g << 8) | (b);
					}
				}

				else if (type == CHUNK_CRNG) // Read in a colour range
				{
					ColourRange range;
				
					infile.ReadWord();
					range.rate = infile.ReadWord();
					range.active = infile.ReadWord();
					range.first = infile.ReadByte();
					range.last = infile.ReadByte();
					m_ColourRanges.push_back(range);
				}

				// Find next chunk (if possible)
				if (len & 1)
					++len;
				infile.SetPos(pos + len);
			}
		}
		infile.Close();
	}
	return hr;
}

// ----------------------------------------------------------------------------
HRESULT ChalkPalette::WritePalette(IFFfile& outfile) const
{
	if (m_NumColours > 0)
	{
		outfile.Chunk(CHUNK_CMAP);
		int i, r, g, b;
		for (i = 0; i < m_NumColours; ++i)
		{
			r = (m_Data[i] >> 16) & 0xff;
			g = (m_Data[i] >> 8) & 0xff;
			b = (m_Data[i]) & 0xff;
			outfile.WriteByte(r);
			outfile.WriteByte(g);
			outfile.WriteByte(b);
		}
		outfile.EndChunk();
	}

	int i;
	for (i = 0; i < m_ColourRanges.size(); ++i)
	{
		const ColourRange& range = m_ColourRanges[i];
		outfile.Chunk(CHUNK_CRNG);
		outfile.WriteWord(range.rate);
		outfile.WriteWord(range.active);
		outfile.WriteByte(range.first);
		outfile.WriteByte(range.last);
		outfile.EndChunk();
	}
	return S_OK;
}


// ----------------------------------------------------------------------------
HRESULT ChalkPalette::WriteLBM(const CFileName& filename) const
{
	IFFfile outfile;

	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	outfile.Form(CHUNK_ILBM);
	WritePalette(outfile);
	outfile.EndForm();
	outfile.Close();
	return hr;
}
