// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FilePCX.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports PCX file format - standard bitmap file format
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

const int CHARSIZE = 8;

// ----------------------------------------------------------------------------
typedef struct {
	BYTE manufacturer;
	BYTE version;
	BYTE encoding;
	BYTE bitsperpixel;
	WORD xmin, ymin, xmax, ymax;
	WORD hres, vres;
	BYTE colormap[16][3];
	BYTE reserved;
	BYTE nplanes;
	WORD bytesperline;
} PCPAINTBRUSH_HEADER;


// ----------------------------------------------------------------------------
// Read in a PCX file
HRESULT ChalkBitmap::ReadPCX(const CFileName& filename)
{
	IFFfile infile; // Note that this auto-closes on destruction
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	PCPAINTBRUSH_HEADER header;
	int i, j, ch, val;

	m_GridX = CHARSIZE;
	m_GridY = CHARSIZE;
	m_Character.m_OffsetX = 0;
	m_Character.m_OffsetY = 0;

	// Read in header
	infile.ReadBytes(reinterpret_cast<BYTE *>(&header), sizeof(header));
	if (header.manufacturer != 0x0A)
		return E_FAIL; // Bad PCX manufacturer
	if (header.encoding != 0x01)
		return E_FAIL; // Bad PCX encoding
	if (header.bitsperpixel != 1)
	{
		if (header.nplanes != 1)
			return E_FAIL; // Cannot load mixed packed PCX files
		else if (header.bitsperpixel != 8)
			return E_FAIL; // Cannot load non 8 bit packed PCX files
	}

	// Read in the palette
	if (header.version == 2) // 16 colour palette
	{
		int i, r, g, b;
		m_Palette.SetSize(16);
		for (i = 0; i < 16; ++i)
		{
			r = header.colormap[i][0];
			g = header.colormap[i][1];
			b = header.colormap[i][2];
			m_Palette.m_Data[i] = (r << 16) | (g << 8) | (b);
		}
	}
	else if (header.version == 5) // 256 colour palette
	{
		infile.SetPos(infile.GetLength() - 769L); // From end of file
		if (infile.ReadByte() == 0x0C) // Extended palette
		{
			int i, r, g, b;
			m_Palette.SetSize(256);
			for (i = 0; i < 256; ++i)
			{
				r = infile.ReadByte();
				g = infile.ReadByte();
				b = infile.ReadByte();
				m_Palette.m_Data[i] = (r << 16) | (g << 8) | (b);
			}
		}
	}

	// Read in the picture data
	int width = header.xmax - header.xmin + 1;
	int height = header.ymax - header.ymin + 1;
	m_Character.SetSize(width, height, 8);
	m_Character.m_OffsetX = header.xmin;
	m_Character.m_OffsetY = header.ymin;

	// Packed byte load
	infile.SetPos(128L);
	ch = 0;
	if (header.bitsperpixel == 8)
	{
		for (i = 0; i < height; ++i)
		{
			ChalkColourIndex *ptr = m_Character.m_Data + i * width;
			j = 0;
			while (j < width)
			{
				ch = infile.ReadByte();
				if (ch == EOF)
					break;
				if ((ch & 0xC0) == 0xC0)
				{
					ch = (ch & 0x3f);
					val = infile.ReadByte();
					while (ch--)
					{
						*ptr++ = val;
						++j;
					}
					ch = 0; // Avoid quitting
				}
				else
				{
					*ptr++ = ch;
					++j;
				}
			}
			if (ch == EOF)
				break;
		}
	}
	infile.Close();
	return hr;
}



// ----------------------------------------------------------------------------
// Save a picture (PCX)
HRESULT ChalkBitmap::WritePCX(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	PCPAINTBRUSH_HEADER header;
	ChalkColourIndex startbyte;
	int count;
	int x, y;
	int width = m_Character.m_Width;
	int height = m_Character.m_Height;
	header.manufacturer = 10;
	header.version = 5;
	header.encoding = 1;
	header.bitsperpixel = 8;
	header.xmin = 0;
	header.xmax = width - 1;
	header.ymin = 0;
	header.ymax = height - 1;
	header.hres = 320;
	header.vres = 200;
	for (x = 0; x < 16; ++x)
	{
		header.colormap[x][0] = BYTE((m_Palette.m_Data[x] >> 16) & 0xff);
		header.colormap[x][1] = BYTE((m_Palette.m_Data[x] >> 8) & 0xff);
		header.colormap[x][2] = BYTE((m_Palette.m_Data[x]) & 0xff);
	}
	header.reserved = 0;
	header.nplanes = 1;
	header.bytesperline = width;

	// Write out the header (only works on little-endian systems)
	outfile.WriteBytes(reinterpret_cast<BYTE *>(&header), sizeof(PCPAINTBRUSH_HEADER));
	for (x = sizeof(PCPAINTBRUSH_HEADER); x < 128; ++x)
		outfile.WriteByte(0);

	// Write out the pixel data
	for (y = 0; y < height; ++y)
	{
		const ChalkColourIndex* ptr = m_Character.m_Data + y * width;
		x = 0;
		while (x < width)
		{
			// Check for a repeating byte value
			startbyte = ptr[x];
			++x;
			count = 1;
			while ((x < width) && (count < 63) && (ptr[x] == startbyte))
			{
				++count;
				++x;
			}
			// If we can pack the sequence, or if we have to add a
		 	// byte before it because the top 2 bits of the value
		 	// are 1's, write a packed sequence of 2 bytes.
		 	// Otherwise, just write the byte value.
			if ((count > 1) || ((startbyte & 0xc0) == 0xc0))
				outfile.WriteByte(0xc0 | count);
			outfile.WriteByte(startbyte);
		}
	}

	// Write out RGB Palette
	{
		outfile.WriteByte(0x0c);
		int i, r, g, b;
		for (i = 0; i < m_Palette.m_NumColours; ++i)
		{
			r = (m_Palette.m_Data[i] >> 16) & 0xff;
			g = (m_Palette.m_Data[i] >> 8) & 0xff;
			b = (m_Palette.m_Data[i]) & 0xff;
			outfile.WriteByte(r);
			outfile.WriteByte(g);
			outfile.WriteByte(b);
		}
		for (x = m_Palette.m_NumColours; x < 256; ++x)
		{
			outfile.WriteByte(0);
			outfile.WriteByte(0);
			outfile.WriteByte(0);
		}
	}

	outfile.Close();
	return hr;
}
