// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FilePXP.cpp
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports PXP file format (from legacy graphics editors)
// using their documented file formats.
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"


// ----------------------------------------------------------------------------
// Load in a pxp file
HRESULT ChalkFont::ReadPXP(const CFileName& filename, ChalkPalette* pPalette, bool merge)
{
	int fontoffset = 0;
	if (merge)
		fontoffset = m_Data.size();

	m_FileName = filename; // Remember my name
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (SUCCEEDED(hr))
	{
		// Is the file IFF?
		long type, len, pos;

		// Parse IFF file
		type = infile.ReadLong();
		len = infile.ReadLong();
		if (type == CHUNK_FORM)
		{
			type = infile.ReadLong();
			if (type == CHUNK_PXXP)
			{
				while (!infile.EndOfFile())
				{
					type = infile.ReadLong();
					len = infile.ReadLong();
					pos = infile.GetPos();

					if (type == CHUNK_PHDR)
					{
						// Read in header information
						if (infile.ReadByte() != 8)
							hr = E_FAIL; // "*** Cannot read PXP files that are not byte per pixel ***\n";
						if (infile.ReadByte() != 1)
							hr = E_FAIL; // "*** Cannot read PXP files with multiple planes ***\n";
						if (infile.ReadByte())
							hr = E_FAIL; //  "*** Cannot read PXP files in column order ***\n";
						if (infile.ReadByte())
							hr = E_FAIL; //  "*** Cannot read PXP files with compression ***\n";
						if (infile.ReadByte() != 4)
							hr = E_FAIL; //  "*** Cannot read PXP files with 3d ***\n";
						infile.ReadByte(); // Format of optional record data (0=byte,1=65816,2=68000)
						infile.ReadWord(); // Default width of sprites
						infile.ReadWord(); // Default height of sprites
						infile.ReadWord(); // Default depth of sprites
						infile.ReadWord(); // Default X offset of sprites
						infile.ReadWord(); // Default Y offset of sprites
						infile.ReadWord(); // Default Z offset of sprites
						if (infile.ReadByte() != 1)
							hr = E_FAIL; //  "*** Cannot read PXP files with word alignment ***\n";
						if (len > 20)
						{
							infile.ReadByte(); // Pad byte
							int animspeed = infile.ReadWord(); // Animation speed
						}
					}

					else if (type == CHUNK_CMAP) // Read in the palette
					{
						if (pPalette)
						{
							int i, r, g, b;
							pPalette->SetSize(len / 3);
							for (i = 0; i < len / 3; ++i)
							{
								r = infile.ReadByte();
								g = infile.ReadByte();
								b = infile.ReadByte();
								pPalette->m_Data[i] = (r << 16) | (g << 8) | (b);
							}
						}
					}

					else if (type == CHUNK_BODY) // Read in the data
					{
						int numframes = infile.ReadWord();
						if (infile.ReadWord() != 8)
							hr = E_FAIL; // "*** Cannot read PXP file in this format ***\n";
						int hdrsize = infile.ReadWord();
						while (hdrsize--)
							infile.ReadByte();
						int i, x, y;

						m_Data.resize(fontoffset + numframes);
						for (i = 0; i < numframes; ++i)
						{
							ChalkCharacter& character = m_Data[fontoffset + i];
							int wid, hei;
							wid = infile.ReadWord();
							hei = infile.ReadWord();
							character.m_OffsetX = infile.ReadWord();
							character.m_OffsetY = infile.ReadWord();
							character.SetSize(wid, hei, 8);
							ChalkColourIndex *pDst = character.m_Data;
							for (y = 0; y < hei; ++y)
								for (x = 0; x < wid; ++x)
									*pDst++ = infile.ReadByte();
						}
					}

					// Find next chunk (if possible)
					if (len & 1)
						++len;
					infile.SetPos(pos + len);
				}
			}
		}
		infile.Close();
	}

	return S_OK;
}

