// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     FileTGA.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Supports TGA (Targa) file format - standard bitmap file format
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"


// ----------------------------------------------------------------------------
//	Types of TARGA files
//	--------------------
//	0 = no data
//	1 = raw map data
//	2 = raw RGB values [only format handled]
//	3 = raw mon
//	9 = run length encoded map data
//	10= run length encoded RGB values
//	11= run length encoded mon



// ----------------------------------------------------------------------------
// Load a picture (TGA)
HRESULT ChalkBitmap::ReadTGA(const CFileName& filename)
{
	IFFfile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return hr;

	int wid, hei, flags, esize;
	int pix;
	int x, y, ystart, yend, ydir;

	// Read in header
	infile.ReadByte(); // Comment field length (0)
	infile.ReadByte(); // map type (only handles  0 = none)
	infile.ReadByte(); // file type (only handles 2 = raw 15 bit RGB)
	infile.ReadIntelWord(); // map origin
	infile.ReadIntelWord(); // map len
	infile.ReadByte(); // map size
	infile.ReadIntelWord(); // x origin
	infile.ReadIntelWord(); // y origin
	wid = infile.ReadIntelWord();  // width
	hei = infile.ReadIntelWord();  // height
	esize = infile.ReadByte(); // esize (16/24/32)
	flags = infile.ReadByte(); // flags (0 or 32)

	// Allocate new memory map
	m_Character.SetSize(wid, hei, 24); // Store internally as 24 bit

	// Load in body
	if (flags & 0x20) // Vertical flip
	{
		ystart = 0;
		yend = hei;
		ydir = 1;
	}
	else
	{
		ystart = hei - 1;
		yend = -1;
		ydir = -1;
	}
	ChalkColourIndex *pRow;
	if (esize == 32)
	{
		for (y = ystart; y != yend; y += ydir)
		{
			pRow = m_Character.m_Data + wid * y;
			for (x = 0; x < wid; ++x)
				*pRow++ = infile.ReadIntelLong() & 0x00ffffff;
		}
	}
	else if (esize == 24)
	{
		for (y = ystart; y != yend; y += ydir)
		{
			pRow = m_Character.m_Data + wid * y;
			for (x = 0; x < wid; ++x)
			{
				pix = infile.ReadByte();
				pix |= infile.ReadByte() << 8;
				pix |= infile.ReadByte() << 16;
				*pRow++ = pix;
			}
		}
	}
	else if (esize == 16)
	{
		for (y = ystart; y != yend; y += ydir)
		{
			pRow = m_Character.m_Data + wid * y;
			for (x = 0; x < wid; ++x)
			{
				ChalkColourIndex value = infile.ReadIntelWord();
				pix =  (value & 0x003f) << 3;
				pix |= (value & 0x07c0) << 6;
				pix |= (value & 0x3800) << 9;
				*pRow++ = pix;
			}
		}
	}
	infile.Close();
	return hr;
}


// ----------------------------------------------------------------------------
// Save a bitmap (TGA format)
HRESULT ChalkBitmap::WriteTGA(const CFileName& filename) const
{
	IFFfile outfile;
	HRESULT hr = outfile.OpenWrite(filename);
	if (FAILED(hr))
		return hr;

	int x, y;
	ChalkColourIndex val;
	int width = m_Character.m_Width;
	int height = m_Character.m_Height;

	// Write out header
	outfile.WriteByte(0); // Comment field length (0)
	outfile.WriteByte(0); // map type
	outfile.WriteByte(2); // file type
	outfile.WriteIntelWord(0); // map origin 
	outfile.WriteIntelWord(0); // map len 
	outfile.WriteByte(0); // map size
	outfile.WriteIntelWord(0); // x origin
	outfile.WriteIntelWord(0); // y origin
	outfile.WriteIntelWord(width); // width
	outfile.WriteIntelWord(height); // height
	outfile.WriteByte(24);     // esize Number of bits per pixel (24)
	outfile.WriteByte(0);      // flags (0 = normal, $20 = vflip)

	// Save out body - 24 bit true colour
	for (y = height - 1; y >= 0; --y)
	{
		const ChalkColourIndex *pRow = m_Character.m_Data + width * y;
		for (x = 0; x < width; ++x)
		{
			val = *pRow++;
			outfile.WriteByte(val);
			outfile.WriteByte(val >> 8);
			outfile.WriteByte(val >> 16);
		}
	}
	outfile.Close();
	return hr;
}
