// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     MemoryBuffer.h
// CREATED:   3 Oct 2001 by Carl Muller
// MODIFIED: 16 Oct 2001 by Carl Muller
//
// This file defines a class that supports an in-memory buffer
// that can be read from/written to like a stream file.
// It is circular to that it has a fixed size.
// It is used for supporting Undo operations
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
class MemoryBuffer : public BinaryStream
{
public :
	MemoryBuffer();
	virtual ~MemoryBuffer();

	// Derived classes should override this
	virtual void MakeRoom()
	{	ReadByte(); }

	// Override SimpleStream functions
	bool CanRead()
	{
		return m_BufferSize != 0;
	}
	bool CanWrite()
	{
		return m_BufferSize != 0;
	}
	void WriteByte(int value)
	{
		int next = m_BufferHead + 1;
		if (next == m_BufferSize)
			next = 0;
		if (next == m_BufferTail)
		{
			// Make some room for the new bytes
			int index = m_BufferIndex;
			m_BufferIndex = m_BufferTail;
			MakeRoom();
			m_BufferTail = m_BufferIndex;
			m_BufferIndex = index;
		}

		// Now write the value
		m_pBuffer[m_BufferHead++] = BYTE(value);
		if (m_BufferHead == m_BufferSize)
			m_BufferHead = 0;
	}
	int ReadByte()
	{
		if (m_BufferIndex == m_BufferHead)
			return EOF;
		else
		{
			int NextChar = m_pBuffer[m_BufferIndex++];
			if (m_BufferIndex >= m_BufferSize)
				m_BufferIndex = 0;
			return NextChar;
		}
	}

	// General functions
	virtual HRESULT Open(int newbufsize = 4096);
	virtual HRESULT Close();
	HRESULT SetReadPos(long pos); // Relative to the end of the file
	HRESULT SetWritePos(long pos); // Relative to the end of the file

	// Reading functions
	bool EndOfFile()
	{	return (m_BufferIndex == m_BufferTail);	}

private :
	// Helper functions
	MemoryBuffer(const MemoryBuffer &s);  // dummy copy constructor
	const MemoryBuffer& operator= (const MemoryBuffer&s);  // dummy assignment operator

private :
	// Variables
	int m_BufferSize;  // How many bytes does the buffer hold
	BYTE *m_pBuffer;   // Raw storage for the buffer
	int m_BufferHead;  // Ahead of newest readable byte (Write pointer)
	int m_BufferTail;  // Oldest readable byte
	int m_BufferIndex; // Current readable byte
};
