// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkDoc.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED:  5 Apr 2001 by Carl Muller
//
// Implementation of the CChalkDoc class
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalk.h"
#include "chalkDoc.h"
#include "chalkView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// ----------------------------------------------------------------------------
// CChalkDoc
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
IMPLEMENT_DYNCREATE(CChalkDoc, CDocument)

BEGIN_MESSAGE_MAP(CChalkDoc, CDocument)
	//{{AFX_MSG_MAP(CChalkDoc)
	ON_UPDATE_COMMAND_UI(ID_STATUS_LAYER, OnUpdateStatusLayer)
	ON_UPDATE_COMMAND_UI(ID_STATUS_POSITION, OnUpdateStatusPosition)
	ON_UPDATE_COMMAND_UI(ID_STATUS_TYPE, OnUpdateStatusType)
	ON_UPDATE_COMMAND_UI(ID_STATUS_FRAME, OnUpdateStatusFrame)
	ON_UPDATE_COMMAND_UI(ID_STATUS_ZOOM, OnUpdateStatusZoom)
	ON_COMMAND(ID_ANIM_NEXT, OnAnimNext)
	ON_COMMAND(ID_ANIM_REWIND, OnAnimRewind)
	ON_COMMAND(ID_ANIM_PREVIOUS, OnAnimPrevious)
	ON_COMMAND(ID_ANIM_FFWD, OnAnimFfwd)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

// ----------------------------------------------------------------------------
// CChalkDoc construction/destruction
CChalkDoc::CChalkDoc()
{
	// For a new document, use the default (test pattern) layer
	ChalkLayer* pLayer;
	pLayer = new ChalkLayer(*this);
	m_Layers.insert(m_Layers.begin(), pLayer);
	m_DocSize = CSize(16, 16);

	// Set the status values to non-empty, to avoid memory leak during pCmdUI->SetText()
	m_StatusFrame = FORMAT_SPACE;
	m_StatusLayer = FORMAT_SPACE;
	m_StatusPosition = FORMAT_SPACE;
	m_StatusZoom = FORMAT_SPACE;
	m_StatusType = FORMAT_SPACE;
}

// ----------------------------------------------------------------------------
CChalkDoc::~CChalkDoc()
{
	Clear();
}

// ----------------------------------------------------------------------------
void CChalkDoc::Clear()
{
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
			delete layer;
	}
	m_Layers.clear();
}

// ----------------------------------------------------------------------------
BOOL CChalkDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	return TRUE;
}

// ----------------------------------------------------------------------------
// CChalkDoc serialization
void CChalkDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// TODO: add storing code here
	}
	else
	{
		// TODO: add loading code here
	}
}

// ----------------------------------------------------------------------------
// CChalkDoc diagnostics
#ifdef _DEBUG
void CChalkDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CChalkDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

// ----------------------------------------------------------------------------
wstring ReadXMLRoot(const CFileName& filename)
{
	// Open up a buffered file
	BufferedFile infile;
	HRESULT hr = infile.OpenRead(filename);
	if (FAILED(hr))
		return L"";

	// Read in the XML header
	infile.ReadString(">", "", NULL);
	infile.ReadString("<", "", NULL);
	wstring root = infile.ReadString(">", "", NULL);

	// Finish off
	infile.Close();
	return root;
}

// ----------------------------------------------------------------------------
void CChalkDoc::AddLayer(const CFileName& filename)
{
	CFileName kidname;

	if (filename.HasExt(FILEEXTENSION_XML))
	{
		// Work out what type of file it is
		wstring root = MakeUpper(ReadXMLRoot(filename));

		if (root == MakeUpper(L"Palette"))
		{
			// Add a palette layer - XML format
			ChalkLayerPaletteBars* pLayer;
			pLayer = new ChalkLayerPaletteBars(*this);
			m_Layers.insert(m_Layers.begin(), pLayer);
			pLayer->m_Data.ReadFile(filename);
		}
		else if (root == MakeUpper(L"Animation"))
		{
			// Add an animation layer - XML format
			ChalkLayerAnimation * pLayer;
			pLayer = new ChalkLayerAnimation(*this);
			m_Layers.insert(m_Layers.begin(), pLayer);
			pLayer->m_Data.ReadFile(filename);
			kidname = pLayer->m_Data.m_KidName;
		}
		else if (root == MakeUpper(L"MapAnimation"))
		{
			// Add a map layer - XML format
			ChalkLayerMap * pLayer;
			pLayer = new ChalkLayerMap(*this);
			m_Layers.insert(m_Layers.begin(), pLayer);
			pLayer->m_Data.ReadFile(filename);
		}
		else if (root == MakeUpper(L"Bitmap"))
		{
			// Add a bitmap layer - XML format
			ChalkLayerBitmap* pLayer;
			pLayer = new ChalkLayerBitmap(*this);
			m_Layers.insert(m_Layers.begin(), pLayer);
			pLayer->m_Data.ReadFile(filename);
		}
	}

	else if (ChalkPalette::SupportsFile(filename))
	{
		// Add a palette layer - COP format
		ChalkLayerPaletteBars* pLayer;
		pLayer = new ChalkLayerPaletteBars(*this);
		m_Layers.insert(m_Layers.begin(), pLayer);
		pLayer->m_Data.ReadFile(filename);
	}
	else if (ChalkAnimation::SupportsFile(filename))
	{
		// Add an animation layer - BOX/FRK format
		ChalkLayerAnimation * pLayer;
		pLayer = new ChalkLayerAnimation(*this);
		m_Layers.insert(m_Layers.begin(), pLayer);
		pLayer->m_Data.ReadFile(filename);
		kidname = pLayer->m_Data.m_KidName;
	}
	else if (ChalkMapAnim::SupportsFile(filename))
	{
		// Add a map layer - MAP format
		ChalkLayerMap * pLayer;
		pLayer = new ChalkLayerMap(*this);
		m_Layers.insert(m_Layers.begin(), pLayer);
		pLayer->m_Data.ReadFile(filename);
	}
	else if (ChalkBitmap::SupportsFile(filename) || ChalkFont::SupportsFile(filename))
	{
		// Add a bitmap layer - LBM/PBM/BMP/TGA/PCX format
		ChalkLayerBitmap* pLayer;
		pLayer = new ChalkLayerBitmap(*this);
		m_Layers.insert(m_Layers.begin(), pLayer);
		pLayer->m_Data.ReadFile(filename);
	}
	if (kidname.IsValid())
		AddLayer(kidname);
}

// ----------------------------------------------------------------------------
// Calculate the size of all the layers
void CChalkDoc::CalcSize()
{
	CSize docsize(0, 0);
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			CSize sz = layer->GetSize();
			docsize.cx = max(docsize.cx, sz.cx);
			docsize.cy = max(docsize.cy, sz.cy);
		}
	}
	m_DocSize = docsize;
}

// ----------------------------------------------------------------------------
// Load a document
BOOL CChalkDoc::OnOpenDocument(LPCTSTR lpszPathName) 
{
	BeginWaitCursor();
	// Remove existing layers
	SetModifiedFlag();
	Clear();

	// Actually load the file
	try {
		AddLayer(CFileName(lpszPathName));
		// Add background layer (in case nothing else is visible)
		ChalkLayer* pLayer;
		pLayer = new ChalkLayer(*this);
		m_Layers.insert(m_Layers.begin(), pLayer);
		pLayer->m_Visible = layer_hidden;

		// Set the topmost layer to be active
		ChalkLayer* top = m_Layers.back();
		if (top && (top->m_Visible == layer_visible))
			SelectLayer(top);
	}
	catch (CFileException* e) {
		EndWaitCursor();
		ReportSaveLoadException(lpszPathName, e, FALSE, IDS_ERROR_LOAD);
		e->Delete();
		return FALSE;
	}
	CalcSize();
	SetModifiedFlag(FALSE);
	EndWaitCursor();
	return TRUE;
}

// ----------------------------------------------------------------------------
// Save a document
BOOL CChalkDoc::OnSaveDocument(LPCTSTR lpszPathName) 
{
	BeginWaitCursor();
	CFileName savename(lpszPathName);

	try
	{
		vector<ChalkLayer*>::iterator i;
		for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
		{
			ChalkLayer* layer = *i;
			if (layer && (layer->m_Visible == layer_active) && (layer->m_pBaseData))
			{
				CFileName layername(layer->m_pBaseData->m_FileName);
				if (layername.IsValid())
				{
					layer->m_pBaseData->WriteFile(savename.Get());
					break; // Only save one file with that filename!
				}
			}
		}
	}
	catch (CFileException* e) {
		EndWaitCursor();
		ReportSaveLoadException(lpszPathName, e, FALSE, IDS_ERROR_LOAD);
		e->Delete();
		return FALSE;
	}
	SetModifiedFlag(FALSE);
	EndWaitCursor();
	return TRUE;
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnUpdateStatusFrame(CCmdUI* pCmdUI) 
{
	pCmdUI->SetText(m_StatusFrame.c_str());
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnUpdateStatusLayer(CCmdUI* pCmdUI) 
{
	pCmdUI->SetText(m_StatusLayer.c_str());
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnUpdateStatusPosition(CCmdUI* pCmdUI) 
{
	pCmdUI->SetText(m_StatusPosition.c_str());
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnUpdateStatusZoom(CCmdUI* pCmdUI) 
{
	pCmdUI->SetText(m_StatusZoom.c_str());
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnUpdateStatusType(CCmdUI* pCmdUI) 
{
	pCmdUI->SetText(m_StatusType.c_str());
}

// ----------------------------------------------------------------------------
void CChalkDoc::SelectLayer(ChalkLayer *which)
{
	int iCount;
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(), iCount = 0; i != m_Layers.end(); ++i, ++iCount)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			if (layer == which)
			{
				tstringstream s;
				s << "Layer " << iCount << ": " << layer->m_TypeName;
				m_StatusLayer = s.str();
				layer->m_Visible = layer_active;
			}
			else if (layer->m_Visible == layer_active)
				layer->m_Visible = layer_visible;
		}
		++iCount;
	}
}

// ----------------------------------------------------------------------------
// All views are invalid
void CChalkDoc::InvalidateViewsAll()
{
	POSITION posn = GetFirstViewPosition();
	while (posn)
	{
		CChalkView *pView = static_cast<CChalkView *>(GetNextView(posn));
		if (pView)
		{
			pView->Invalidate(FALSE); // Repaint the view (don't flicker)
		}
	}
}

// ----------------------------------------------------------------------------
// This rectangle is invalid in all views
void CChalkDoc::InvalidateViewsRect(CRect rect, CRect border)
{
	POSITION posn = GetFirstViewPosition();
	while (posn)
	{
		CChalkView *pView = static_cast<CChalkView *>(GetNextView(posn));
		if (pView)
		{
			CRect rt;
			float zoom = pView->m_Display.GetZoom();
			rt.left = zoom * rect.left + border.left;
			rt.top = zoom * rect.top + border.top;
			rt.right = zoom * rect.right + border.right;
			rt.bottom = zoom * rect.bottom + border.bottom;
			rt -= pView->GetDeviceScrollPosition(); // Take scroll position into account
			pView->InvalidateRect(rt, FALSE); // Repaint the rectangle (don't flicker)
		}
	}
}

// ----------------------------------------------------------------------------
// Update views when the current frame changes
void CChalkDoc::ChangeFrame()
{
	// Update the status bar
	m_StatusFrame = FORMAT_SPACE;
	vector<ChalkLayer*>::const_iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			if (layer->m_Visible == layer_active)
			{
				tstringstream s;
				s << "Frame " << layer->m_CurrentFrame + 1 << "/" << layer->GetNumFrames();
				m_StatusFrame = s.str();
				break;
			}
		}
	}

	// Update all the views
	InvalidateViewsAll();
}


// ----------------------------------------------------------------------------
void CChalkDoc::OnAnimRewind() 
{
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			layer->m_CurrentFrame = 0;
		}
	}
	ChangeFrame();
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnAnimPrevious() 
{
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			if (layer->m_CurrentFrame <= 0)
				layer->m_CurrentFrame = layer->GetNumFrames();
			layer->m_CurrentFrame--;
			if (layer->m_CurrentFrame <= 0)
				layer->m_CurrentFrame = 0;
		}
	}
	ChangeFrame();
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnAnimNext() 
{
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			layer->m_CurrentFrame++;
			if (layer->m_CurrentFrame >= layer->GetNumFrames())
				layer->m_CurrentFrame = 0;
		}
	}
	ChangeFrame();
}

// ----------------------------------------------------------------------------
void CChalkDoc::OnAnimFfwd() 
{
	vector<ChalkLayer*>::iterator i;
	for (i = m_Layers.begin(); i != m_Layers.end(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			layer->m_CurrentFrame = max(0, layer->GetNumFrames() - 1);
		}
	}
	ChangeFrame();
}

