// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkView.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED:  5 Apr 2001 by Carl Muller
//
// Implementation of the CChalkView class
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalk.h"

#include "chalkDoc.h"
#include "chalkView.h"

#include "PropSheetBase.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

const int TIMER_SCROLL = 100; // Identifier
const int AUTOSCROLL_SPEED = 20; // Twenty times per second

// ----------------------------------------------------------------------------
// CChalkView
IMPLEMENT_DYNCREATE(CChalkView, CScrollView)

BEGIN_MESSAGE_MAP(CChalkView, CScrollView)
	//{{AFX_MSG_MAP(CChalkView)
	ON_WM_SIZE()
	ON_WM_ERASEBKGND()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_KEYDOWN()
	ON_UPDATE_COMMAND_UI(ID_DRAW_SELECT, OnUpdateDrawSelect)
	ON_COMMAND(ID_DRAW_SELECT, OnDrawSelect)
	ON_UPDATE_COMMAND_UI(ID_DRAW_DRAW, OnUpdateDrawDraw)
	ON_COMMAND(ID_DRAW_DRAW, OnDrawDraw)
	ON_UPDATE_COMMAND_UI(ID_DRAW_RECTANGLE, OnUpdateDrawRectangle)
	ON_COMMAND(ID_DRAW_RECTANGLE, OnDrawRectangle)
	ON_UPDATE_COMMAND_UI(ID_DRAW_SPRITE, OnUpdateDrawSprite)
	ON_COMMAND(ID_DRAW_SPRITE, OnDrawSprite)
	ON_UPDATE_COMMAND_UI(ID_DRAW_BOX, OnUpdateDrawBox)
	ON_COMMAND(ID_DRAW_BOX, OnDrawBox)
	ON_UPDATE_COMMAND_UI(ID_DRAW_OFFSETS, OnUpdateDrawOffsets)
	ON_COMMAND(ID_DRAW_OFFSETS, OnDrawOffsets)
	ON_UPDATE_COMMAND_UI(ID_DRAW_PATH, OnUpdateDrawPath)
	ON_COMMAND(ID_DRAW_PATH, OnDrawPath)
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_TIMER()
	ON_WM_CHAR()
	ON_WM_CREATE()
	ON_WM_DESTROY()
	ON_UPDATE_COMMAND_UI(ID_DRAW_GRID, OnUpdateDrawGrid)
	ON_COMMAND(ID_DRAW_GRID, OnDrawGrid)
	ON_COMMAND(ID_DRAW_TEXT, OnDrawText)
	ON_UPDATE_COMMAND_UI(ID_DRAW_TEXT, OnUpdateDrawText)
	ON_COMMAND(ID_DRAW_ATTRIBUTES, OnDrawAttributes)
	ON_UPDATE_COMMAND_UI(ID_DRAW_ATTRIBUTES, OnUpdateDrawAttributes)
	ON_COMMAND(ID_EDIT_PROPERTIES, OnEditProperties)
	ON_WM_RBUTTONDBLCLK()
	ON_WM_LBUTTONDBLCLK()
	ON_COMMAND(ID_LAYER_PROPERTIES, OnLayerProperties)
	ON_COMMAND(ID_DRAW_REPLACEALL, OnDrawReplaceall)
	ON_UPDATE_COMMAND_UI(ID_DRAW_REPLACEALL, OnUpdateDrawReplaceall)
	ON_COMMAND(ID_DRAW_LASSO, OnDrawLasso)
	ON_UPDATE_COMMAND_UI(ID_DRAW_LASSO, OnUpdateDrawLasso)
	ON_UPDATE_COMMAND_UI(ID_FRAME_PROPERTIES, OnUpdateFrameProperties)
	ON_COMMAND(ID_FRAME_PROPERTIES, OnFrameProperties)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PROPERTIES, OnUpdateEditProperties)
	ON_COMMAND(ID_BRUSH_DECREMENT, OnBrushDecrement)
	ON_UPDATE_COMMAND_UI(ID_BRUSH_DECREMENT, OnUpdateBrushDecrement)
	ON_COMMAND(ID_BRUSH_INCREMENT, OnBrushIncrement)
	ON_UPDATE_COMMAND_UI(ID_BRUSH_INCREMENT, OnUpdateBrushIncrement)
	ON_COMMAND(ID_BRUSH_FLIPX, OnBrushFlipx)
	ON_UPDATE_COMMAND_UI(ID_BRUSH_FLIPX, OnUpdateBrushFlipx)
	ON_COMMAND(ID_BRUSH_FLIPY, OnBrushFlipy)
	ON_UPDATE_COMMAND_UI(ID_BRUSH_FLIPY, OnUpdateBrushFlipy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_MOVE, OnUpdateEditMove)
	ON_COMMAND(ID_EDIT_MOVE, OnEditMove)
	ON_COMMAND(ID_EDIT_PURGE, OnEditPurge)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PURGE, OnUpdateEditPurge)
	ON_COMMAND(ID_EDIT_SORT, OnEditSort)
	ON_UPDATE_COMMAND_UI(ID_EDIT_SORT, OnUpdateEditSort)
	ON_COMMAND(ID_EDIT_ALIGN, OnEditAlign)
	ON_UPDATE_COMMAND_UI(ID_EDIT_ALIGN, OnUpdateEditAlign)
	ON_UPDATE_COMMAND_UI(ID_LAYER_PROPERTIES, OnUpdateLayerProperties)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CUT, OnUpdateEditCut)
	ON_COMMAND(ID_EDIT_CUT, OnEditCut)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdateEditPaste)
	ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
	ON_UPDATE_COMMAND_UI(ID_EDIT_DELETE, OnUpdateEditDelete)
	ON_COMMAND(ID_EDIT_DELETE, OnEditDelete)
	ON_UPDATE_COMMAND_UI(ID_EDIT_SELECT_ALL, OnUpdateEditSelectAll)
	ON_COMMAND(ID_EDIT_SELECT_ALL, OnEditSelectAll)
	ON_COMMAND(ID_FRAME_DELETE, OnFrameDelete)
	ON_UPDATE_COMMAND_UI(ID_FRAME_DELETE, OnUpdateFrameDelete)
	ON_COMMAND(ID_FRAME_INSERT, OnFrameInsert)
	ON_UPDATE_COMMAND_UI(ID_FRAME_INSERT, OnUpdateFrameInsert)
	ON_UPDATE_COMMAND_UI(ID_FILE_TOPALETTE, OnUpdateFileTopalette)
	ON_COMMAND(ID_FILE_TOPALETTE, OnFileTopalette)
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CScrollView::OnFilePrintPreview)
	ON_COMMAND(ID_ZOOM_100, OnZoom100)
	ON_COMMAND(ID_ZOOM_200, OnZoom200)
	ON_COMMAND(ID_ZOOM_400, OnZoom400)
	ON_COMMAND(ID_ZOOM_800, OnZoom800)
	ON_COMMAND(ID_ZOOM_50, OnZoom50)
	ON_COMMAND(ID_ZOOM_25, OnZoom25)
	ON_COMMAND(ID_ZOOM_12, OnZoom12)
END_MESSAGE_MAP()

// ----------------------------------------------------------------------------
// CChalkView construction/destruction
CChalkView::CChalkView()
{
	m_IsPrinting = false;
	m_OldZoomMult = 1;
	m_OldZoomDiv = 1;
	m_Timer = 0;
	m_AutoScroll = autoscroll_none;
	m_LastMouseMove = CPoint(0,0);
	m_LastMouseFlags = 0;
	m_LastMouseLeft = false;
}

// ----------------------------------------------------------------------------
CChalkView::~CChalkView()
{
}

// ----------------------------------------------------------------------------
BOOL CChalkView::PreCreateWindow(CREATESTRUCT& cs)
{
	// TODO: Modify the Window class or styles here by modifying
	//  the CREATESTRUCT cs

	return CScrollView::PreCreateWindow(cs);
}

// ----------------------------------------------------------------------------
int CChalkView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CScrollView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	m_Timer = SetTimer(TIMER_SCROLL, 1000 / AUTOSCROLL_SPEED, NULL);
	return 0;
}

// ----------------------------------------------------------------------------
void CChalkView::OnDestroy() 
{
	KillTimer(m_Timer);

	CScrollView::OnDestroy();
}

// ----------------------------------------------------------------------------
// CChalkView drawing
void CChalkView::OnDraw(CDC* pDC)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	CSize sz = m_Display.GetBitmapSize();
	CPoint topleft = GetDeviceScrollPosition();
	CRect bounds;
	pDC->GetClipBox(bounds);
	if (m_IsPrinting)
		topleft = CPoint(0, 0);

	// Is a background required?
	bool solid = false;
	vector<ChalkLayer*>::iterator i = pDoc->m_Layers.begin();
	if (i != pDoc->m_Layers.end())
	{
		ChalkLayer* baselayer = *i;
		for (++i; i != pDoc->m_Layers.end(); ++i)
		{
			ChalkLayer* layer = *i;
			if (layer && (layer->m_Visible != layer_hidden))
				if (layer->m_Solid)
					solid = true;
		}
		baselayer->m_Visible = solid ? layer_hidden : layer_visible;
	}

	// For print/print preview, plot the screen in segments
	int xoffset, yoffset;
	for (yoffset = 0; yoffset < bounds.Height(); yoffset += sz.cy)
	{
		for (xoffset = 0; xoffset < bounds.Width(); xoffset += sz.cx)
		{
			// Render all the layers (including background)
			m_Display.StartPaintScene();
			for (i = pDoc->m_Layers.begin(); i != pDoc->m_Layers.end(); ++i)
			{
				ChalkLayer* layer = *i;
				if (layer && (layer->m_Visible != layer_hidden))
				{
					layer->m_OffsetX = int((topleft.x + xoffset) * layer->m_RelativeX);
					layer->m_OffsetY = int((topleft.y + yoffset) * layer->m_RelativeY);
					CRect layerbounds;
					layerbounds.left   = max(bounds.left   - topleft.x, 0);
					layerbounds.top    = max(bounds.top    - topleft.y, 0);
					layerbounds.right  = min(bounds.right  - topleft.x, sz.cx);
					layerbounds.bottom = min(bounds.bottom - topleft.y, sz.cy);
					layerbounds.left   += layer->m_OffsetX;
					layerbounds.top    += layer->m_OffsetY;
					layerbounds.right  += layer->m_OffsetX;
					layerbounds.bottom += layer->m_OffsetY;
					m_Display.SetBounds(CPoint(layer->m_OffsetX, layer->m_OffsetY), layerbounds);
					layer->Draw(m_Display, layerbounds);
				}
			}

			// Actually display the scene
			CRect dbounds;
			dbounds.left = bounds.left + xoffset;
			dbounds.top = bounds.top + yoffset;
			dbounds.right = min(bounds.left + xoffset + sz.cx, bounds.right);
			dbounds.bottom = min(bounds.top + yoffset + sz.cy, bounds.bottom);
			m_Display.EndPaintScene(dbounds, pDC);
		}
	}
}

// ----------------------------------------------------------------------------
// CChalkView printing
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
BOOL CChalkView::OnPreparePrinting(CPrintInfo* pInfo)
{
	// default preparation
	return DoPreparePrinting(pInfo);
}

// ----------------------------------------------------------------------------
void CChalkView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	m_IsPrinting = true;
	m_OldZoomMult = m_Display.GetZoomMult();
	m_OldZoomDiv = m_Display.GetZoomDiv();
	m_Display.SetZoom(2, 1); // Use zooming to allow dithering
	// TODO: add extra initialization before printing
}

// ----------------------------------------------------------------------------
void CChalkView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add cleanup after printing
	m_IsPrinting = false;
	m_Display.SetZoom(m_OldZoomMult, m_OldZoomDiv); // Back to normal
}

// ----------------------------------------------------------------------------
// CChalkView diagnostics
#ifdef _DEBUG
void CChalkView::AssertValid() const
{
	CScrollView::AssertValid();
}

void CChalkView::Dump(CDumpContext& dc) const
{
	CScrollView::Dump(dc);
}

CChalkDoc* CChalkView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CChalkDoc)));
	return static_cast<CChalkDoc*>(m_pDocument);
}
#endif //_DEBUG

// ----------------------------------------------------------------------------
void CChalkView::OnInitialUpdate() 
{
	CRect rect;
	GetClientRect(&rect);
	// Ensure that the bitmap has a minimum size, in case of printing
	CSize sz = rect.Size();
	sz.cx = max(sz.cx, 32);
	sz.cy = max(sz.cy, 32);
	m_Display.SetBitmapSize(sz, this);
	SetScrollSizes(MM_TEXT, CSize(1024,1024));
	
	CScrollView::OnInitialUpdate();
}

// ----------------------------------------------------------------------------
void CChalkView::OnSize(UINT nType, int cx, int cy) 
{
	CScrollView::OnSize(nType, cx, cy);

	CRect rect;
	GetClientRect(&rect);
	// Ensure that the bitmap has a minimum size, in case of printing
	CSize sz = rect.Size();
	sz.cx = max(sz.cx, 32);
	sz.cy = max(sz.cy, 32);
	m_Display.SetBitmapSize(sz, this);
}

// ----------------------------------------------------------------------------
BOOL CChalkView::OnEraseBkgnd(CDC* pDC) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	CRect rect;
	GetClientRect(&rect);
	CSize currentsize = GetTotalSize();
	CSize docsize = pDoc->m_DocSize;
	if (docsize.cx == 0)
		docsize.cx = max(16, rect.Width());
	else
		docsize.cx = m_Display.GetZoom() * docsize.cx;
	if (docsize.cy == 0)
		docsize.cy = max(16, rect.Height());
	else
		docsize.cy = m_Display.GetZoom() * docsize.cy;
	if (currentsize != docsize)
	{
		SetScrollSizes(MM_TEXT, docsize);
		return TRUE;
	}
	return FALSE;
}

// ----------------------------------------------------------------------------
// Interaction
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Convert a point from windows co-ordinates to world co-ordinates
CPoint CChalkView::ScalePoint(CPoint point)
{
	CPoint result;
	result.x = (float) point.x / m_Display.GetZoom();
	result.y = (float) point.y / m_Display.GetZoom();
	return result;
}

// ----------------------------------------------------------------------------
void CChalkView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	SetCapture();
	m_AutoScroll = autoscroll_none;
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;
	m_LastMouseLeft = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnLButtonDblClk(this, nFlags, pt))
				break;
	}
	
	CScrollView::OnLButtonDblClk(nFlags, point);
}

// ----------------------------------------------------------------------------
void CChalkView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	SetCapture();
	m_AutoScroll = autoscroll_none;
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;
	m_LastMouseLeft = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnLButtonDown(this, nFlags, pt))
				break;
	}
	
	CScrollView::OnLButtonDown(nFlags, point);
}

// ----------------------------------------------------------------------------
void CChalkView::OnRButtonDblClk(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	SetCapture();
	m_AutoScroll = autoscroll_none;
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;
	m_LastMouseLeft = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnRButtonDblClk(this, nFlags, pt))
				break;
	}
	
	CScrollView::OnRButtonDblClk(nFlags, point);
}

// ----------------------------------------------------------------------------
void CChalkView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	SetCapture();
	m_AutoScroll = autoscroll_none;
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;
	m_LastMouseLeft = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnRButtonDown(this, nFlags, pt))
				break;
	}
	
	CScrollView::OnRButtonDown(nFlags, point);
}

// ----------------------------------------------------------------------------
void CChalkView::OnMouseMove(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	m_AutoScroll = autoscroll_none;
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnMouseMove(this, nFlags, pt))
				break;
	}
	
	CScrollView::OnMouseMove(nFlags, point);
}

// ----------------------------------------------------------------------------
void CChalkView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;
	m_LastMouseLeft = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnLButtonUp(this, nFlags, pt))
				break;
	}
	ReleaseCapture();
	m_AutoScroll = autoscroll_none;
	
	CScrollView::OnLButtonUp(nFlags, point);
}

// ----------------------------------------------------------------------------
void CChalkView::OnRButtonUp(UINT nFlags, CPoint point) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	point += GetDeviceScrollPosition();
	CPoint pt = ScalePoint(point);
	m_LastMouseMove = point;
	m_LastMouseFlags = nFlags;
	m_LastMouseLeft = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnRButtonUp(this, nFlags, pt))
				break;
	}
	ReleaseCapture();
	m_AutoScroll = autoscroll_none;
	
	CScrollView::OnRButtonUp(nFlags, point);
}



// ----------------------------------------------------------------------------
void CChalkView::OnTimer(UINT nIDEvent) 
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Has the mouse left the window while the buttons were up?
	bool bLeftWindow = false;
	if ((m_LastMouseFlags & (MK_LBUTTON | MK_RBUTTON)) == 0)
	{
		CPoint pt, pt2;
		GetCursorPos(&pt);
		pt2 = pt;
		ScreenToClient(&pt);
		CRect rect;
		GetClientRect(&rect);
		// Am I over the window?
		if (rect.PtInRect(pt) && (::WindowFromPoint(pt2) == GetSafeHwnd()))
			m_LastMouseLeft = false;
		else if (!m_LastMouseLeft)
		{
			bLeftWindow = true; // So that the tracking rectangle is hidden
			m_LastMouseLeft = true;
			m_LastMouseMove = CPoint(-1,-1);
		}
	}

	// Handle autoscroll
	if (!bLeftWindow)
	{
		CPoint oldpos = GetDeviceScrollPosition();
		if (m_AutoScroll & autoscroll_right)
			OnKeyDown(VK_RIGHT, 1, 0);
		if (m_AutoScroll & autoscroll_left)
			OnKeyDown(VK_LEFT, 1, 0);
		if (m_AutoScroll & autoscroll_down)
			OnKeyDown(VK_DOWN, 1, 0);
		if (m_AutoScroll & autoscroll_up)
			OnKeyDown(VK_UP, 1, 0);
		CPoint newpos = GetDeviceScrollPosition();
		m_LastMouseMove += (newpos - oldpos);
	}

	// From top to bottom layer
	CPoint pt = ScalePoint(m_LastMouseMove);
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
		{
			// Special timer behavior?
			if (layer->OnTimer(this, nIDEvent))
				break;
			// Normal mouse move behavior?
			if (m_AutoScroll || bLeftWindow)
				if (layer->OnMouseMove(this, m_LastMouseFlags, pt))
					break;
		}
	}
	
	CScrollView::OnTimer(nIDEvent);
}

// ----------------------------------------------------------------------------
// Handle keypresses within a view pane, or the autoscroll messages
// generated when dragging with the mouse
void CChalkView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	CPoint pt = GetDeviceScrollPosition();
	int xpos = pt.x;
	int ypos = pt.y;
	CSplitterWnd* pSplitter = reinterpret_cast<CSplitterWnd *>(GetParent());

	if (!pSplitter) // Paranoid!
	{
		CScrollView::OnKeyDown(nChar, nRepCnt, nFlags);
	}
	else
	{
		switch (nChar) {
		case VK_LEFT:
			pSplitter->DoScrollBy(this, CSize(-8,0));
			break;

		case VK_RIGHT:
			pSplitter->DoScrollBy(this, CSize(8,0));
			break;

		case VK_UP:
			pSplitter->DoScrollBy(this, CSize(0,-8));
			break;

		case VK_DOWN:
			pSplitter->DoScrollBy(this, CSize(0,8));
			break;

		case VK_HOME:
			pSplitter->DoScrollBy(this, CSize(-128,0));
			break;

		case VK_END:
			pSplitter->DoScrollBy(this, CSize(128,0));
			break;

		case VK_PRIOR:
			pSplitter->DoScrollBy(this, CSize(0,-128));
			break;

		case VK_NEXT:
			pSplitter->DoScrollBy(this, CSize(0,128));
			break;

		default:
			CScrollView::OnKeyDown(nChar, nRepCnt, nFlags);
			break;
		}
	}
}

// ----------------------------------------------------------------------------
void CChalkView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			if (layer->OnChar(this, nChar, nRepCnt))
				break;
	}
	
	CScrollView::OnChar(nChar, nRepCnt, nFlags);
}

// ----------------------------------------------------------------------------
// Handle automatic scrolling if mouse is dragged outside the view
void CChalkView::AutoScroll(CPoint pt)
{
	CRect rect;
	GetClientRect(&rect);
	rect += GetScrollPosition();
	pt.x = m_Display.GetZoom() * pt.x;
	pt.y = m_Display.GetZoom() * pt.y;

	// Handle autoscroll
	m_AutoScroll = autoscroll_none;
	if (pt.x > rect.right)
		m_AutoScroll = autoscroll_right;
	if (pt.x < rect.left)
		m_AutoScroll = autoscroll_left;
	if (pt.y > rect.bottom)
		m_AutoScroll = autoscroll(m_AutoScroll | autoscroll_down);
	if (pt.y < rect.top)
		m_AutoScroll = autoscroll(m_AutoScroll | autoscroll_up);
}

// ----------------------------------------------------------------------------
void CChalkView::UpdateDrawMode(CCmdUI* pCmdUI, DrawMode drawmode)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Does the active layer support this drawing mode?
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
		{
			if (layer->m_DrawModesSupported & drawmode)
			{
				if (layer->m_DrawMode & drawmode)
					pCmdUI->SetCheck(1);
				else
					pCmdUI->SetCheck(0);
				pCmdUI->Enable(TRUE);
				return;
			}
		}
	}

	// Can we change to another layer?
	pCmdUI->SetCheck(0);
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_visible))
		{
			if (layer->m_DrawModesSupported & drawmode)
			{
				pCmdUI->Enable(TRUE);
				return;
			}
		}
	}
	pCmdUI->Enable(FALSE);
}

// ----------------------------------------------------------------------------
void CChalkView::UpdateDrawAttribute(CCmdUI* pCmdUI, DrawAttributes drawattr)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// Do any layers support this drawing mode?
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && ((layer->m_Visible == layer_active) || (layer->m_Visible == layer_visible)))
		{
			if (layer->m_DrawAttributesSupported & drawattr)
			{
				if (layer->m_DrawAttributes & drawattr)
					pCmdUI->SetCheck(1);
				else
					pCmdUI->SetCheck(0);
				pCmdUI->Enable(TRUE);
				return;
			}
		}
	}

	// Can we change to another layer?
	pCmdUI->SetCheck(0);
	pCmdUI->Enable(FALSE);
}

// ----------------------------------------------------------------------------
void CChalkView::SetDrawMode(DrawMode drawmode)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	bool bDone = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
		{
			if (layer->m_DrawModesSupported & drawmode)
			{
				layer->m_DrawMode = drawmode;
				bDone = true;
			}
			else
				layer->m_DrawMode = drawmode_none;
		}
	}

	// Can that not be done with the current layer?
	if (!bDone)
	{
		for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
		{
			ChalkLayer* layer = *i;
			if (layer && (layer->m_Visible == layer_visible))
			{
				if (layer->m_DrawModesSupported & drawmode)
				{
					// Change which layer is current
					pDoc->SelectLayer(layer);
					layer->m_DrawMode = drawmode;
					break;
				}
			}
		}
	}
}

// ----------------------------------------------------------------------------
void CChalkView::ToggleDrawAttribute(DrawAttributes drawattr)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	bool bDone = false;

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer)
		{
			if (layer->m_DrawAttributesSupported & drawattr)
				layer->m_DrawAttributes = DrawAttributes(layer->m_DrawAttributes ^ drawattr);
		}
	}
	Invalidate(FALSE); // Update the entire view
}

// ----------------------------------------------------------------------------
void CChalkView::UpdateDrawCommand(CCmdUI* pCmdUI, DrawCommands drawcmd)
{
	ChalkLayer* pLayer = GetActiveLayer();
	if (pLayer)
	{
		if (pLayer->m_DrawCommandsSupported & drawcmd)
			pCmdUI->Enable(TRUE);
		else
			pCmdUI->Enable(FALSE);
	}
	else
		pCmdUI->Enable(FALSE);
}

// ----------------------------------------------------------------------------
void CChalkView::PerformDrawCommand(DrawCommands drawcmd)
{
	ChalkLayer* pLayer = GetActiveLayer();
	if (pLayer)
		if (pLayer->m_DrawCommandsSupported & drawcmd)
			pLayer->EditCommand(drawcmd);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawSelect(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_select);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawSelect() 
{
	SetDrawMode(drawmode_select);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawDraw(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_draw);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawDraw() 
{
	SetDrawMode(drawmode_draw);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawRectangle(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_rectangle);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawRectangle() 
{
	SetDrawMode(drawmode_rectangle);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawReplaceall() 
{
	SetDrawMode(drawmode_replaceall);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawReplaceall(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_replaceall);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawLasso() 
{
	SetDrawMode(drawmode_lasso);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawLasso(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_lasso);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawSprite(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_sprite);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawSprite() 
{
	SetDrawMode(drawmode_sprite);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawBox(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_box);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawBox() 
{
	SetDrawMode(drawmode_box);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawOffsets(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_offset);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawOffsets() 
{
	SetDrawMode(drawmode_offset);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawPath(CCmdUI* pCmdUI) 
{
	UpdateDrawMode(pCmdUI, drawmode_path);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawPath() 
{
	SetDrawMode(drawmode_path);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawGrid(CCmdUI* pCmdUI) 
{
	UpdateDrawAttribute(pCmdUI, drawattr_grid);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawGrid() 
{
	ToggleDrawAttribute(drawattr_grid);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawText(CCmdUI* pCmdUI) 
{
	UpdateDrawAttribute(pCmdUI, drawattr_showtext);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawText() 
{
	ToggleDrawAttribute(drawattr_showtext);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateDrawAttributes(CCmdUI* pCmdUI) 
{
	UpdateDrawAttribute(pCmdUI, drawattr_attributes);
}

// ----------------------------------------------------------------------------
void CChalkView::OnDrawAttributes() 
{
	ToggleDrawAttribute(drawattr_attributes);
}

// ----------------------------------------------------------------------------
ChalkLayer* CChalkView::GetActiveLayer()
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// From top to bottom layer
	vector<ChalkLayer*>::reverse_iterator i;
	for (i = pDoc->m_Layers.rbegin(); i != pDoc->m_Layers.rend(); ++i)
	{
		ChalkLayer* layer = *i;
		if (layer && (layer->m_Visible == layer_active))
			return layer;
	}
	return NULL;
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditProperties(CCmdUI* pCmdUI) 
{
	ChalkObject* selected = NULL;
	ChalkLayer* pLayer = GetActiveLayer();

	if (pLayer)
		selected = pLayer->GetSelectedObject();
	if (selected)
		pCmdUI->Enable(TRUE);
	else
		pCmdUI->Enable(FALSE);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditProperties() 
{
	ChalkObject *selected = NULL;
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	ChalkLayer* pLayer = GetActiveLayer();

	if (pLayer)
		selected = pLayer->GetSelectedObject();
	if (selected)
	{
		if (selected->m_TypeId == GUID_SPRITE) // My own RTTI
		{
			CPropertySheetSprite dlg(static_cast<ChalkSprite *>(selected), pDoc);
			dlg.DoModal();
		}
		else if (selected->m_TypeId == GUID_BOX) // My own RTTI
		{
			CPropertySheetBox dlg(static_cast<ChalkBox *>(selected), pDoc);
			dlg.DoModal();
		}
	}
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateLayerProperties(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_layer_properties);
}

// ----------------------------------------------------------------------------
void CChalkView::OnLayerProperties() 
{
	PerformDrawCommand(drawcmd_layer_properties);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateFrameProperties(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_frame_properties);
}

// ----------------------------------------------------------------------------
void CChalkView::OnFrameProperties() 
{
	PerformDrawCommand(drawcmd_frame_properties);
}


// ----------------------------------------------------------------------------
void CChalkView::OnBrushDecrement() 
{
	PerformDrawCommand(drawcmd_decrement);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateBrushDecrement(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_decrement);
}

// ----------------------------------------------------------------------------
void CChalkView::OnBrushIncrement() 
{
	PerformDrawCommand(drawcmd_increment);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateBrushIncrement(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_increment);
}

// ----------------------------------------------------------------------------
void CChalkView::OnBrushFlipx() 
{
	PerformDrawCommand(drawcmd_flipx);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateBrushFlipx(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_flipx);
}

// ----------------------------------------------------------------------------
void CChalkView::OnBrushFlipy() 
{
	PerformDrawCommand(drawcmd_flipy);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateBrushFlipy(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_flipy);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditMove(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_move);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditMove() 
{
	PerformDrawCommand(drawcmd_move);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditPurge() 
{
	PerformDrawCommand(drawcmd_purge);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditPurge(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_purge);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditSort() 
{
	PerformDrawCommand(drawcmd_sort);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditSort(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_sort);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditAlign() 
{
	PerformDrawCommand(drawcmd_align);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditAlign(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_align);
}


// ----------------------------------------------------------------------------
void CChalkView::OnEditCut() 
{
	PerformDrawCommand(drawcmd_edit_copy);
	PerformDrawCommand(drawcmd_edit_delete);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditCut(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_edit_cut);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditCopy() 
{
	PerformDrawCommand(drawcmd_edit_copy);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditCopy(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_edit_copy);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditPaste() 
{
	PerformDrawCommand(drawcmd_edit_paste);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditPaste(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_edit_paste);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditDelete() 
{
	PerformDrawCommand(drawcmd_edit_delete);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditDelete(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_edit_delete);
}

// ----------------------------------------------------------------------------
void CChalkView::OnEditSelectAll() 
{
	PerformDrawCommand(drawcmd_edit_selectall);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateEditSelectAll(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_edit_selectall);
}

// ----------------------------------------------------------------------------
void CChalkView::OnFrameDelete() 
{
	PerformDrawCommand(drawcmd_frame_delete);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateFrameDelete(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_frame_delete);
}

// ----------------------------------------------------------------------------
void CChalkView::OnFrameInsert() 
{
	PerformDrawCommand(drawcmd_frame_insert);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateFrameInsert(CCmdUI* pCmdUI) 
{
	UpdateDrawCommand(pCmdUI, drawcmd_frame_insert);
}

// ----------------------------------------------------------------------------
void CChalkView::SetZoom(int zoommult, int zoomdiv)
{
	CChalkDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	tstringstream S;
	S << (100.0 * zoommult) / zoomdiv << "%";
	pDoc->m_StatusZoom = S.str();
	m_Display.SetZoom(zoommult, zoomdiv);
	OnEraseBkgnd(NULL);
	Invalidate(FALSE); // Update the entire view
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom12(void)
{
	SetZoom(1, 8);
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom25(void)
{
	SetZoom(1, 4);
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom50(void)
{
	SetZoom(1, 2);
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom100(void)
{
	SetZoom(1, 1);
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom200(void)
{
	SetZoom(2, 1);
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom400(void)
{
	SetZoom(4, 1);
}

// ----------------------------------------------------------------------------
void CChalkView::OnZoom800(void)
{
	SetZoom(8, 1);
}

// ----------------------------------------------------------------------------
void CChalkView::OnUpdateFileTopalette(CCmdUI* pCmdUI) 
{
	BOOL bEnable = FALSE;
	ChalkLayer* pLayer = GetActiveLayer();
	if (pLayer)
	{
		if (pLayer->GetPalette())
			bEnable = TRUE;
	}
	pCmdUI->Enable(bEnable);
}

// ----------------------------------------------------------------------------
void CChalkView::OnFileTopalette()
{
	ChalkLayer* pLayer = GetActiveLayer();
	if (pLayer)
	{
		ChalkPalette *pPalette = pLayer->GetPalette();
		if (pPalette)
		{
			CChalkDoc *pDoc = theApp.OnFileNewEmpty();
			if (pDoc)
			{
				// Add a palette layer
				ChalkLayerPaletteBars* pLayerNew;
				pLayerNew = new ChalkLayerPaletteBars(*pDoc);
				pDoc->m_Layers.push_back(pLayerNew);
				pLayerNew->m_Data = *pPalette;
				pDoc->SelectLayer(pLayerNew);
				pDoc->CalcSize();
			}
		}
	}
}
