// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkBitmap.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 23 Oct 2001 by Carl Muller
//
// Implements class which supports bitmap objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"


// ----------------------------------------------------------------------------
static int rounddiv(int a, int b)
{
	return ( ((a) + (b) - 1) / (b) );
}


// ----------------------------------------------------------------------------
// ChalkBitmap 
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkBitmap::ChalkBitmap()  : ChalkObject(GUID_VOID, GUID_BITMAP),
	m_GridX(16),
	m_GridY(16)
{
}

// ----------------------------------------------------------------------------
HRESULT ChalkBitmap::Write(ChalkStream& os) const
{
	os.WriteElement1("Bitmap");
	os.WriteInteger("gridx", m_GridX);
	os.WriteInteger("gridy", m_GridY);
	m_Palette.Write(os);
	m_Character.Write(os);
	os.WriteEndElement("Bitmap");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkBitmap::Read(ChalkStream& is)
{
	is.ReadElement1("Bitmap");
	m_GridX = is.ReadInteger("gridx");
	m_GridY = is.ReadInteger("gridy");
	m_Palette.Read(is);
	m_Character.Read(is);
	is.ReadEndElement("Bitmap");
	return S_OK;
}

// ----------------------------------------------------------------------------
// Does this class support this alternative file format?
bool ChalkBitmap::SupportsFile(const CFileName& filename)
{
	if (filename.HasExt(FILEEXTENSION_LBM))
		return true;
	if (filename.HasExt(FILEEXTENSION_PBM))
		return true;
	if (filename.HasExt(FILEEXTENSION_BBM))
		return true;
	if (filename.HasExt(FILEEXTENSION_PCX))
		return true;
	if (filename.HasExt(FILEEXTENSION_TGA))
		return true;
	if (filename.HasExt(FILEEXTENSION_BMP))
		return true;
	return false;
}

// ----------------------------------------------------------------------------
// Distribute the read alternative file command
HRESULT ChalkBitmap::ReadFile(const CFileName& filename)
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return ReadXML(filename);
	if (filename.HasExt(FILEEXTENSION_LBM))
		return ReadLBM(filename);
	if (filename.HasExt(FILEEXTENSION_PBM))
		return ReadLBM(filename);
	if (filename.HasExt(FILEEXTENSION_BBM))
		return ReadLBM(filename);
	if (filename.HasExt(FILEEXTENSION_PCX))
		return ReadPCX(filename);
	if (filename.HasExt(FILEEXTENSION_TGA))
		return ReadTGA(filename);
	if (filename.HasExt(FILEEXTENSION_BMP))
		return ReadBMP(filename);
	if (ChalkFont::SupportsFile(filename))
		return ReadFont(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}


// ----------------------------------------------------------------------------
// Distribute the write alternative file command
HRESULT ChalkBitmap::WriteFile(const CFileName& filename) const
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return WriteXML(filename);
	if (filename.HasExt(FILEEXTENSION_LBM))
		return WriteLBM(filename);
	if (filename.HasExt(FILEEXTENSION_PBM))
		return WritePBM(filename);
	if (filename.HasExt(FILEEXTENSION_BBM))
		return WritePBM(filename);
	if (filename.HasExt(FILEEXTENSION_PCX))
		return WritePCX(filename);
	if (filename.HasExt(FILEEXTENSION_TGA))
		return WriteTGA(filename);
	if (filename.HasExt(FILEEXTENSION_BMP))
		return WriteBMP(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}


// ----------------------------------------------------------------------------
// Load a character set
HRESULT ChalkBitmap::ReadFont(const CFileName& filename)
{
	ChalkFont font;
	HRESULT hr = font.ReadFile(filename);
	if (FAILED(hr))
		return hr;

	// Create a bitmap from the font
	int numblocks = 16;
	int numchars = font.m_Data.size();
	if (numchars <= 0)
		return E_FAIL;
	int gx = font.m_Data[0].m_Width;
	int gy = font.m_Data[0].m_Height;
	int numplanes = font.m_Data[0].m_NumPlanes;
	int width = gx * numblocks;
	int height = gy * rounddiv(numchars, numblocks);
	if (width * height <= 0)
		return E_FAIL;
	m_Character.SetSize(width, height, numplanes);
	m_Palette.SetFalseColour();

	// Copy characters into the bitmap
	int i, x, y, xpos, ypos;
	for (i = 0; i < numchars; ++i)
	{
		xpos = gx * (i % numblocks);
		ypos = gy * (i / numblocks);
		for (y = 0; y < gy; ++y)
		{
			ChalkColourIndex *pDst = m_Character.m_Data + width * (ypos + y) + xpos;
			const ChalkColourIndex *pSrc = font.m_Data[i].m_Data + gx * y;
			for (x = 0; x < gx; ++x)
				*pDst++ = *pSrc++;
		}
	}

	return hr;
}
