// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkLayer.h
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 16 Jul 2000 by Carl Muller
//
// This defines the "layer" system used by chalk.
// Each view consists of a number of layers, with (hopefully)
// the bottom one being solid and the others semi-transparent
// (e.g. a map layer (.MAP file) on the bottom and an animation layer
// (.FRK file) over it.
// ----------------------------------------------------------------------------

#include "chalkdisplay.h"
#include "chalkobjects.h"

class CChalkView;
class CChalkDoc;

// ----------------------------------------------------------------------------
// What are the attributes of this layer?
typedef enum {
	layer_hidden,
	layer_dimmed,
	layer_visible,
	layer_active,
} LayerVisibility;

// Draw modes for left mouse button
typedef enum {
	drawmode_none       =   0,
	drawmode_select     =   1, // Select objects
	// Map draw modes
	drawmode_draw       =   2, // Paste the brush
	drawmode_rectangle  =   4, // Pattern the brush in a rectangle
	drawmode_fill       =   8, // Fill with the brush
	drawmode_replaceall =  16, // Replace clicked block with brush
	// Object draw modes
	drawmode_lasso      =  32, // Lasso box from bitmap layer
	drawmode_sprite     =  64, // Create a sprite
	drawmode_box        = 128, // Create a box
	drawmode_offset     = 256, // Set a box's offsets
	drawmode_path       = 512, // Create a path
} DrawMode;

// Draw attributes
typedef enum {
	drawattr_none       = 0,
	drawattr_grid       = 1, // Align to grid boundaries
	drawattr_showtext   = 2, // Show text comments
	drawattr_attributes = 4, // Show map attributes
} DrawAttributes;

// Draw commands
typedef enum {
	drawcmd_none				= 0x00000000,
	drawcmd_edit_cut			= 0x00000001,
	drawcmd_edit_copy			= 0x00000002,
	drawcmd_edit_paste			= 0x00000004,
	drawcmd_edit_delete			= 0x00000008,
	drawcmd_edit_selectall		= 0x00000010,
	drawcmd_flipx				= 0x00000020,
	drawcmd_flipy				= 0x00000040,
	drawcmd_align				= 0x00000080,
	drawcmd_move				= 0x00000100,
	drawcmd_purge				= 0x00000200,
	drawcmd_sort				= 0x00000400,
	drawcmd_increment			= 0x00000800,
	drawcmd_decrement			= 0x00001000,
	drawcmd_frame_properties	= 0x00002000,
	drawcmd_frame_insert		= 0x00004000,
	drawcmd_frame_delete		= 0x00008000,
	drawcmd_layer_properties	= 0x00010000,
	drawcmd_file_topalette		= 0x00020000,
} DrawCommands;

// ----------------------------------------------------------------------------
// Create a string suitable for showing the position and size of a selection
inline tstring FormatPositionSize(int x1, int y1, int x2, int y2)
{
	tstringstream S;
	S << "(" << x1 << "," << y1 << ") [" << x2 - y1 << "," << y2 - y1 << "]";
	return S.str();
}

// ----------------------------------------------------------------------------
// Create a string suitable for showing the position of a selection
inline tstring FormatPosition(int x1, int y1)
{
	tstringstream S;
	S << "(" << x1 << "," << y1 << ")";
	return S.str();
}

// ----------------------------------------------------------------------------
// The base class for layers
// It shows up as a test pattern if it is instantiated
class ChalkLayer
{
public:
	ChalkLayer(CChalkDoc& Doc);
	virtual ~ChalkLayer();

	// Implemented by derived classes
	virtual void Draw(ChalkDisplay& display, CRect rt);
	virtual CSize GetSize(); // Get size of this layer (0 = unlimited)
	virtual int GetNumFrames(); // Get number of animation frames in this layer
	virtual ChalkObject* GetSelectedObject(); // Returns the main selected object (focus)
	virtual ChalkPalette* GetPalette(); // Returns pointer to my palette object

	// Handle user interaction
	virtual bool OnLButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	virtual bool OnRButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	virtual bool OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	virtual bool OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	virtual bool OnMouseMove(CChalkView *pView, UINT nFlags, CPoint point);
	virtual bool OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	virtual bool OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	virtual bool OnTimer(CChalkView *pView, UINT nIDEvent);
	virtual bool OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt);
	virtual void EditCommand(DrawCommands cmd); // Perform a drawing command


	// Helper functions
	void SetSelectBox(CChalkView *pView, CRect rt);

public:
	CChalkDoc& m_Doc; // For access to Windows
	ChalkDocument& m_Document; // Shortcut to access to undo buffer/clipboard
	tstring m_TypeName; // The name of this type of layer (constant)
	ChalkObject* m_pBaseData; // Alias to my main data object (which can have a filename)

	// Class attributes
	bool m_Solid; // Is this a solid or a see-through layer?
	DrawMode m_DrawMode; // What action should I take on mouse clicks?
	DrawMode m_DrawModesSupported; // What actions can this type of layer take? (constant)
	DrawAttributes m_DrawAttributes; // What modes are active?
	DrawAttributes m_DrawAttributesSupported; // What modes do I support?
	DrawCommands m_DrawCommandsSupported; // Which drawing commands do I support?
	CPoint m_ClickPos; // Where did the click/drag start
	CPoint m_LastMovePos; // Where was the click/drag last processed?
	CRect m_SelectionBox; // For displaying selected area

	// Data attributes
	double m_RelativeX; // What ratio should we scroll left/right at?
	double m_RelativeY; // What ratio should we scroll up/down at?

	// Current data attributes
	LayerVisibility m_Visible; // Am I visible or active?
	int m_OffsetX; // What is the X offset at the top-left of the window?
	int m_OffsetY; // What is the Y offset at the top-left of the window?
	int m_CurrentFrame; // What is the current animation frame for this layer?
};

// ----------------------------------------------------------------------------
// This is a colour bar layer (e.g. a .COP file)
class ChalkLayerPaletteBars : public ChalkLayer
{
public:
	ChalkLayerPaletteBars(CChalkDoc& Doc);
	~ChalkLayerPaletteBars();
	void Draw(ChalkDisplay& display, CRect rt);
	CSize GetSize(); // Get size of this layer
	ChalkPalette* GetPalette(); // Get this layer's palette
	void EditCommand(DrawCommands cmd); // Perform a drawing command

public:
	ChalkPalette m_Data;
	ChalkPalette m_Brush;
};

// ----------------------------------------------------------------------------
// This is a bitmap picture layer (e.g. a .TGA or a .LBM file)
class ChalkLayerBitmap : public ChalkLayer
{
public:
	ChalkLayerBitmap(CChalkDoc& Doc);
	~ChalkLayerBitmap();
	void Draw(ChalkDisplay& display, CRect rt);
	CSize GetSize(); // Get size of this layer
	ChalkPalette* GetPalette(); // Get this layer's palette
	void EditCommand(DrawCommands cmd); // Perform a drawing command

	// Handle user interaction
	bool OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnMouseMove(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt);

	// User interface functions that do stuff
	void Paste(CPoint pt);
	void PasteRect(CRect rect);
	void CopyRect(CRect rect);
	void BrushAdd(int value);

public:
	ChalkBitmap m_Data;
	ChalkCharacter m_Brush;
};

// ----------------------------------------------------------------------------
// This is a map layer (e.g. a .MAP file)
class ChalkLayerMap : public ChalkLayer
{
public:
	ChalkLayerMap(CChalkDoc& Doc);
	~ChalkLayerMap();
	void Draw(ChalkDisplay& display, CRect rt);
	CSize GetSize(); // Get size of this layer
	ChalkPalette* GetPalette(); // Get this layer's palette
	int GetNumFrames(); // Get number of animation frames in this layer
	void EditCommand(DrawCommands cmd); // Perform a drawing command

	// Handle user interaction
	bool OnLButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnRButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnMouseMove(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt);

	// User interface functions that do stuff
	void Paste(CPoint pt);
	void PasteRect(CRect rect);
	void CopyRect(CRect rect);
	void BrushAdd(int value);
	void EditMapBlock(CPoint pt);
	void Replace(CPoint pt);

public:
	ChalkMapAnim m_Data;
	ChalkMap m_Brush;
};

// ----------------------------------------------------------------------------
// This is an animation layer (e.g. a .BOX or .FRK file)
class ChalkLayerAnimation : public ChalkLayer
{
public:
	ChalkLayerAnimation(CChalkDoc& Doc);
	~ChalkLayerAnimation();
	void Draw(ChalkDisplay& display, CRect rt);
	CSize GetSize(); // Get size of this layer
	int GetNumFrames(); // Get number of animation frames in this layer
	ChalkObject* GetSelectedObject(); // Returns the main selected object
	void EditCommand(DrawCommands cmd); // Perform a drawing command

	// Handle user interaction
	bool OnLButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnRButtonDblClk(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint point);	// true if click has been handled
	bool OnMouseMove(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint point);
	bool OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt);

	// Helper functions
	void CheckAlignToGrid(CPoint& point);
	void InvalidateSelected();
	void SelectionBox(CRect rect); // Select the objects within the rectangle
	void DeleteSelected();
	void MoveSelected(int xoffset, int yoffset);
	void AddToSelected(int offset);
	void BrushAdd(int offset);
	void FlipSelectedX();
	void FlipSelectedY();
	void NewSprite(CPoint point);
	void NewBox(CRect rect);
	void SetBoxOffsets(ChalkBox* pBox, int xoffset, int yoffset);

public:
	ChalkAnimation m_Data; // Persistant data
	ChalkSprite m_BrushSprite; // Default attributes for creating new sprites
	ChalkBox m_BrushBox; // Default attributes for creating new boxes
	ChalkPath m_BrushPath; // Default attributes for creating new paths
};
