// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkLayerBitmap.cpp
// CREATED:  21 Jun 2000 by Carl Muller
// MODIFIED:  4 Apr 2001 by Carl Muller
//
// Implements visible bitmap objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkDoc.h"
#include "chalkView.h"
#include "PropSheetBase.h"
#include "resource.h"

// ----------------------------------------------------------------------------
// ChalkLayerBitmap
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkLayerBitmap::ChalkLayerBitmap(CChalkDoc& Doc) :
	ChalkLayer(Doc)
{
	m_TypeName = _T("Bitmap");
	m_pBaseData = &m_Data;
	// What actions can this type of layer take?
	m_DrawModesSupported = DrawMode(drawmode_select | drawmode_draw |
		drawmode_rectangle /* ... | drawmode_replaceall | drawmode_lasso */ );
	m_DrawAttributesSupported = DrawAttributes(drawattr_grid); // What modes do I support?
	m_DrawCommandsSupported = drawcmd_layer_properties;
}

// ----------------------------------------------------------------------------
ChalkLayerBitmap::~ChalkLayerBitmap()
{
}

// ----------------------------------------------------------------------------
// Draw a bitmap
void ChalkLayerBitmap::Draw(ChalkDisplay& display, CRect rt)
{
	int srcwidth = m_Data.m_Character.m_Width;
	int srcheight = m_Data.m_Character.m_Height;
	int palettesize = m_Data.m_Palette.m_NumColours;
	if ((srcwidth < 1) || (srcheight < 1)) // Sanity check
	{
		ChalkLayer::Draw(display, rt); // Default to the test pattern if there was a problem
		return;
	}

	int bitmapwidth = display.GetBitmapSize().cx;
	PIXEL *pRow = display.m_pBits;
	ChalkColourIndex *pData = m_Data.m_Character.m_Data; // Raw data
	int zoommult = display.GetZoomMult();
	int zoomdiv = display.GetZoomDiv();
	int xloopcount = (rt.right - rt.left) / zoommult;
	int xloopextra = (rt.right - rt.left) % zoommult;

	// Display the bitmap
	if (rt.right > rt.left)
	{
		if ((palettesize < 1) || (palettesize > 256)) // Display true colour image (e.g. TGA)
		{
			for (int dsty = rt.top; dsty < rt.bottom; ++dsty)
			{
				int dstx = rt.left;
				int srcy = (dsty * zoomdiv / zoommult);
				int srcx = (dstx * zoomdiv / zoommult);
				if ((srcy >= srcheight) || (srcx >= srcwidth))
					break;
				ChalkColourIndex* pX = pData + srcx + srcwidth * srcy;

				PIXEL *pPixel = pRow;
				int n = xloopcount;
				while (n--)
				{
					PIXEL p = *pX;
					int m = zoommult;
					while (m--)
						*pPixel++ = p;
					srcx += zoomdiv;
					pX += zoomdiv;
					if (srcx >= srcwidth)
					{
						srcx = 0;
						pX -= srcwidth;
						break;
					}
				}
				if (xloopextra)
				{
					PIXEL p = *pX;
					int m = xloopextra;
					while (m--)
						*pPixel++ = p;
				}
				pRow += bitmapwidth;
			}
		}

		else // Paletted image
		{
			PIXEL *pPalette = m_Data.m_Palette.m_Data;

			for (int dsty = rt.top; dsty < rt.bottom; ++dsty)
			{
				int dstx = rt.left;
				int srcy = (dsty * zoomdiv / zoommult);
				int srcx = (dstx * zoomdiv / zoommult);
				if ((srcy >= srcheight) || (srcx >= srcwidth))
					break;
				ChalkColourIndex* pX = pData + srcx + srcwidth * srcy;

				PIXEL *pPixel = pRow;
				int n = xloopcount;
				while (n--)
				{
					PIXEL p = pPalette[*pX];
					int m = zoommult;
					while (m--)
						*pPixel++ = p;
					srcx += zoomdiv;
					pX += zoomdiv;
					if (srcx >= srcwidth)
					{
						srcx = 0;
						pX -= srcwidth;
						break;
					}
				}
				if (xloopextra)
				{
					PIXEL p = pPalette[*pX];
					int m = xloopextra;
					while (m--)
						*pPixel++ = p;
				}
				pRow += bitmapwidth;
			}
		}

		// Fill to the right
		int xfinish = srcwidth * zoommult / zoomdiv;
		int yfinish = srcheight * zoommult / zoomdiv;
		display.DrawBox(CRect(0, 0, 0, 0), CRect(xfinish, 0, rt.right, yfinish), COLOUR_DKGREY);
		// Fill to the bottom
		display.DrawBox(CRect(0, 0, 0, 0), CRect(0, yfinish, rt.right, rt.bottom), COLOUR_DKGREY);
	}

	// Draw the selection box
	if (!m_SelectionBox.IsRectEmpty())
	{
		int x1 = m_SelectionBox.left;
		int y1 = m_SelectionBox.top;
		int x2 = m_SelectionBox.right;
		int y2 = m_SelectionBox.bottom;
		display.DrawBox(CRect(x1, y1, x2, y1), CRect(0, 0, 0, 1), COLOUR_WHITE);
		display.DrawBox(CRect(x1, y2, x2, y2), CRect(0, -1, 0, 0), COLOUR_WHITE);
		display.DrawBox(CRect(x1, y1, x1, y2), CRect(0, 0, 1, 0), COLOUR_WHITE);
		display.DrawBox(CRect(x2, y1, x2, y2), CRect(-1, 0, 0, 0), COLOUR_WHITE);
	}
}

// ----------------------------------------------------------------------------
CSize ChalkLayerBitmap::GetSize()
{
	int width = m_Data.m_Character.m_Width;
	int height = m_Data.m_Character.m_Height;
	return CSize(width, height);
}

// ----------------------------------------------------------------------------
// Returns pointer to my palette object
ChalkPalette* ChalkLayerBitmap::GetPalette()
{
	return &m_Data.m_Palette;
}

// ----------------------------------------------------------------------------
bool ChalkLayerBitmap::OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = 1;
	int gy = 1;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_GridX / gx;
		smally = m_Data.m_GridY / gy;
		x = smallx * (pt.x / m_Data.m_GridX);
		y = smally * (pt.y / m_Data.m_GridY);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int x1, x2, y1, y2;
	CRect rt;

	switch (m_DrawMode) {

	// Copy an area
	case drawmode_select:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;
		SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
		return true;

	// Normal draw
	case drawmode_draw:
		m_Document.RememberBeginGroup(_T("Draw"));
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		Paste(m_ClickPos);
		m_Doc.m_StatusPosition = FormatPosition(m_ClickPos.x, m_ClickPos.y);
		if ((bx > 0) && (by > 0))
		{
			// Show paste box (this updates the view)
			x1 = x;
			x2 = x1 + bx;
			y1 = y;
			y2 = y1 + by;
			SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		}
		return true;

	// Rectangle draw
	case drawmode_rectangle:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		return true;
	};
	return false;
}


// ----------------------------------------------------------------------------
bool ChalkLayerBitmap::OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = 1;
	int gy = 1;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_GridX / gx;
		smally = m_Data.m_GridY / gy;
		x = smallx * (pt.x / m_Data.m_GridX);
		y = smally * (pt.y / m_Data.m_GridY);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int x1, x2, y1, y2;
	CRect rt;

	switch (m_DrawMode) {

	// Copy an area
	case drawmode_select:
	case drawmode_draw:
	case drawmode_rectangle:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;
		SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerBitmap::OnMouseMove(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = 1;
	int gy = 1;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_GridX / gx;
		smally = m_Data.m_GridY / gy;
		x = smallx * (pt.x / m_Data.m_GridX);
		y = smally * (pt.y / m_Data.m_GridY);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int x1, x2, y1, y2;
	CRect rt;

	if (nFlags & MK_LBUTTON)
	{
		switch (m_DrawMode) {

		// Normal draw
		case drawmode_draw:
			pView->AutoScroll(pt);
			// Handle drag paste
			if ( (x != m_ClickPos.x) || (y != m_ClickPos.y) )
			{
				m_ClickPos.x = x;
				m_ClickPos.y = y;
				Paste(m_ClickPos);
				m_Doc.m_StatusPosition = FormatPosition(m_ClickPos.x, m_ClickPos.y);
				if ((bx > 0) && (by > 0))
				{
					// Show paste box (this updates the view)
					x1 = x;
					x2 = x1 + bx;
					y1 = y;
					y2 = y1 + by;
					SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
				}
			}
			return true;

		// Select mode
		case drawmode_select:
		// Rectangle draw
		case drawmode_rectangle:
			pView->AutoScroll(pt);

			// Show selection box
			x1 = min(m_ClickPos.x, x);
			x2 = max(m_ClickPos.x, x) + smallx;
			y1 = min(m_ClickPos.y, y);
			y2 = max(m_ClickPos.y, y) + smally;
			SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			return true;
		}
	}
	else if (nFlags & MK_RBUTTON)
	{
		switch (m_DrawMode) {
		// Select some blocks
		case drawmode_draw:
		case drawmode_select:
		case drawmode_rectangle:
			pView->AutoScroll(pt);

			// Show selection box
			x1 = min(m_ClickPos.x, x);
			x2 = max(m_ClickPos.x, x) + smallx;
			y1 = min(m_ClickPos.y, y);
			y2 = max(m_ClickPos.y, y) + smally;
			SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			return true;
		};
	}
	else if ((pt.x < 0) || (pt.y < 0)) // Moving mouse outside of area
	{
		m_Doc.m_StatusPosition = FORMAT_SPACE;
		SetSelectBox(pView, CRect(0, 0, 0, 0));
	}
	else // Moving mouse around without clicking
	{
		m_Doc.m_StatusPosition = FormatPosition(x, y);

		switch (m_DrawMode) {

		// Normal draw
		case drawmode_draw:
			if ((bx > 0) && (by > 0))
			{
				// Show paste box
				x1 = x;
				x2 = x1 + bx;
				y1 = y;
				y2 = y1 + by;
				SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
			}
		};
	}
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerBitmap::OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = 1;
	int gy = 1;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_GridX / gx;
		smally = m_Data.m_GridY / gy;
		x = smallx * (pt.x / m_Data.m_GridX);
		y = smally * (pt.y / m_Data.m_GridY);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	SetSelectBox(pView, CRect(0,0,0,0));
	int x1, x2, y1, y2;

	switch (m_DrawMode) {

	// select mode
	case drawmode_select:
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;

		// Set solid rectangle
		CopyRect(CRect(x1, y1, x2, y2));
		return true;

	// Normal draw
	case drawmode_draw:
		// Normal paste
		m_Document.RememberEndGroup();
		return true;

	// Rectangle draw
	case drawmode_rectangle:
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;

		// Set solid rectangle
		PasteRect(CRect(x1, y1, x2, y2));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerBitmap::OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = 1;
	int gy = 1;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_GridX / gx;
		smally = m_Data.m_GridY / gy;
		x = smallx * (pt.x / m_Data.m_GridX);
		y = smally * (pt.y / m_Data.m_GridY);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	SetSelectBox(pView, CRect(0,0,0,0));
	int x1, x2, y1, y2;

	switch (m_DrawMode) {

	// select mode
	case drawmode_select:
	case drawmode_draw:
	case drawmode_rectangle:
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;

		// Set solid rectangle
		CopyRect(CRect(x1, y1, x2, y2));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerBitmap::OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt)
{
	switch (nChar) {
	case '+': BrushAdd(nRepCnt); break;
	case '-': BrushAdd(-int(nRepCnt)); break;
	default: break;
	};
	return false;
}

// ----------------------------------------------------------------------------
// Paste a copy of the brush down at a point
void ChalkLayerBitmap::Paste(CPoint pt)
{
	ChalkCharacter& chr = m_Data.m_Character;
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int mx = chr.m_Width;
	int my = chr.m_Height;
	int x1 = max(pt.x, 0);
	int y1 = max(pt.y, 0);
	int x2 = min(x1 + bx, mx);
	int y2 = min(y1 + by, my);
	if ((x2 > x1) && (y2 > y1))
	{
		int x, y;
		for (y = y1; y < y2; ++y)
		{
			const ChalkColourIndex* pSrc = &m_Brush.m_Data[bx * (y - y1)];
			ChalkColourIndex* pDst = &chr.m_Data[mx * y + x1];
			for (x = x1; x < x2; ++x)
				*pDst++ = *pSrc++;
		}
		// Signal that the data has changed
		m_Doc.SetModifiedFlag();
		m_Doc.InvalidateViewsRect(CRect(x1, y1, x2, y2), CRect(0,0,0,0));
	}
}

// ----------------------------------------------------------------------------
// Fill a rectangle with the brush (zero aligned)
void ChalkLayerBitmap::PasteRect(CRect rect)
{
	ChalkCharacter& chr = m_Data.m_Character;
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int mx = chr.m_Width;
	int my = chr.m_Height;
	int x1 = max(rect.left, 0);
	int x2 = min(rect.right, mx);
	int y1 = max(rect.top, 0);
	int y2 = min(rect.bottom, my);
	if ((x2 > x1) && (y2 > y1) && (bx > 0) && (by > 0))
	{
		int x, y;
		for (y = y1; y < y2; ++y)
		{
			ChalkColourIndex* pDst = &chr.m_Data[mx * y + x1];
			for (x = x1; x < x2; ++x)
				*pDst++ = m_Brush.m_Data[(y % by) * bx + (x % bx)];
		}
		// Signal that the data has changed
		m_Doc.SetModifiedFlag();
		m_Doc.InvalidateViewsRect(CRect(x1, y1, x2, y2), CRect(0,0,0,0));
	}
}

// ----------------------------------------------------------------------------
// Copy the data into the brush
void ChalkLayerBitmap::CopyRect(CRect rect)
{
	ChalkCharacter& chr = m_Data.m_Character;
	int np = chr.m_NumPlanes;
	int mx = chr.m_Width;
	int my = chr.m_Height;
	int x1 = max(rect.left, 0);
	int x2 = min(rect.right, mx);
	int y1 = max(rect.top, 0);
	int y2 = min(rect.bottom, my);
	if ((x2 > x1) && (y2 > y1))
	{
		m_Brush.SetSize(x2 - x1, y2 - y1, np);
		ChalkColourIndex* pDst = m_Brush.m_Data;
		int x, y;
		for (y = y1; y < y2; ++y)
		{
			const ChalkColourIndex* pSrc = &chr.m_Data[mx * y + x1];
			for (x = x1; x < x2; ++x)
				*pDst++ = *pSrc++;
		}
	}
	else
		m_Brush.SetSize(0, 0, np);
}

// ----------------------------------------------------------------------------
// Add a value to the brush
void ChalkLayerBitmap::BrushAdd(int value)
{
	int palettesize = m_Data.m_Palette.m_NumColours;
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int x, y;
	ChalkColourIndex* pSrc = m_Brush.m_Data;
	if ((palettesize < 1) || (palettesize > 256)) // Handle true colour image (e.g. TGA)
	{
		int r, g, b;
		for (y = 0; y < by; ++y)
			for (x = 0; x < bx; ++x, ++pSrc)
			{
				r = (*pSrc & 0xff) + value;
				r = max(0, min(255, r));
				g = (*pSrc >> 8 & 0xff) + value;
				g = max(0, min(255, r));
				b = (*pSrc >> 16 & 0xff) + value;
				b = max(0, min(255, r));
				*pSrc = r | (g << 8) | (b << 16);
			}
	}
	else
	{
		for (y = 0; y < by; ++y)
			for (x = 0; x < bx; ++x, ++pSrc)
				*pSrc = max(0, min(255, *pSrc + value));
	}
}


// ----------------------------------------------------------------------------
// Perform a drawing command
void ChalkLayerBitmap::EditCommand(DrawCommands cmd)
{
	switch (cmd) {
	case drawcmd_layer_properties:
		{
			CPropertySheetBitmap dlg(this, &m_Doc);
			dlg.DoModal();
		}
		break;

	default:
		AfxMessageBox(IDS_ERROR_NOTIMPLEMENTED);
		break;
	};
}
