// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkLayer.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 25 Oct 2001 by Carl Muller
//
// Implements the user interface for the map layer (.MAP files)
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkDoc.h"
#include "chalkView.h"
#include "PropSheetBase.h"
#include "resource.h"

// ----------------------------------------------------------------------------
// ChalkLayerMap
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkLayerMap::ChalkLayerMap(CChalkDoc& Doc) :
	ChalkLayer(Doc)
{
	m_TypeName = _T("Map");
	m_pBaseData = &m_Data;
	// What actions can this type of layer take?
	m_DrawModesSupported = DrawMode(drawmode_select | drawmode_draw |
		drawmode_rectangle /* ... | drawmode_replaceall */ );
	m_DrawAttributesSupported = DrawAttributes(drawattr_grid | drawattr_attributes); // What modes do I support?
	m_DrawCommandsSupported = DrawCommands(drawcmd_flipx | drawcmd_flipy |
		drawcmd_increment | drawcmd_decrement | drawcmd_layer_properties
		| drawcmd_frame_insert | drawcmd_frame_delete );
}

// ----------------------------------------------------------------------------
ChalkLayerMap::~ChalkLayerMap()
{
}

// ----------------------------------------------------------------------------
// Draw a map
void ChalkLayerMap::Draw(ChalkDisplay& display, CRect rt)
{
	if (m_CurrentFrame >= m_Data.m_Data.size())
		return;
	if (!m_Data.m_pBlocks)
		return;
	ChalkMap& map = m_Data.m_Data[m_CurrentFrame];

	// Draw all the blocks
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int mapwidth = map.m_Width;
	int mapheight = map.m_Height;
	int width = mapwidth * gx;
	int height = mapheight * gy;
	int zoommult = display.GetZoomMult();
	int zoomdiv = display.GetZoomDiv();
	ChalkMapBlock attr;
	attr.m_PaletteOffset = 0;
	attr.m_Special = SPECIAL_TRANSPARENT;
	
	if ((rt.right > rt.left) && (gx > 0) && (gy > 0))
	{
		int xstart = ((rt.left * zoomdiv / zoommult) / gx) * gx;
		int ystart = ((rt.top * zoomdiv / zoommult) / gy) * gy;
		int xend = rt.right * zoomdiv / zoommult;
		int yend = rt.bottom * zoomdiv / zoommult;
		for (int y = ystart; y < yend; y += gy)
		{
			int xx = ((xstart) / gx); // Map column of block
			int yy = ((y) / gy); // Map row of block
			if (xx >= mapwidth || yy >= mapheight)
				break;
			ChalkMapBlock* pBlocks = &map.m_Data[mapwidth * yy];

			for (int x = xstart; (x < xend) && (xx < mapwidth); x += gx, ++xx)
			{
				// Display main block
				m_Data.m_pBlocks->DrawBlock(display, rt, CPoint(x, y),
					pBlocks[xx], m_Data.m_Palette);

				// Display attributes
				if (m_DrawAttributes & drawattr_attributes)
					if (m_Data.m_pAttrBlocks)
					{
						attr.m_BlockIndex = pBlocks[xx].m_Attribute;
						m_Data.m_pAttrBlocks->DrawBlock(display, rt, CPoint(x, y),
							attr, m_Data.m_AttrPalette);
					}
				// Is the block selected? (e.g. by a find operation)
				//...

			}
		}

		// Fill to the right
		int xfinish = width * zoommult / zoomdiv;
		int yfinish = height * zoommult / zoomdiv;
		display.DrawBox(CRect(0, 0, 0, 0), CRect(xfinish, 0, rt.right, yfinish), COLOUR_DKGREY);
		// Fill to the bottom
		display.DrawBox(CRect(0, 0, 0, 0), CRect(0, yfinish, rt.right, rt.bottom), COLOUR_DKGREY);
	}

	// Draw the selection box
	if (!m_SelectionBox.IsRectEmpty())
	{
		int x1 = m_SelectionBox.left;
		int y1 = m_SelectionBox.top;
		int x2 = m_SelectionBox.right;
		int y2 = m_SelectionBox.bottom;
		display.DrawBox(CRect(x1, y1, x2, y1), CRect(0, 0, 0, 1), COLOUR_WHITE);
		display.DrawBox(CRect(x1, y2, x2, y2), CRect(0, -1, 0, 0), COLOUR_WHITE);
		display.DrawBox(CRect(x1, y1, x1, y2), CRect(0, 0, 1, 0), COLOUR_WHITE);
		display.DrawBox(CRect(x2, y1, x2, y2), CRect(-1, 0, 0, 0), COLOUR_WHITE);
	}
}

// ----------------------------------------------------------------------------
CSize ChalkLayerMap::GetSize()
{
	if (m_CurrentFrame >= m_Data.m_Data.size())
		return CSize(0, 0);
	ChalkMap& map = m_Data.m_Data[m_CurrentFrame];

	// Draw all the blocks
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int width = map.m_Width * gx;
	int height = map.m_Height * gy;
	return CSize(width, height);
}

// ----------------------------------------------------------------------------
// Get number of animation frames in this layer
int ChalkLayerMap::GetNumFrames()
{
	return m_Data.m_Data.size();
}

// ----------------------------------------------------------------------------
// Returns pointer to my palette object
ChalkPalette* ChalkLayerMap::GetPalette()
{
	if (m_Data.m_pBlocks)
		return m_Data.m_pBlocks->GetPalette();
	return NULL;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnLButtonDown(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_Grid2X / gx;
		smally = m_Data.m_Grid2Y / gy;
		x = smallx * (pt.x / m_Data.m_Grid2X);
		y = smally * (pt.y / m_Data.m_Grid2Y);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	int x1, x2, y1, y2;
	CRect rt;

	switch (m_DrawMode) {

	// Copy an area
	case drawmode_select:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;
		SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
		return true;

	// Normal draw
	case drawmode_draw:
		m_Document.RememberBeginGroup(_T("Draw"));
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		Paste(m_ClickPos);
		m_Doc.m_StatusPosition = FormatPosition(m_ClickPos.x, m_ClickPos.y);
		if ((bx > 0) && (by > 0))
		{
			// Show paste box
			x1 = x;
			x2 = x1 + bx;
			y1 = y;
			y2 = y1 + by;
			SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		}
		return true;

	// Rectangle draw
	case drawmode_rectangle:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnLButtonDblClk(CChalkView *pView, UINT nFlags, CPoint pt)
{
	if (m_DrawMode == drawmode_select)
		return OnRButtonDblClk(pView, nFlags, pt);
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnRButtonDblClk(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int x = pt.x / gx;
	int y = pt.y / gy;
	int x1, x2, y1, y2;
	CRect rt;

	switch (m_DrawMode) {

	// Edit the map block
	case drawmode_select:
	case drawmode_draw:
	case drawmode_rectangle:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + 1; // Single block
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + 1;
		SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		m_Doc.m_StatusPosition = FormatPosition(m_ClickPos.x, m_ClickPos.y);
		EditMapBlock(CPoint(x1, y1));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnRButtonDown(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_Grid2X / gx;
		smally = m_Data.m_Grid2Y / gy;
		x = smallx * (pt.x / m_Data.m_Grid2X);
		y = smally * (pt.y / m_Data.m_Grid2Y);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	int x1, x2, y1, y2;
	CRect rt;

	switch (m_DrawMode) {

	// Copy an area
	case drawmode_select:
	case drawmode_draw:
	case drawmode_rectangle:
		m_ClickPos.x = x;
		m_ClickPos.y = y;
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;
		SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
		m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnMouseMove(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_Grid2X / gx;
		smally = m_Data.m_Grid2Y / gy;
		x = smallx * (pt.x / m_Data.m_Grid2X);
		y = smally * (pt.y / m_Data.m_Grid2Y);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	int x1, x2, y1, y2;
	CRect rt;

	if (nFlags & MK_LBUTTON)
	{
		switch (m_DrawMode) {

		// Normal draw
		case drawmode_draw:
			pView->AutoScroll(pt);
			// Handle drag paste
			if ( (x != m_ClickPos.x) || (y != m_ClickPos.y) )
			{
				m_ClickPos.x = x;
				m_ClickPos.y = y;
				Paste(m_ClickPos);
				m_Doc.m_StatusPosition = FormatPosition(m_ClickPos.x, m_ClickPos.y);
				if ((bx > 0) && (by > 0))
				{
					// Show paste box
					x1 = x;
					x2 = x1 + bx;
					y1 = y;
					y2 = y1 + by;
					SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
				}
			}
			return true;

		// Select mode
		case drawmode_select:
		// Rectangle draw
		case drawmode_rectangle:
			pView->AutoScroll(pt);

			// Show selection box
			x1 = min(m_ClickPos.x, x);
			x2 = max(m_ClickPos.x, x) + smallx;
			y1 = min(m_ClickPos.y, y);
			y2 = max(m_ClickPos.y, y) + smally;
			SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			return true;
		}
	}
	else if (nFlags & MK_RBUTTON)
	{
		switch (m_DrawMode) {
		// Select some blocks
		case drawmode_draw:
		case drawmode_select:
		case drawmode_rectangle:
			pView->AutoScroll(pt);

			// Show selection box
			x1 = min(m_ClickPos.x, x);
			x2 = max(m_ClickPos.x, x) + smallx;
			y1 = min(m_ClickPos.y, y);
			y2 = max(m_ClickPos.y, y) + smally;
			SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
			m_Doc.m_StatusPosition = FormatPositionSize(x1, y1, x2, y2);
			return true;
		};
	}
	else if ((pt.x < 0) || (pt.y < 0)) // Moving mouse outside of area
	{
		m_Doc.m_StatusPosition = FORMAT_SPACE;
		SetSelectBox(pView, CRect(0, 0, 0, 0));
	}
	else // Moving mouse around without clicking
	{
		m_Doc.m_StatusPosition = FormatPosition(x, y);

		switch (m_DrawMode) {

		// Normal draw
		case drawmode_draw:
			if ((bx > 0) && (by > 0))
			{
				// Show paste box
				x1 = x;
				x2 = x1 + bx;
				y1 = y;
				y2 = y1 + by;
				SetSelectBox(pView, CRect(x1 * gx, y1 * gy, x2 * gx, y2 * gy));
			}
		};
	}
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnLButtonUp(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_Grid2X / gx;
		smally = m_Data.m_Grid2Y / gy;
		x = smallx * (pt.x / m_Data.m_Grid2X);
		y = smally * (pt.y / m_Data.m_Grid2Y);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	SetSelectBox(pView, CRect(0,0,0,0));
	int x1, x2, y1, y2;

	switch (m_DrawMode) {

	// select mode
	case drawmode_select:
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;

		// Set solid rectangle
		CopyRect(CRect(x1, y1, x2, y2));
		return true;

	// Normal draw
	case drawmode_draw:
		// Normal paste
		m_Document.RememberEndGroup();
		return true;

	// Rectangle draw
	case drawmode_rectangle:
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;

		// Set solid rectangle
		PasteRect(CRect(x1, y1, x2, y2));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnRButtonUp(CChalkView *pView, UINT nFlags, CPoint pt)
{
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int x, y, smallx, smally;
	if (m_DrawAttributes & drawattr_grid)
	{
		smallx = m_Data.m_Grid2X / gx;
		smally = m_Data.m_Grid2Y / gy;
		x = smallx * (pt.x / m_Data.m_Grid2X);
		y = smally * (pt.y / m_Data.m_Grid2Y);
	}
	else
	{
		x = pt.x / gx;
		y = pt.y / gy;
		smallx = smally = 1;
	}
	SetSelectBox(pView, CRect(0,0,0,0));
	int x1, x2, y1, y2;

	switch (m_DrawMode) {

	// select mode
	case drawmode_select:
	case drawmode_draw:
	case drawmode_rectangle:
		x1 = min(m_ClickPos.x, x);
		x2 = max(m_ClickPos.x, x) + smallx;
		y1 = min(m_ClickPos.y, y);
		y2 = max(m_ClickPos.y, y) + smally;

		// Set solid rectangle
		CopyRect(CRect(x1, y1, x2, y2));
		return true;
	};
	return false;
}

// ----------------------------------------------------------------------------
bool ChalkLayerMap::OnChar(CChalkView *pView, UINT nChar, UINT nRepCnt)
{
	switch (nChar) {
	case '+': BrushAdd(nRepCnt); return true;
	case '-': BrushAdd(-int(nRepCnt)); return true;
	case 'x': EditCommand(drawcmd_flipx); return true;
	case 'y': EditCommand(drawcmd_flipy); return true;
	default: break;
	};
	return false;
}

// ----------------------------------------------------------------------------
// Perform a drawing command
void ChalkLayerMap::EditCommand(DrawCommands cmd)
{
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int x, y, pos1, pos2;
	ChalkMapBlock temp;

	switch (cmd) {

	case drawcmd_flipx:
		// Flip the brush horizontally
		for (y = 0; y < by; ++y)
		{
			for (x = 0; x < bx; ++x) // Flip the block attributes
			{
				pos1 = x + bx * y;
				m_Brush.m_Data[pos1].m_Special =
					m_Brush.m_Data[pos1].m_Special ^ SPECIAL_XFLIP;
			}
			for (x = 0; x < bx / 2; ++x) // Swap the blocks around
			{
				pos1 = x + bx * y;
				pos2 = (bx - x - 1) + bx * y;
				temp = m_Brush.m_Data[pos2];
				m_Brush.m_Data[pos2] = m_Brush.m_Data[pos1];
				m_Brush.m_Data[pos1] = temp;
			}
		}
		break;

	case drawcmd_flipy:
		// Flip the brush vertically
		for (x = 0; x < bx; ++x)
		{
			for (y = 0; y < by; ++y) // Flip the block attributes
			{
				pos1 = x + bx * y;
				m_Brush.m_Data[pos1].m_Special =
					m_Brush.m_Data[pos1].m_Special ^ SPECIAL_YFLIP;
			}
			for (y = 0; y < by / 2; ++y) // Swap the blocks around
			{
				pos1 = x + bx * y;
				pos2 = x + bx * (by - y - 1);
				temp = m_Brush.m_Data[pos2];
				m_Brush.m_Data[pos2] = m_Brush.m_Data[pos1];
				m_Brush.m_Data[pos1] = temp;
			}
		}
		break;

	case drawcmd_increment:
		// Add one to the brush
		BrushAdd(1);
		break;

	case drawcmd_decrement:
		// Subtract one from the brush
		BrushAdd(-1);
		break;

	case drawcmd_layer_properties:
		{
			CPropertySheetMap dlg(this, &m_Doc);
			dlg.DoModal();
		}
		break;

	case drawcmd_frame_insert:
		// Insert a frame
		m_Data.m_Data.insert(&m_Data.m_Data.at(m_CurrentFrame), m_Data.m_Data[m_CurrentFrame]);
		m_Doc.ChangeFrame();
		break;

	case drawcmd_frame_delete:
		// Delete this frame
		m_Data.m_Data.erase(&m_Data.m_Data.at(m_CurrentFrame));
		if (m_CurrentFrame > 0 && m_CurrentFrame > m_Data.m_Data.size())
			m_CurrentFrame--;
		m_Doc.ChangeFrame();
		break;

	default:
		break;
	};
}


// ----------------------------------------------------------------------------
// Add a value to the brush
void ChalkLayerMap::BrushAdd(int value)
{
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int x, y;
	ChalkMapBlock* pSrc = m_Brush.m_Data;
	for (y = 0; y < by; ++y)
	{
		if (m_DrawAttributes & drawattr_attributes)
		{
			for (x = 0; x < bx; ++x)
			{
				pSrc->m_Attribute = max(0, pSrc->m_Attribute + value);
				pSrc++;
			}
		}
		else
		{
			for (x = 0; x < bx; ++x)
			{
				pSrc->m_BlockIndex = max(0, pSrc->m_BlockIndex + value);
				pSrc++;
			}
		}
	}
}


// ----------------------------------------------------------------------------
// Copy the data into the brush
void ChalkLayerMap::CopyRect(CRect rect)
{
	if (m_CurrentFrame >= m_Data.m_Data.size())
		return;
	ChalkMap& map = m_Data.m_Data[m_CurrentFrame];
	int mx = map.m_Width;
	int my = map.m_Height;
	int x1 = max(rect.left, 0);
	int x2 = min(rect.right, mx);
	int y1 = max(rect.top, 0);
	int y2 = min(rect.bottom, my);
	if ((x2 > x1) && (y2 > y1))
	{
		m_Brush.SetSize(x2 - x1, y2 - y1);
		ChalkMapBlock* pDst = m_Brush.m_Data;
		int x, y;
		for (y = y1; y < y2; ++y)
		{
			const ChalkMapBlock* pSrc = &map.m_Data[mx * y + x1];
			for (x = x1; x < x2; ++x)
				*pDst++ = *pSrc++;
		}
	}
	else
		m_Brush.SetSize(0, 0);
}


// ----------------------------------------------------------------------------
// Edit the data of a specified map block
void ChalkLayerMap::EditMapBlock(CPoint pt)
{
	if (m_CurrentFrame >= m_Data.m_Data.size())
		return;
	ChalkMap& map = m_Data.m_Data[m_CurrentFrame];
	int mx = map.m_Width;
	int my = map.m_Height;
	if ((pt.x >= 0) && (pt.x < mx) && (pt.y >= 0) && (pt.y < my))
	{
		ChalkMapBlock* pSrc = &map.m_Data[mx * pt.y + pt.x];
		CPropertySheetMapBlock dlg(pSrc, &m_Doc);
		dlg.DoModal();
	}
}


// ----------------------------------------------------------------------------
// Commands that actually change the (persistant) data
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Paste a copy of the brush down at a point
void ChalkLayerMap::Paste(CPoint pt)
{
	if (m_CurrentFrame >= m_Data.m_Data.size())
		return;
	ChalkMap& map = m_Data.m_Data[m_CurrentFrame];
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int mx = map.m_Width;
	int my = map.m_Height;
	int x1 = max(pt.x, 0);
	int y1 = max(pt.y, 0);
	int x2 = min(x1 + bx, mx);
	int y2 = min(y1 + by, my);
	if ((x2 > x1) && (y2 > y1))
	{
		int x, y;
		for (y = y1; y < y2; ++y)
		{
			const ChalkMapBlock* pSrc = &m_Brush.m_Data[bx * (y - y1)];
			ChalkMapBlock* pDst = &map.m_Data[mx * y + x1];
			if (m_DrawAttributes & drawattr_attributes)
				for (x = x1; x < x2; ++x)
					(pDst++)->m_Attribute = (pSrc++)->m_Attribute;
			else
				for (x = x1; x < x2; ++x)
					*pDst++ = *pSrc++;
		}
		// Signal that the data has changed
		m_Doc.SetModifiedFlag();
		m_Doc.InvalidateViewsRect(CRect(x1 * gx, y1 + gy, x2 * gx, y2 * gy), CRect(0,0,0,0));
	}
}

// ----------------------------------------------------------------------------
// Fill a rectangle with the brush (zero aligned)
void ChalkLayerMap::PasteRect(CRect rect)
{
	if (m_CurrentFrame >= m_Data.m_Data.size())
		return;
	ChalkMap& map = m_Data.m_Data[m_CurrentFrame];
	int gx = m_Data.m_Grid1X;
	int gy = m_Data.m_Grid1Y;
	int bx = m_Brush.m_Width;
	int by = m_Brush.m_Height;
	int mx = map.m_Width;
	int my = map.m_Height;
	int x1 = max(rect.left, 0);
	int x2 = min(rect.right, mx);
	int y1 = max(rect.top, 0);
	int y2 = min(rect.bottom, my);
	if ((x2 > x1) && (y2 > y1) && (bx > 0) && (by > 0))
	{
		int x, y;
		for (y = y1; y < y2; ++y)
		{
			ChalkMapBlock* pDst = &map.m_Data[mx * y + x1];
			if (m_DrawAttributes & drawattr_attributes)
				for (x = x1; x < x2; ++x)
					(pDst++)->m_Attribute = m_Brush.m_Data[(y % by) * bx + (x % bx)].m_Attribute;
			else
				for (x = x1; x < x2; ++x)
					*pDst++ = m_Brush.m_Data[(y % by) * bx + (x % bx)];
		}
		// Signal that the data has changed
		m_Doc.SetModifiedFlag();
		m_Doc.InvalidateViewsRect(CRect(x1 * gx, y1 + gy, x2 * gx, y2 * gy), CRect(0,0,0,0));
	}
}

