// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkObj.h
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED:  5 May 2000 by Carl Muller
//
// This defines the base game object classes
// ----------------------------------------------------------------------------

#include "chalkstream.h"
#include "memorybuffer.h"
#include "undobuffer.h"

// ----------------------------------------------------------------------------
// Global functions
inline UINT AFXAPI HashKey(GUID& key)
{ return (key.Data1 >> 16) ^ key.Data1 ^ key.Data2; }

class ChalkObject;

// ----------------------------------------------------------------------------
// Class used to support undo buffer
class ChalkDocument
{
public:
	ChalkDocument();

	// Support history palette (called by ChalkObject derived classes or their friends)
	void RememberDelete(ChalkObject& object); // Call this before deleting an object
	void RememberAdd(ChalkObject& object); // Call this after creating an object
	void RememberBegin(ChalkObject& object, LPCTSTR operation); // Call this before changing an object
	void RememberEnd(ChalkObject& object); // Call this after changing an object
	void RememberBeginGroup(LPCTSTR operation);
	void RememberEndGroup();

	// Perform actions
	void PerformUndo();
	void PerformRedo();

private:
	// Raw storage
	UndoBuffer m_BeforeBuffer;
	UndoBuffer m_AfterBuffer;
	UndoBuffer m_UndoBuffer;
	// Streamed storage
	ChalkStreamBinary m_Before; // Contains last remembered object
	ChalkStreamBinary m_After; // Contains current object
	ChalkStreamBinary m_Undo; // Contains list of buffer deltas
};


// ----------------------------------------------------------------------------
// The base class for persistant objects
class ChalkObject
{
public:
	ChalkObject(GUID id, GUID typeguid);
	const ChalkObject& operator= (const ChalkObject& s); // Assignment operator
	virtual ~ChalkObject();

	// Implemented by derived classes
	virtual HRESULT Write(ChalkStream& os) const; // XML
	virtual HRESULT Read(ChalkStream& is); // XML
	virtual HRESULT WriteFile(const CFileName& filename) const; // Legacy
	virtual HRESULT ReadFile(const CFileName& filename); // Legacy

	// Help derived classes read/write XML files
	HRESULT WriteXML(const CFileName& filename) const;
	HRESULT ReadXML(const CFileName& filename);

public: // Data members
	GUID m_Id;   // ID for this particular object
	GUID m_TypeId; // ID for this type of object
	COleDateTime m_DateCreated;    // Date/time the GUID was allocated
	tstring m_UserCreated; // Domain\Username of user who created this object
	tstring m_ComputerCreated; // Computer name of user who created this object
	COleDateTime m_DateModified;   // Date/time the object was altered
	tstring m_UserModified; // Domain\Username of user who last modified this object
	tstring m_ComputerModified; // Computer name of user who last modified this object

	// Persistant data members
	wstring m_Comment;  // User specified comment (UNICODE)
	wstring m_Script;   // User specified script (UNICODE)
	mutable CFileName m_FileName; // Filename of object, if it is stored in a file

	// Current data members
	bool m_Dirty; // Have I been changed since I was last saved?
	bool m_Selected; // Am I currently selected by the user?
};

bool operator< (GUID, GUID);

// ----------------------------------------------------------------------------
// This is a repository for all objects
class ChalkObjectMap
{
public:
	ChalkObjectMap();
	~ChalkObjectMap();
	void AddObject(ChalkObject* object);
	ChalkObject* GetObject(GUID id) const;
	void AddType(GUID id, LPCTSTR chalktypename);
	tstring GetTypeName(GUID id) const;

public: // Data members
	map<GUID, ChalkObject*> m_Data;
};

extern ChalkObjectMap g_ChalkObjectMap;

// ----------------------------------------------------------------------------
// This contains information about the class of an object
class ChalkType : public ChalkObject
{
public:
	ChalkType(GUID typeguid, LPCTSTR chalktypename);
	~ChalkType();
	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	tstring m_TypeName; // Human readable name for this type
};


// ----------------------------------------------------------------------------
// This contains information about a computer system
class ChalkSystem : public ChalkObject
{
public:
	ChalkSystem(GUID systemid, LPCTSTR systemname, int bitsperpalette,
		int blockmask, int palettemask, int paletteshift,
		int yflipmask, int xflipmask, int prioritymask);
	~ChalkSystem();
	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	tstring m_SystemName; // Name of this system (in English)
	int m_BitsPerPalette; // Bits available for specifying RGB colours (or 0 for fixed palette)
	int m_BlockMask;      // Mask for character index
	int m_PaletteMask ;   // Mask for palette colour
	int m_PaletteShift;   // Shift value (+ or -) for palette index
	int m_YflipMask;      // Bit value for vertical flip for characters
	int m_XflipMask;      // Bit value for horizontal flip for characters
	int m_PriorityMask;   // Bit value (or mask) for priority
};
