// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkObjects.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 25 Oct 2001 by Carl Muller
//
// Implements miscellaneous game objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
// ChalkCharacter
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkCharacter::ChalkCharacter() : ChalkObject(GUID_VOID, GUID_CHARACTER),
	m_Width(0),
	m_Height(0),
	m_OffsetX(0),
	m_OffsetY(0),
	m_NumPlanes(8),
	m_Data(NULL)
{
}

// ----------------------------------------------------------------------------
// copy constructor
ChalkCharacter::ChalkCharacter(const ChalkCharacter &that)
	: ChalkObject(that.m_Id, GUID_CHARACTER),
	m_Width(that.m_Width),
	m_Height(that.m_Height),
	m_OffsetX(that.m_OffsetX),
	m_OffsetY(that.m_OffsetY),
	m_NumPlanes(that.m_NumPlanes),
	m_Data(NULL)
{
	if (m_Width * m_Height > 0)
	{
		m_Data = new ChalkColourIndex[m_Width * m_Height];
		memcpy(m_Data, that.m_Data, m_Width * m_Height * sizeof(ChalkColourIndex));
	}
}

// ----------------------------------------------------------------------------
// assignment operator
const ChalkCharacter& ChalkCharacter::operator= (const ChalkCharacter& that)
{
	if (this != &that)
	{
		if (m_Data)
			delete [] m_Data;
		m_Data = NULL;

		ChalkObject::operator= (that);
		m_Width = that.m_Width;
		m_Height = that.m_Height;
		m_OffsetX = that.m_OffsetX;
		m_OffsetY = that.m_OffsetY;
		m_NumPlanes = that.m_NumPlanes;
		if (m_Width * m_Height > 0)
		{
			m_Data = new ChalkColourIndex[m_Width * m_Height];
			memcpy(m_Data, that.m_Data, m_Width * m_Height * sizeof(ChalkColourIndex));
		}
	}
	return *this;
}

// ----------------------------------------------------------------------------
ChalkCharacter::~ChalkCharacter()
{
	if (m_Data)
		delete [] m_Data;
}

// ----------------------------------------------------------------------------
HRESULT ChalkCharacter::Write(ChalkStream& os) const
{
	os.WriteElement1("Character");
	os.WriteInteger("sizex", m_Width);
	os.WriteInteger("sizey", m_Height);
	os.WriteInteger("numplanes", m_NumPlanes);
	os.WriteInteger("offsetx", m_OffsetX);
	os.WriteInteger("offsety", m_OffsetY);
	if (m_NumPlanes > 8)
	{
		// Save 32 bits per pixel
		os.WriteBlob4("data4", m_Width * m_Height, m_Data);
	}
	else
	{
		// Save 8 bits per pixel
		vector<BYTE> temp;
		temp.resize(m_Width * m_Height);
		std::copy(m_Data, m_Data + m_Width * m_Height, temp.begin());
		os.WriteBlob1("data1", temp.size(), &temp[0]);
	}
	os.WriteEndElement("Character");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkCharacter::Read(ChalkStream& is)
{
	is.ReadElement1("Character");
	int width = is.ReadInteger("sizex");
	int height = is.ReadInteger("sizey");
	int numplanes = is.ReadInteger("numplanes");
	SetSize(width, height, numplanes);
	m_OffsetX = is.ReadInteger("offsetx");
	m_OffsetY = is.ReadInteger("offsety");
	if (m_NumPlanes > 8)
	{
		// Load 32 bits per pixel
		vector<DWORD> data = is.ReadBlob4("data4");
		if (!data.empty() && (data.size() <= width * height))
		{
			std::copy(data.begin(), data.end(), m_Data);
		}
	}
	else
	{
		// Load 8 bits per pixel
		vector<BYTE> data = is.ReadBlob1("data1");
		if (!data.empty() && (data.size() <= width * height))
		{
			std::copy(data.begin(), data.end(), m_Data);
		}
	}
	is.ReadEndElement("Character");
	return S_OK;
}

// ----------------------------------------------------------------------------
void ChalkCharacter::SetSize(int width, int height, int numplanes)
{
	if (m_Data)
		delete [] m_Data;
	m_Data = NULL;
	m_Width = width;
	m_Height = height;
	m_NumPlanes = numplanes;
	if (width * height > 0)
	{
		m_Data = new ChalkColourIndex[width * height];
		// Clear to black, in case the load routine misses some pixels
		// e.g. if a box goes past the edge of a bitmap file
		memset(m_Data, 0, width * height * sizeof(ChalkColourIndex));
	}
}


// ----------------------------------------------------------------------------
// Resize a bitmap preserving the old data
void ChalkCharacter::Resize(int width, int height)
{
	// Copy old data across
	ChalkColourIndex* pData = NULL;
	if (width * height > 0)
	{
		pData = new ChalkColourIndex[width * height];
		memset(pData, 0, width * height * sizeof(ChalkColourIndex));
		int my = min(height, m_Height);
		int mx = min(width, m_Width);
		for (int y = 0; y < my; ++y)
		{
			int posSrc = y * m_Width;
			int posDst = y * width;
			for (int x = 0; x < mx; ++x)
			{
				pData[posDst++] = m_Data[posSrc++];
			}
		}
	}

	// Use the new data
	m_Width = width;
	m_Height = height;
	if (m_Data)
		delete [] m_Data;
	m_Data = pData;
}


// ----------------------------------------------------------------------------
// ChalkBox
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkBox::ChalkBox() : ChalkObject(GUID_VOID, GUID_BOX),
	m_Xpos(0),
	m_Ypos(0),
	m_Zpos(0),
	m_Width(0),
	m_Height(0),
	m_Depth(0),
	m_OffsetX(0),
	m_OffsetY(0),
	m_OffsetZ(0),
	m_Type(0),
	m_Shape(shape_normal)
{
}


// ----------------------------------------------------------------------------
// Support ChalkObject interface
HRESULT ChalkBox::Write(ChalkStream& os) const
{
	os.WriteElement1("Box");
	os.WriteInteger("x", m_Xpos);
	os.WriteInteger("y", m_Ypos);
	os.WriteInteger("z", m_Zpos);
	os.WriteInteger("sizex", m_Width);
	os.WriteInteger("sizey", m_Height);
	os.WriteInteger("sizez", m_Depth);
	os.WriteInteger("offsetx", m_OffsetX);
	os.WriteInteger("offsety", m_OffsetY);
	os.WriteInteger("offsetz", m_OffsetZ);
	os.WriteInteger("type", m_Type);
	os.WriteInteger("shape", m_Shape);
	os.WriteEndElement("Box");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkBox::Read(ChalkStream& is)
{
	is.ReadElement1("Box");
	m_Xpos = is.ReadInteger("x");
	m_Ypos = is.ReadInteger("y");
	m_Zpos = is.ReadInteger("z");
	m_Width = is.ReadInteger("sizex");
	m_Height = is.ReadInteger("sizey");
	m_Depth = is.ReadInteger("sizez");
	m_OffsetX = is.ReadInteger("offsetx");
	m_OffsetY = is.ReadInteger("offsety");
	m_OffsetZ = is.ReadInteger("offsetz");
	m_Type = is.ReadInteger("type");
	m_Shape = static_cast<ShapeBox>(is.ReadInteger("shape"));
	is.ReadEndElement("Box");
	return S_OK;
}

// ----------------------------------------------------------------------------
// ChalkPoint
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkPoint::ChalkPoint()  : ChalkObject(GUID_VOID, GUID_POINT),
	m_Xpos(0),
	m_Ypos(0),
	m_Zpos(0)
{
}


// ----------------------------------------------------------------------------
HRESULT ChalkPoint::Write(ChalkStream& os) const
{
	os.WriteElement1("Point");
	os.WriteInteger("x", m_Xpos);
	os.WriteInteger("y", m_Ypos);
	os.WriteInteger("z", m_Zpos);
	os.WriteEndElement("Point");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkPoint::Read(ChalkStream& is)
{
	is.ReadElement1("Point");
	m_Xpos = is.ReadInteger("x");
	m_Ypos = is.ReadInteger("y");
	m_Zpos = is.ReadInteger("z");
	is.ReadEndElement("Point");
	return S_OK;
}

// ----------------------------------------------------------------------------
// ChalkPath
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkPath::ChalkPath()  : ChalkObject(GUID_VOID, GUID_PATH),
	m_Type(0),
	m_ObjDefn(0),
	m_Speed(1.0)
{
}


// ----------------------------------------------------------------------------
HRESULT ChalkPath::Write(ChalkStream& os) const
{
	os.WriteElement1("Path");
	os.WriteInteger("type", m_Type);
	os.WriteInteger("objdefn", m_ObjDefn);
	os.WriteReal("speed", m_Speed);
	os.WriteElement("Points", m_Data.size());
	// This line doesn't compile - perhaps because Write is virtual.
	// for_each(m_Data.begin(), m_Data.end(), bind2nd(mem_fun_ref(&ChalkPoint::Write), os));
	for (vector<ChalkPoint>::const_iterator i = m_Data.begin(); i != m_Data.end(); ++i)
	{
		i->Write(os);
	}
	os.WriteEndElement("Points");
	os.WriteEndElement("Path");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkPath::Read(ChalkStream& is)
{
	is.ReadElement1("Path");
	m_Type = is.ReadInteger("Type");
	m_ObjDefn = is.ReadInteger("ObjDefn");
	m_Speed = is.ReadReal("Speed");
	int nPoints = is.ReadElement("Points");
	m_Data.clear();
	for (int i = 0; i < nPoints; ++i)
	{
		ChalkPoint p;
		p.Read(is);
		m_Data.push_back(p);
	}
	is.ReadEndElement("Points");
	is.ReadEndElement("Path");
	return S_OK;
}

// ----------------------------------------------------------------------------
// ChalkSprite
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkSprite::ChalkSprite() : ChalkObject(GUID_VOID, GUID_SPRITE),
	m_Xpos(0),
	m_Ypos(0),
	m_Zpos(0),
	m_ObjDefn(0),
	m_PaletteOffset(0),
	m_Special(0),
	m_Attribute(0)
{
}


// ----------------------------------------------------------------------------
HRESULT ChalkSprite::Write(ChalkStream& os) const
{
	os.WriteElement1("Sprite");
	ChalkObject::Write(os);
	os.WriteInteger("x", m_Xpos);
	os.WriteInteger("y", m_Ypos);
	os.WriteInteger("z", m_Zpos);
	os.WriteInteger("objdefn", m_ObjDefn);
	os.WriteInteger("paletteoffset", m_PaletteOffset);
	os.WriteInteger("special", m_Special);
	os.WriteInteger("attribute", m_Attribute);
	os.WriteEndElement("Sprite");
	return S_OK;
}

// ----------------------------------------------------------------------------
HRESULT ChalkSprite::Read(ChalkStream& is)
{
	is.ReadElement1("Sprite");
	ChalkObject::Read(is);
	m_Xpos = is.ReadInteger("x");
	m_Ypos = is.ReadInteger("y");
	m_Zpos = is.ReadInteger("z");
	m_ObjDefn = is.ReadInteger("objdefn");
	m_PaletteOffset = is.ReadInteger("paletteoffset");
	m_Special = is.ReadInteger("special");
	m_Attribute = is.ReadInteger("attribute");
	is.ReadEndElement("sprite");
	return S_OK;
}
