// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkObjects.h
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Oct 2001 by Carl Muller
//
// Defines the major game objects
// ----------------------------------------------------------------------------

#include "chalkobj.h"

typedef unsigned long ChalkColour;       // RGBA
typedef unsigned long ChalkColourIndex;  // Index into a colour palette
typedef unsigned long ChalkBlockIndex;   // A block number
typedef unsigned long ChalkSpecial;      // Special features
typedef unsigned long ChalkAttribute;
const ChalkSpecial SPECIAL_YFLIP    = 1; // Flip the character vertically
const ChalkSpecial SPECIAL_XFLIP    = 2; // Flip the character horizontally
const ChalkSpecial SPECIAL_PRIORITY = 4; // Prioritise the character 
const ChalkSpecial SPECIAL_TRANSPARENT = 8; // Colour 0 is transparent

#include "constants.h"

class ChalkDisplay;

// ----------------------------------------------------------------------------
struct ColourRange
{
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

	int first;
	int last;
	int rate;
	int active;
};

// ----------------------------------------------------------------------------
// A palette
class ChalkPalette : public ChalkObject
{
public:
	ChalkPalette(); // Default constructor
	ChalkPalette(const ChalkPalette &s);  // Copy constructor
	~ChalkPalette();
	const ChalkPalette& operator= (const ChalkPalette&s);  // Assignment operator

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

	// Support alternative file formats
	static bool SupportsFile(const CFileName& filename); // Does this class support this file extension
	HRESULT ReadFile(const CFileName& filename);
	HRESULT WriteFile(const CFileName& filename) const;
	HRESULT WritePalette(IFFfile& outfile) const; // Helper function for CMAP/CRNG

	// Support specific alternative file formats
	HRESULT ReadLBM(const CFileName& filename);
	HRESULT WriteLBM(const CFileName& filename) const;

	void SetSize(int NumColours);
	void Set(int NumColours, const ChalkColour* pData);
	void SetFalseColour();

public:	// Data members
	ChalkColour* m_Data; // Palette data
	int m_NumColours; // Number of decent colours
	int m_Allocated; // Colours allocated (minimum 256)
	vector<ColourRange> m_ColourRanges;
};


// ----------------------------------------------------------------------------
// A single character
class ChalkCharacter : public ChalkObject
{
public:
	ChalkCharacter();
	~ChalkCharacter();
	ChalkCharacter(const ChalkCharacter &s);  // copy constructor
	const ChalkCharacter& operator= (const ChalkCharacter&s);  // assignment operator

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

	void SetSize(int width, int height, int numplanes);
	void Resize(int width, int height);

public:	// Data members
	int m_Width, m_Height;    // Size of character data
	int m_OffsetX, m_OffsetY; // Offset (for fonts, sprites)
	int m_NumPlanes; // Format of character data (does not affect memory storage)
	ChalkColourIndex *m_Data; // Raw data
};

// ----------------------------------------------------------------------------
// A picture class
class ChalkBitmap : public ChalkObject
{
public:
	ChalkBitmap();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

	// Support alternative file formats
	static bool SupportsFile(const CFileName& filename); // Does this class support this file extension
	HRESULT ReadFile(const CFileName& filename);
	HRESULT WriteFile(const CFileName& filename) const;

	// Support specific alternative file formats
	HRESULT ReadLBM(const CFileName& filename);
	HRESULT ReadPCX(const CFileName& filename);
	HRESULT ReadTGA(const CFileName& filename);
	HRESULT ReadBMP(const CFileName& filename);
	HRESULT ReadFont(const CFileName& filename);

	HRESULT WriteLBM(const CFileName& filename) const;
	HRESULT WritePBM(const CFileName& filename) const;
	HRESULT WritePCX(const CFileName& filename) const;
	HRESULT WriteTGA(const CFileName& filename) const;
	HRESULT WriteBMP(const CFileName& filename) const;

public:	// Data members
	ChalkPalette m_Palette;
	ChalkCharacter m_Character;
	int m_GridX;
	int m_GridY;
};

// ----------------------------------------------------------------------------
class ChalkMapBlock
{
public:
	ChalkMapBlock();
	void Set(DWORD value, ChalkSystem* systemtype);
	DWORD Get(ChalkSystem* systemtype) const;

public: // Data members
	ChalkBlockIndex m_BlockIndex; // Block or character index
	ChalkColourIndex m_PaletteOffset; // Offset to add to colours
	ChalkSpecial m_Special; // Special attributes (xflip, yflip, priority, transparent)
	ChalkAttribute m_Attribute; // Gameplay attribute (user defined)
};

// ----------------------------------------------------------------------------
// An abstract base class, for things that can be tessalated as part of a map
class ChalkMapBase : public ChalkObject
{
public:
	ChalkMapBase(GUID id, GUID typeguid);

	// These functions must be implemented by derived classes
	virtual void DrawBlock(ChalkDisplay& display, CRect rt,
		CPoint pt, ChalkMapBlock& block, ChalkPalette& Palette) const;
	virtual ChalkPalette* GetPalette();

public: // Data members
	GUID m_SystemType; // What type of system is this on (affects loading MAPs)
	int m_Grid1X; // Size in pixels of what we are made of
	int m_Grid1Y; // Size in pixels of what we are made of
	int m_Grid2X; // Size in pixels of what we can make
	int m_Grid2Y; // Size in pixels of what we can make
};

// ----------------------------------------------------------------------------
// A character set, or a collection of icons
class ChalkFont : public ChalkMapBase
{
public:
	ChalkFont();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);
	void DrawBlock(ChalkDisplay& display, CRect rt,
		CPoint pt, ChalkMapBlock& block, ChalkPalette& Palette) const;

	// Support alternative file formats
	static bool SupportsFile(const CFileName& filename); // Does this class support this file extension
	HRESULT ReadFile(const CFileName& filename);
	HRESULT WriteFile(const CFileName& filename) const;

	// Support specific alternative file formats
	HRESULT ReadCHR(const CFileName& filename);
	HRESULT ReadSET(const CFileName& filename);
	HRESULT ReadGBC(const CFileName& filename);
	HRESULT ReadSF2(const CFileName& filename);
	HRESULT ReadSF4(const CFileName& filename);
	HRESULT ReadMCH(const CFileName& filename);
	HRESULT ReadGCH(const CFileName& filename);
	HRESULT ReadSCH(const CFileName& filename);
	HRESULT ReadSF8(const CFileName& filename);
	HRESULT ReadSFB(const CFileName& filename);
	HRESULT ReadBOX(const CFileName& filename,
		ChalkPalette* pPalette = NULL, bool merge = false); // Load in a boxed bitmap (special case)
	HRESULT ReadPXP(const CFileName& filename,
		ChalkPalette* pPalette = NULL, bool merge = false); // Load in a pxp file
	HRESULT ReadBitmap(const CFileName& filename,
		ChalkPalette* pPalette = NULL, bool merge = false); // Load in a bitmap (special case)

	HRESULT WriteCHR(const CFileName& filename) const;
	HRESULT WriteSET(const CFileName& filename) const;
	HRESULT WriteGBC(const CFileName& filename) const;
	HRESULT WriteSF2(const CFileName& filename) const;
	HRESULT WriteSF4(const CFileName& filename) const;
	HRESULT WriteMCH(const CFileName& filename) const;
	HRESULT WriteGCH(const CFileName& filename) const;
	HRESULT WriteSCH(const CFileName& filename) const;
	HRESULT WriteSF8(const CFileName& filename) const;
	HRESULT WriteSFB(const CFileName& filename) const;

	// Helper functions
	void Clear() { m_Data.clear(); }
	void CreateHexFont(); // Generate an 8*8 hexadecimal font

public:	// Data members
	vector<ChalkCharacter> m_Data;
};


// ----------------------------------------------------------------------------
class ChalkMap : public ChalkObject
{
public:
	ChalkMap();
	ChalkMap(const ChalkMap &s);  // copy constructor
	const ChalkMap& operator= (const ChalkMap&s);  // assignment operator
	~ChalkMap();
	void SetSize(int width, int height);
	void Resize(int width, int height); // Preserving old data

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	int m_Width, m_Height;
	ChalkMapBlock *m_Data;
};

// ----------------------------------------------------------------------------
class ChalkMapAnim : public ChalkMapBase
{
public:
	ChalkMapAnim();
	~ChalkMapAnim();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);
	void DrawBlock(ChalkDisplay& display, CRect rt,
		CPoint pt, ChalkMapBlock& block, ChalkPalette& Palette) const;
	ChalkPalette* GetPalette();

	// Support alternative file formats
	static bool SupportsFile(const CFileName& filename); // Does this class support this file extension
	HRESULT ReadFile(const CFileName& filename);
	HRESULT WriteFile(const CFileName& filename) const;

	// Support specific alternative file formats
	HRESULT ReadMAP(const CFileName& filename);
	HRESULT WriteMAP(const CFileName& filename) const;

	HRESULT LoadBlocks(const CFileName& blockname, const CFileName& attfontname,
		int matrixx, int matrixy);

public:	// Data members
	vector<ChalkMap> m_Data;
	ChalkPalette m_Palette;
	ChalkMapBase* m_pBlocks; // Blocks object
	ChalkMapBase* m_pAttrBlocks; // Attribute blocks object
	ChalkPalette m_AttrPalette; // Attribute blocks palette
	CFileName m_KidName; // Filename of next layer
	double m_Speed; // Animation speed in Hz
private:
	ChalkMapAnim(const ChalkMapAnim &s);  // dummy copy constructor
	const ChalkMapAnim& operator= (const ChalkMapAnim&s);  // dummy assignment operator
};

// ----------------------------------------------------------------------------
// Box shapes
typedef enum
{
	shape_normal,
	shape_oblique,
	shape_isometric,
} ShapeBox;

// ----------------------------------------------------------------------------
class ChalkBox : public ChalkObject
{
public:
	ChalkBox();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	int m_Xpos, m_Ypos, m_Zpos;
	int m_Width, m_Height, m_Depth;
	int m_OffsetX, m_OffsetY, m_OffsetZ;
	int m_Type;
	ShapeBox m_Shape;
};

// ----------------------------------------------------------------------------
class ChalkPoint : public ChalkObject
{
public:
	ChalkPoint();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	int m_Xpos, m_Ypos, m_Zpos;
};

// ----------------------------------------------------------------------------
class ChalkPath : public ChalkObject
{
public:
	ChalkPath();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	vector<ChalkPoint> m_Data;
	int m_Type;
	ChalkBlockIndex m_ObjDefn;
	double m_Speed;
};

// ----------------------------------------------------------------------------
class ChalkSprite : public ChalkObject
{
public:
	ChalkSprite();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	int m_Xpos, m_Ypos, m_Zpos;
	ChalkBlockIndex m_ObjDefn;
	ChalkColourIndex m_PaletteOffset; // Offset to add to colours
	ChalkSpecial m_Special; // Special attributes (xflip, yflip, priority, transparent)
	ChalkAttribute m_Attribute; // Gameplay attribute (user defined)
};

// ----------------------------------------------------------------------------
class ChalkFrame : public ChalkObject
{
public:
	ChalkFrame();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

public: // Data members
	vector<ChalkSprite> m_Sprites;
	vector<ChalkBox> m_Boxes;
	vector<ChalkPath> m_Paths;

	// Helper functions
	void SelectAll(bool bSprites, bool bBoxes, bool bPaths); // Select all objects or none
	ChalkObject* GetFirstSelected(); // Return the first selected object
	ChalkObject* GetNextSelected(); // What should be selected next? (tab order)
	ChalkObject* GetPrevSelected(); // What should be selected next? (tab order)
	ChalkObject* GetNextSelected(ChalkFont& SpriteFont, CPoint pt,
		bool bBoxOnly = false); // What should be selected next? (click)
	CRect GetSelectedExtent() const; // What is the extent of selected items
};

// ----------------------------------------------------------------------------
class ChalkAnimation : public ChalkObject
{
public:
	ChalkAnimation();

	// Support ChalkObject interface
	HRESULT Write(ChalkStream& os) const;
	HRESULT Read(ChalkStream& is);

	// Support alternative file formats
	static bool SupportsFile(const CFileName& filename); // Does this class support this file extension
	HRESULT ReadFile(const CFileName& filename);
	HRESULT WriteFile(const CFileName& filename) const;

	// Support specific alternative file formats
	HRESULT ReadBOX(const CFileName& filename);
	HRESULT WriteBOX(const CFileName& filename) const;
	HRESULT ReadFRK(const CFileName& filename);
	HRESULT WriteFRK(const CFileName& filename) const;

	HRESULT LoadBlocks();

public: // Data members
	vector<ChalkFrame> m_Frames;
	ChalkFont m_Font; // Base of the sprites
	ChalkPalette m_Palette; // Palette for the sprites
	int m_GridX;
	int m_GridY;
	CFileName m_KidName; // Filename of next layer
	vector<CFileName> m_DefnNames; // Filenames of sprite definitions
	double m_Speed; // Animation speed in Hz
};
