// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkPalette.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 23 Oct 2001 by Carl Muller
//
// Implements object which supports RGB palettes
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkobjects.h"

// ----------------------------------------------------------------------------
// ChalkPalette
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkPalette::ChalkPalette() :
	ChalkObject(GUID_VOID, GUID_PALETTE),
	m_Data(NULL),
	m_NumColours(0),
	m_Allocated(0)
{
}

// ----------------------------------------------------------------------------
// Copy constructor
ChalkPalette::ChalkPalette(const ChalkPalette &that) :
	ChalkObject(that.m_Id, GUID_PALETTE),
	m_Data(NULL),
	m_NumColours(0),
	m_Allocated(0)
{
	*this = that; // Call operator=
}

// ----------------------------------------------------------------------------
ChalkPalette::~ChalkPalette()
{
	if (m_Data)
		delete [] m_Data;
}

// ----------------------------------------------------------------------------
// Assignment operator
const ChalkPalette& ChalkPalette::operator= (const ChalkPalette& that)
{
	int i, n;

	if (this != &that)
	{
		SetSize(that.m_NumColours);
		for (i = 0; i < m_NumColours; ++i)
			m_Data[i] = that.m_Data[i];
		n = that.m_ColourRanges.size();
		m_ColourRanges.resize(n);
		for (i = 0; i < n; ++i)
			m_ColourRanges[i] = that.m_ColourRanges[i];
	}
	return *this;
}

// ----------------------------------------------------------------------------
// Support XML/Undo/Clipboard
HRESULT ChalkPalette::Write(ChalkStream& os) const
{
	os.WriteElement1("Palette");
	os.WriteBlob4("colours", m_NumColours, m_Data);
	int n = m_ColourRanges.size();
	os.WriteElement("ColourRanges", n);
	for (int i = 0; i < n; ++i)
	{
		os.WriteElement1("ColourRange");
		os.WriteInteger("first", m_ColourRanges[i].first);
		os.WriteInteger("last", m_ColourRanges[i].last);
		os.WriteInteger("rate", m_ColourRanges[i].rate);
		os.WriteInteger("active", m_ColourRanges[i].active);
		os.WriteEndElement("ColourRange");
	}
	os.WriteEndElement("ColourRanges");
	os.WriteEndElement("Palette");
	return S_OK;
}

// ----------------------------------------------------------------------------
// Support XML/Undo/Clipboard
HRESULT ChalkPalette::Read(ChalkStream& is)
{
	is.ReadElement1("Palette");
	vector<DWORD> data = is.ReadBlob4("colours");
	if (!data.empty())
	{
		Set(data.size(), &data[0]);
	}

	int n = is.ReadElement("ColourRanges");
	m_ColourRanges.resize(n);
	for (int i = 0; i < n; ++i)
	{
		is.ReadElement1("ColourRange");
		m_ColourRanges[i].first = is.ReadInteger("first");
		m_ColourRanges[i].last = is.ReadInteger("last");
		m_ColourRanges[i].rate = is.ReadInteger("rate");
		m_ColourRanges[i].active = is.ReadInteger("active");
		is.ReadEndElement("ColourRange");
	}
	is.ReadEndElement("ColourRanges");
	is.ReadEndElement("Palette");
	return S_OK;
}

// ----------------------------------------------------------------------------
// Does this class support this alternative file format?
bool ChalkPalette::SupportsFile(const CFileName& filename)
{
	if (filename.HasExt(FILEEXTENSION_COP))
		return true;
	return false;
}

// ----------------------------------------------------------------------------
// Distribute the read alternative file command
HRESULT ChalkPalette::ReadFile(const CFileName& filename)
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return ReadXML(filename);
	if (filename.HasExt(FILEEXTENSION_COP))
		return ReadLBM(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
// Distribute the write alternative file command
HRESULT ChalkPalette::WriteFile(const CFileName& filename) const
{
	m_FileName = filename.Get();
	if (filename.HasExt(FILEEXTENSION_XML))
		return WriteXML(filename);
	if (filename.HasExt(FILEEXTENSION_COP))
		return WriteLBM(filename);
	m_FileName.Clear();
	return E_NOTIMPL;
}

// ----------------------------------------------------------------------------
// Set the palette size
void ChalkPalette::SetSize(int NumColours)
{
	int i;

	if (m_Data)
		delete [] m_Data;
	m_Data = NULL;
	m_NumColours = NumColours;
	m_Allocated = max(256, NumColours);
	m_Data = new ChalkColour[m_Allocated];
	for (i = 0; i < m_Allocated; ++i)
		m_Data[i] = COLOUR_MAGENTA;
}

// ----------------------------------------------------------------------------
// Copy specific colours into the palette
void ChalkPalette::Set(int NumColours, const ChalkColour* pData)
{
	int i;

	SetSize(NumColours);
	for (i = 0; i < NumColours; ++i)
		m_Data[i] = pData[i];
}

// ----------------------------------------------------------------------------
// Make a palette visible
void ChalkPalette::SetFalseColour()
{
	SetSize(256);

	int i, r, g, b;
	for (i = 0; i < 256; ++i)
	{
		r = (i * 47) & 255;
		g = (i * 123) & 255;
		b = (i * 197) & 255;
		m_Data[i] = (r << 16) | (g << 8) | (b);
	}
}

