// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkStream.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 24 Oct 2001 by Carl Muller
//
// Supports stream objects
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "chalkObjects.h"

// ----------------------------------------------------------------------------
// ChalkStreamText
// This stream is fancy - for writing XML streams
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Translation table for writing Base64 blobs
// See RFC 1521 for rough idea
class Base64
{
public:
	Base64();
	vector<BYTE> Decode(const vector<BYTE>& value);
	vector<BYTE> Encode(const vector<BYTE>& value);

private:
	// Meaning
	enum {
		// 0..63 represent 6-bit data
		base64pad    = 64, // Pad and finish the data
		base64end    = 65, // Finish the data
		base64unused = 66, // Ignore
		base64max    = 67  // String terminator
	};
	// Characters
	enum {
		char64pad = '=',
		char64end = '<'
	};
	// Meaning
	enum {
		// 0..15 represent 4-bit data
		base16unused = 16, // Ignore
		base16max    = 17  // String terminator
	};
	char tobase64[base64max];
	char frombase64[256];

	// Support hexadecimal tables also
public:
	char tobase16[base16max];
	char frombase16[256];
} g_Base64;


// ----------------------------------------------------------------------------
// Initialise tables required for encoding/decoding
Base64::Base64()
{
	strncpy(tobase64,
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=<",
		sizeof(tobase64));

	strncpy(tobase16, "0123456789ABCDEF", sizeof(tobase16));

	int i;
	for (i = 0; i < 256; ++i)
	{
		frombase64[i] = base64unused;
		frombase16[i] = base16unused;
	}
	for (i = 0; i < base64unused; ++i)
	{
		frombase64[tobase64[i]] = i;
	}
	for (i = 0; i < base16unused; ++i)
	{
		frombase16[tobase16[i]] = i;
	}
}


// ----------------------------------------------------------------------------
// This decoding terminates on = or <
vector<BYTE> Base64::Decode(const vector<BYTE>& value)
{
	vector<BYTE> result;

	int data = 0;
	int pos = 0;
	for (vector<BYTE>::const_iterator i = value.begin(); i != value.end(); ++i)
	{
		char ch = frombase64[*i];
		if (ch == base64unused)
			continue;
		if (ch == base64pad)
			break;
		if (ch == base64end)
			break;
		switch (pos++)
		{
		case 0:
			data = ch << 2;
			break;
		case 1:
			result.push_back(data | ((ch & 48) >> 4));
			data = (ch & 15) << 4;
			break;
		case 2:
			result.push_back(data | ((ch & 60) >> 2));
			data = (ch & 3) << 6;
			break;
		case 3:
			result.push_back(data | ch);
			data = 0;
			pos = 0;
			break;
		}
	}
	return result;
}


// ----------------------------------------------------------------------------
vector<BYTE> Base64::Encode(const vector<BYTE>& value)
{
	int nIn = value.size();
	int nCharsPerLine = 72;
	int nLines = nIn / ((nCharsPerLine / 4) * 3);
	int nOut = (2 * nLines) + (4 * ((nIn + 3 - 1) / 3));
	vector<BYTE> result;
	result.reserve(nOut);

	// Convert the bulk of the data
	int nLineChars = 0;
	int i;
	for (i = 0; i <= nIn - 3; i += 3)
	{
		// Convert 3 octets into 4 characters
		result.push_back(tobase64[(value[i] & 252) >> 2]);
		result.push_back(tobase64[
			((value[i] & 3) << 4) | ((value[i+1] & 240) >> 4) ]);
		result.push_back(tobase64[
			((value[i+1] & 15) << 2) | ((value[i+2] & 192) >> 6) ]);
		result.push_back(tobase64[value[i+2] & 63]);

		// Break up long lines
		nLineChars += 4;
		if (nLineChars >= nCharsPerLine)
		{
			result.push_back('\r');
			result.push_back('\n');
			nLineChars = 0;
		}
	}

	// Add padding at the end
	switch (nIn % 3)
	{
	case 0:
		break;

	case 1:
		result.push_back(tobase64[(value[i] & 252) >> 2]);
		result.push_back(tobase64[(value[i] & 3) << 4]);
		result.push_back(char64pad);
		result.push_back(char64pad);
		break;

	case 2:
		result.push_back(tobase64[(value[i] & 252) >> 2]);
		result.push_back(tobase64[
			((value[i] & 3) << 4) | ((value[i+1] & 240) >> 4) ]);
		result.push_back(tobase64[(value[i+1] & 15) << 2]);
		result.push_back(char64pad);
		break;
	}

	return result;
}

// ----------------------------------------------------------------------------
//
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
void XMLNode::Clear()
{
	m_Name.resize(0);
	m_Attributes.clear();
	m_Text.clear();
	vector<XMLNode*>::iterator i;
	for (i = m_Kids.begin(); i != m_Kids.end(); ++i)
	{
		XMLNode* node = *i;
		if (node)
			delete node;
	}
	m_Kids.clear();
}

// ----------------------------------------------------------------------------
void XMLNode::Read(BinaryStream& is, int ch)
{
	Clear();

	// Ignore leading whitespace
	while (isspace(ch))
		ch = is.ReadByte();
	if (ch == EOF)
		return;

	// Get name of element
	m_Name = ch;
	m_Name += is.ReadString("> \t\v\r\n", "", &ch);
	if (ch == EOF)
		return;

	// Handle attributes
	while (ch != '>')
	{
		wstring key = is.ReadString(">=", " \t\v\r\n", &ch);
		if (ch == '=')
		{
			while (isspace(ch))
				ch = is.ReadByte();
			if (ch == '"')
			{
				wstring value = is.ReadString("\"", "", &ch);
				m_Attributes[key] = value;
				ch = is.ReadByte();
			}
			else
			{
				wstring value;
				value = ch;
				value += is.ReadString("> \t\v\r\n", "", &ch);
			}
		}
		if (ch == EOF)
			return;
	}
	// Handle text
	ch = is.ReadByte();
	while (ch != EOF)
	{
		if (ch == '<')
		{
			ch = is.ReadByte();
			if (ch == '/')
			{
				wstring endname = is.ReadString(">", " \t\v\r\n", &ch);
				// endname should be equal to m_Name
				break;
			}
			else
			{
				// Add a kid
				XMLNode* pItem = new XMLNode();
				if (pItem)
				{
					pItem->Read(is, ch);
					m_Kids.push_back(pItem);
				}
			}
		}
		else
		{
			m_Text.push_back(ch);
			ch = is.ReadByte();
		}
	}
};

// ----------------------------------------------------------------------------
//
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkStreamText::ChalkStreamText(BinaryStream& BaseStream) :
	m_Stream(BaseStream),
	m_pXML(NULL)
{
	if (m_Stream.CanRead())
	{
		// Read in the complete XML tree
		int ch = 0;
		while (ch != '<' && ch != EOF)
		{
			ch = m_Stream.ReadByte();
		}
		if (ch != EOF)
		{
			ch = m_Stream.ReadByte();
			if (ch != EOF)
			{
				m_XML.Read(m_Stream, ch);
			}
		}
	}
}


// ----------------------------------------------------------------------------
void ChalkStreamText::WriteElement1(const string& elementname)
{
	stringstream s;

	s << "<" << elementname << ">";
	m_Stream.WriteString(s.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteElement(const string& elementname, long numkids)
{
	stringstream s;

	s << "<" << elementname << " Count=\"" << numkids << "\">";
	m_Stream.WriteString(s.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteEndElement(const string& elementname)
{
	stringstream s;
	s << "</" << elementname << ">\r\n";
	m_Stream.WriteString(s.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteInteger(const string& elementname, long value)
{
	stringstream s;
	s << "<" << elementname << ">" << value << "</" << elementname << ">";
	m_Stream.WriteString(s.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteReal(const string& elementname, double value)
{
	stringstream s;
	s << "<" << elementname << ">" << value << "</" << elementname << ">";
	m_Stream.WriteString(s.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteString(const string& elementname, const wstring& value)
{
	stringstream s;
	s << "<" << elementname << ">";
	m_Stream.WriteString(s.str());
	m_Stream.WriteString(value);

	stringstream s2;
	s2 << "</" << elementname << ">";
	m_Stream.WriteString(s2.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteString(const string& elementname, const string& value)
{
	stringstream s;
	s << "<" << elementname << ">";
	m_Stream.WriteString(s.str());
	m_Stream.WriteString(value);

	stringstream s2;
	s2 << "</" << elementname << ">";
	m_Stream.WriteString(s2.str());
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteGUID(const string& elementname, GUID value)
{
	m_Stream.WriteString("<");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");

	// TBD: make this less verbose by using the standard library instead of longhand
	const char* tohex = &g_Base64.tobase16[0];

	// Write out the encoded data
	m_Stream.WriteByte(tohex[(value.Data1 >> 28) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1 >> 24) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1 >> 20) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1 >> 16) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1 >> 12) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1 >>  8) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1 >>  4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data1      ) & 15]);
	m_Stream.WriteByte('-');
	m_Stream.WriteByte(tohex[(value.Data2 >> 12) & 15]);
	m_Stream.WriteByte(tohex[(value.Data2 >>  8) & 15]);
	m_Stream.WriteByte(tohex[(value.Data2 >>  4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data2      ) & 15]);
	m_Stream.WriteByte('-');
	m_Stream.WriteByte(tohex[(value.Data3 >> 12) & 15]);
	m_Stream.WriteByte(tohex[(value.Data3 >>  8) & 15]);
	m_Stream.WriteByte(tohex[(value.Data3 >>  4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data3      ) & 15]);
	m_Stream.WriteByte('-');
	m_Stream.WriteByte(tohex[(value.Data4[0] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[0]     ) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[1] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[1]     ) & 15]);
	m_Stream.WriteByte('-');
	m_Stream.WriteByte(tohex[(value.Data4[2] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[2]     ) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[3] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[3]     ) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[4] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[4]     ) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[5] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[5]     ) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[6] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[6]     ) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[7] >> 4) & 15]);
	m_Stream.WriteByte(tohex[(value.Data4[7]     ) & 15]);

	m_Stream.WriteString("</");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteBlob1(const string& elementname, long length, const BYTE* value)
{
	// Copy data into correct data type
	vector<BYTE> raw;
	raw.assign(value, value + length);

	// Encode the data
	vector<BYTE> coded = g_Base64.Encode(raw);

	// Write out the encoded data
	m_Stream.WriteString("\r\n<");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
	m_Stream.WriteBytes(&coded[0], coded.size());
	m_Stream.WriteString("</");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteBlob2(const string& elementname, long length, const WORD* value)
{
	// Copy data into correct format (little-endian)
	vector<BYTE> raw;
	raw.reserve(length * 2);
	for (int i = 0; i < length; ++i)
	{
		raw.push_back(*value & 0xff);
		raw.push_back((*value >> 8) & 0xff);
		++value;
	}

	// Encode the data
	vector<BYTE> coded = g_Base64.Encode(raw);

	// Write out the encoded data
	m_Stream.WriteString("\r\n<");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
	m_Stream.WriteBytes(&coded[0], coded.size());
	m_Stream.WriteString("</");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
}

// ----------------------------------------------------------------------------
void ChalkStreamText::WriteBlob4(const string& elementname, long length, const DWORD* value)
{
	// Copy data into correct format (little-endian)
	vector<BYTE> raw;
	raw.reserve(length * 4);
	for (int i = 0; i < length; ++i)
	{
		raw.push_back(*value & 0xff);
		raw.push_back((*value >> 8) & 0xff);
		raw.push_back((*value >> 16) & 0xff);
		raw.push_back((*value >> 24) & 0xff);
		++value;
	}

	// Encode the data
	vector<BYTE> coded = g_Base64.Encode(raw);

	// Write out the encoded data
	m_Stream.WriteString("\r\n<");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
	m_Stream.WriteBytes(&coded[0], coded.size());
	m_Stream.WriteString("</");
	m_Stream.WriteString(elementname);
	m_Stream.WriteString(">");
}

// ----------------------------------------------------------------------------
vector<BYTE> ChalkStreamText::ReadRaw(const string& elementname, bool& bFailed)
{
	if (m_pXML)
	{
		wstring name = MakeUpper(WidenString(elementname));
		for (vector<XMLNode*>::iterator p = m_pXML->m_Kids.begin(); p != m_pXML->m_Kids.end(); ++p)
		{
			if (!(*p)->m_BeenRead && (name == MakeUpper((*p)->m_Name)))
			{
				// Success
				(*p)->m_BeenRead = true;
				bFailed = false;
				return (*p)->m_Text;
			}
		}
	}

	vector<BYTE> result;
	bFailed = true;
	return result;
}

// ----------------------------------------------------------------------------
long ChalkStreamText::ReadInteger(const string& elementname, long defaultvalue)
{
	bool bFailed = true;
	vector<BYTE> item = ReadRaw(elementname, bFailed);
	if (bFailed)
		return defaultvalue;
	return atol(reinterpret_cast<const char *>(&item[0]));
}

// ----------------------------------------------------------------------------
double ChalkStreamText::ReadReal(const string& elementname, double defaultvalue)
{
	bool bFailed = true;
	vector<BYTE> item = ReadRaw(elementname, bFailed);
	if (bFailed)
		return defaultvalue;
	return atof(reinterpret_cast<const char *>(&item[0]));
}


// ----------------------------------------------------------------------------
wstring ChalkStreamText::ReadString(const string& elementname, const wstring& defaultvalue)
{
	bool bFailed = true;
	vector<BYTE> item = ReadRaw(elementname, bFailed);
	if (bFailed)
		return defaultvalue;
	return WidenString(item);
}

// ----------------------------------------------------------------------------
GUID ChalkStreamText::ReadGUID(const string& elementname)
{
	GUID value = { 0, 0, 0,0,0,0,0,0,0,0 };
	const char* fromhex = &g_Base64.frombase16[0];

	bool bFailed = true;
	vector<BYTE> item = ReadRaw(elementname, bFailed);

	// TBD: make this less verbose by using the standard library instead of longhand
	int pos = 0;
	for (vector<BYTE>::const_iterator i = item.begin(); i != item.end(); ++i)
	{
		DWORD val = fromhex[*i];
		if (val < 16)
		{
			switch (pos) 
			{
			case 0: value.Data1 |= (val << 28); break;
			case 1: value.Data1 |= (val << 24); break;
			case 2: value.Data1 |= (val << 20); break;
			case 3: value.Data1 |= (val << 16); break;
			case 4: value.Data1 |= (val << 12); break;
			case 5: value.Data1 |= (val <<  8); break;
			case 6: value.Data1 |= (val <<  4); break;
			case 7: value.Data1 |= (val      ); break;

			case 8 : value.Data2 |= (val << 12); break;
			case 9 : value.Data2 |= (val <<  8); break;
			case 10: value.Data2 |= (val <<  4); break;
			case 11: value.Data2 |= (val      ); break;

			case 12: value.Data3 |= (val << 12); break;
			case 13: value.Data3 |= (val <<  8); break;
			case 14: value.Data3 |= (val <<  4); break;
			case 15: value.Data3 |= (val      ); break;

			case 16: value.Data4[0] |= (val << 4); break;
			case 17: value.Data4[0] |= (val     ); break;
			case 18: value.Data4[1] |= (val << 4); break;
			case 19: value.Data4[1] |= (val     ); break;
			case 20: value.Data4[2] |= (val << 4); break;
			case 21: value.Data4[2] |= (val     ); break;
			case 22: value.Data4[3] |= (val << 4); break;
			case 23: value.Data4[3] |= (val     ); break;
			case 24: value.Data4[4] |= (val << 4); break;
			case 25: value.Data4[4] |= (val     ); break;
			case 26: value.Data4[5] |= (val << 4); break;
			case 27: value.Data4[5] |= (val     ); break;
			case 28: value.Data4[6] |= (val << 4); break;
			case 29: value.Data4[6] |= (val     ); break;
			case 30: value.Data4[7] |= (val << 4); break;
			case 31: value.Data4[7] |= (val     ); break;
			default: break;
			}
			++pos;
		}
	}
	return value;
}


// ----------------------------------------------------------------------------
vector<BYTE> ChalkStreamText::ReadBlob1(const string& elementname)
{
	bool bFailed = true;
	vector<BYTE> item = ReadRaw(elementname, bFailed);
	if (!bFailed)
	{
		vector<BYTE> result = g_Base64.Decode(item);
		return result;
	}
	vector<BYTE> result;
	return result;
}

// ----------------------------------------------------------------------------
vector<WORD> ChalkStreamText::ReadBlob2(const string& elementname)
{
	vector<BYTE> decoded = ReadBlob1(elementname);

	vector<WORD> result;
	int n = decoded.size();
	result.reserve(n / 2);
	int i;
	for (i = 0; i + 2 <= n; i += 2)
	{
		result.push_back(
			  (DWORD(decoded[i+1]) << 8)
			| (DWORD(decoded[i])) );
	}

	return result;
}

// ----------------------------------------------------------------------------
vector<DWORD> ChalkStreamText::ReadBlob4(const string& elementname)
{
	vector<BYTE> decoded = ReadBlob1(elementname);

	vector<DWORD> result;
	int n = decoded.size();
	result.reserve(n / 4);
	int i;
	for (i = 0; i + 4 <= n; i += 4)
	{
		result.push_back(
			  (DWORD(decoded[i+3]) << 24)
			| (DWORD(decoded[i+2]) << 16)
			| (DWORD(decoded[i+1]) << 8)
			| (DWORD(decoded[i])) );
	}

	return result;
}

// ----------------------------------------------------------------------------
void ChalkStreamText::ReadElement1(const string& elementname)
{
	ReadElement(elementname);
}

// ----------------------------------------------------------------------------
long ChalkStreamText::ReadElement(const string& elementname)
{
	wstring name = MakeUpper(WidenString(elementname));
	if (m_pXML)
	{
		int i = 0;
		for (vector<XMLNode*>::iterator p = m_pXML->m_Kids.begin(); p != m_pXML->m_Kids.end(); ++p)
		{
			if (!(*p)->m_BeenRead && (name == MakeUpper((*p)->m_Name)))
			{
				// Success
				m_XMLindex.push_back(i);
				m_pXML = *p;
				m_pXML->m_BeenRead = true;
				return m_pXML->m_Kids.size();
			}
			++i;
		}
		// No success
		m_XMLindex.push_back(-1);
		m_pXML = NULL;
	}

	else if (m_XMLindex.empty())
	{
		if (!m_XML.m_BeenRead && (name == MakeUpper(m_XML.m_Name)))
		{
			// Success
			m_pXML = &m_XML;
			m_pXML->m_BeenRead = true;
			return m_pXML->m_Kids.size();
		}
		// No success
	}

	else
	{
		// Trying to navigate through a non-existant branch
		m_XMLindex.push_back(-1);
	}
	return 0;
}

// ----------------------------------------------------------------------------
void ChalkStreamText::ReadEndElement(const string& elementname)
{
	if (!m_XMLindex.empty())
	{
		m_XMLindex.pop_back();

		// Find the parent
		if (m_XMLindex.empty())
			m_pXML = &m_XML;
		else if (m_XMLindex.back() < 0)
			m_pXML = NULL;
		else
		{
			m_pXML = &m_XML;
			for (int i = 0; i < m_XMLindex.size(); ++i)
			{
				m_pXML = m_pXML->m_Kids[m_XMLindex[i]];
			}
		}
	}
}



// ----------------------------------------------------------------------------
// ChalkStreamBinary
//
// This class is fragile for reading and writing,
// so it is only suitable for in memory use (such as undo buffers)
// not for file storage.
// Read and Write functions of client classes *must* store members in
// consistent ways if they want to work.
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
ChalkStreamBinary::ChalkStreamBinary(BinaryStream& BaseStream) :
	m_Stream(BaseStream)
{
}


// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteElement1(const string&)
{
	// Do nothing
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteElement(const string&, long numkids)
{
	m_Stream.WriteLong(numkids);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteEndElement(const string&)
{
	// Do nothing
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteInteger(const string&, long value)
{
	m_Stream.WriteLong(value);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteReal(const string&, double value)
{
	// This is fragile with respect to endianness and compiler idiosyncracies
	union { long val1[2]; double val2; } val;

	val.val2 = value;
	m_Stream.WriteLong(val.val1[0]);
	m_Stream.WriteLong(val.val1[1]);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteString(const string&, const wstring& value)
{
	m_Stream.WriteString(value);
	m_Stream.WriteByte(0);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteString(const string&, const string& value)
{
	m_Stream.WriteString(value);
	m_Stream.WriteByte(0);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteGUID(const string&, GUID value)
{
	int i;

	m_Stream.WriteIntelLong(value.Data1);
	m_Stream.WriteIntelWord(value.Data2);
	m_Stream.WriteIntelWord(value.Data3);
	for (i = 0; i < 8; ++i)
		m_Stream.WriteByte(value.Data4[i]);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteBlob1(const string&, long length, const BYTE* value)
{
	m_Stream.WriteLong(length);
	m_Stream.WriteBytes(value, length);
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteBlob2(const string&, long length, const WORD* value)
{
	m_Stream.WriteLong(length);
	for (long i = 0; i < length; ++i)
	{
		m_Stream.WriteWord(value[i]);
	}
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::WriteBlob4(const string&, long length, const DWORD* value)
{
	m_Stream.WriteLong(length);
	for (long i = 0; i < length; ++i)
	{
		m_Stream.WriteLong(value[i]);
	}
}

// ----------------------------------------------------------------------------
long ChalkStreamBinary::ReadInteger(const string&, long)
{
	return m_Stream.ReadLong();
}

// ----------------------------------------------------------------------------
double ChalkStreamBinary::ReadReal(const string&, double)
{
	// This is fragile with respect to endianness and compiler idiosyncracies
	union { long val1[2]; double val2; } val;

	val.val1[0] = m_Stream.ReadLong();
	val.val1[1] = m_Stream.ReadLong();
	return val.val2;
}


// ----------------------------------------------------------------------------
wstring ChalkStreamBinary::ReadString(const string&, const wstring&)
{
	return m_Stream.ReadString("", "", NULL);
}

// ----------------------------------------------------------------------------
GUID ChalkStreamBinary::ReadGUID(const string& elementname)
{
	GUID value;
	int i;

	value.Data1 = m_Stream.ReadIntelLong();
	value.Data2 = m_Stream.ReadIntelWord();
	value.Data3 = m_Stream.ReadIntelWord();
	for (i = 0; i < 8; ++i)
		value.Data4[i] = m_Stream.ReadByte();
	return value;
}


// ----------------------------------------------------------------------------
vector<BYTE> ChalkStreamBinary::ReadBlob1(const string& elementname)
{
	vector<BYTE> result;
	long length = m_Stream.ReadLong();
	if (length > 0)
	{
		result.resize(length);
		m_Stream.ReadBytes(&result[0], length);
	}
	return result;
}

// ----------------------------------------------------------------------------
vector<WORD> ChalkStreamBinary::ReadBlob2(const string& elementname)
{
	vector<WORD> result;
	long length = m_Stream.ReadLong();
	if (length > 0)
	{
		result.resize(length);
		for (long i = 0; i != length; ++i)
		{
			result[i] = m_Stream.ReadWord();
		}
	}
	return result;
}

// ----------------------------------------------------------------------------
vector<DWORD> ChalkStreamBinary::ReadBlob4(const string& elementname)
{
	vector<DWORD> result;
	long length = m_Stream.ReadLong();
	if (length > 0)
	{
		result.resize(length);
		for (long i = 0; i != length; ++i)
		{
			result[i] = m_Stream.ReadLong();
		}
	}
	return result;
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::ReadElement1(const string& elementname)
{
	// Do nothing
}

// ----------------------------------------------------------------------------
long ChalkStreamBinary::ReadElement(const string& elementname)
{
	return m_Stream.ReadLong();
}

// ----------------------------------------------------------------------------
void ChalkStreamBinary::ReadEndElement(const string& elementname)
{
	// Do nothing
}
