// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     ChalkStream.h
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 23 Oct 2001 by Carl Muller
//
// Defines support for XML streams, FILE based
// ----------------------------------------------------------------------------

#include "filename.h"
#include "binarystream.h"
#include "bufferedfile.h"
#include "iff.h"

// ----------------------------------------------------------------------------
// Chalk stream interface
struct ChalkStream
{
	// Writing functions, called by ChalkObjects
	virtual void WriteElement1(const string& elementname) = 0;
	virtual void WriteElement(const string& elementname, long numkids) = 0;
	virtual void WriteEndElement(const string& elementname) = 0;
	virtual void WriteInteger(const string& elementname, long value) = 0;
	virtual void WriteReal(const string& elementname, double value) = 0;
	virtual void WriteString(const string& elementname, const wstring& value) = 0;
	virtual void WriteString(const string& elementname, const string& value) = 0;
	virtual void WriteGUID(const string& elementname, GUID value) = 0;
	virtual void WriteBlob1(const string& elementname, long length, const BYTE* value) = 0;
	virtual void WriteBlob2(const string& elementname, long length, const WORD* value) = 0;
	virtual void WriteBlob4(const string& elementname, long length, const DWORD* value) = 0;

	// Reading functions, called by ChalkObjects
	virtual void ReadElement1(const string& elementname) = 0;
	virtual long ReadElement(const string& elementname) = 0;
	virtual void ReadEndElement(const string& elementname) = 0;
	virtual long ReadInteger(const string& elementname, long defaultvalue = 0) = 0;
	virtual double ReadReal(const string& elementname, double defaultvalue = 0.0) = 0;
	virtual wstring ReadString(const string& elementname, const wstring& defaultvalue = L"") = 0;
	virtual GUID ReadGUID(const string& elementname) = 0;
	virtual vector<BYTE> ReadBlob1(const string& elementname) = 0;
	virtual vector<WORD> ReadBlob2(const string& elementname) = 0;
	virtual vector<DWORD> ReadBlob4(const string& elementname) = 0;
};

// ----------------------------------------------------------------------------
class XMLNode
{
public:
	XMLNode() : m_BeenRead(false) {}
	~XMLNode() { Clear(); }
	void Clear();
	void Read(BinaryStream& is, int ch);

public:
	bool m_BeenRead; // Has this node been processed?
	wstring m_Name;
	map<wstring, wstring> m_Attributes;
	vector<BYTE> m_Text; // May contain large data
	vector<XMLNode*> m_Kids;

private:
	// Helper functions
	XMLNode(const XMLNode& s);  // dummy copy constructor
	const XMLNode& operator= (const XMLNode& s);  // dummy assignment operator
};


// ----------------------------------------------------------------------------
// For saving XML files
class ChalkStreamText : public ChalkStream
{
public:
	ChalkStreamText(BinaryStream& BaseStream);

	// Writing functions, called by ChalkObjects
	void WriteElement1(const string& elementname);
	void WriteElement(const string& elementname, long numkids);
	void WriteEndElement(const string& elementname);
	void WriteInteger(const string& elementname, long value);
	void WriteReal(const string& elementname, double value);
	void WriteString(const string& elementname, const wstring& value);
	void WriteString(const string& elementname, const string& value);
	void WriteGUID(const string& elementname, GUID value);
	void WriteBlob1(const string& elementname, long length, const BYTE* value);
	void WriteBlob2(const string& elementname, long length, const WORD* value);
	void WriteBlob4(const string& elementname, long length, const DWORD* value);

	// Reading functions, called by ChalkObjects
	void ReadElement1(const string& elementname);
	long ReadElement(const string& elementname);
	void ReadEndElement(const string& elementname);
	long ReadInteger(const string& elementname, long defaultvalue = 0);
	double ReadReal(const string& elementname, double defaultvalue = 0.0);
	wstring ReadString(const string& elementname, const wstring& defaultvalue = L"");
	GUID ReadGUID(const string& elementname);
	vector<BYTE> ReadBlob1(const string& elementname);
	vector<WORD> ReadBlob2(const string& elementname);
	vector<DWORD> ReadBlob4(const string& elementname);

private:
	BinaryStream& m_Stream;
	XMLNode m_XML; // XML stream as loaded in
	vector<int> m_XMLindex; // Index to current XML node
	XMLNode* m_pXML; // Pointer to current XML node

	// Helper functions
	vector<BYTE> ReadRaw(const string& elementname, bool& bFailed);
	ChalkStreamText(const ChalkStreamText &s);  // dummy copy constructor
	const ChalkStreamText& operator= (const ChalkStreamText &s);  // dummy assignment operator
};


// ----------------------------------------------------------------------------
// For Undo buffers
class ChalkStreamBinary : public ChalkStream
{
public:
	ChalkStreamBinary(BinaryStream& BaseStream);

	// Writing functions, called by ChalkObjects
	void WriteElement1(const string& elementname);
	void WriteElement(const string& elementname, long numkids);
	void WriteEndElement(const string& elementname);
	void WriteInteger(const string& elementname, long value);
	void WriteReal(const string& elementname, double value);
	void WriteString(const string& elementname, const wstring& value);
	void WriteString(const string& elementname, const string& value);
	void WriteGUID(const string& elementname, GUID value);
	void WriteBlob1(const string& elementname, long length, const BYTE* value);
	void WriteBlob2(const string& elementname, long length, const WORD* value);
	void WriteBlob4(const string& elementname, long length, const DWORD* value);

	// Reading functions, called by ChalkObjects
	void ReadElement1(const string& elementname);
	long ReadElement(const string& elementname);
	void ReadEndElement(const string& elementname);
	long ReadInteger(const string& elementname, long defaultvalue = 0);
	double ReadReal(const string& elementname, double defaultvalue = 0.0);
	wstring ReadString(const string& elementname, const wstring& defaultvalue = L"");
	GUID ReadGUID(const string& elementname);
	vector<BYTE> ReadBlob1(const string& elementname);
	vector<WORD> ReadBlob2(const string& elementname);
	vector<DWORD> ReadBlob4(const string& elementname);

private:
	BinaryStream& m_Stream;
	// Helper functions
	ChalkStreamBinary(const ChalkStreamBinary &s);  // dummy copy constructor
	const ChalkStreamBinary& operator= (const ChalkStreamBinary &s);  // dummy assignment operator
};
