// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     Filename.cpp
// CREATED:  30 Apr 2000 by Carl Muller
// MODIFIED: 24 Oct 2001 by Carl Muller
//
// Implements the filename class
// 
// TBD! This class is not UNC aware yet (makeabsolute, makerelative, etc.)
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "filename.h"
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <direct.h>

// ----------------------------------------------------------------------------
// Widen 8 bits to 16 bit string - for use with element data - support UTF8
wstring WidenString(const vector<BYTE>& value)
{
	wstring temp;
	int by, ch;
	for (vector<BYTE>::const_iterator i = value.begin(); i != value.end(); ++i)
	{
		by = *i;
		if (by & 0x80)
		{
			if ((by & 0x40) == 0)
			{
				ch = 0; // Bad position for this char!
			}
			else if (by & 0x20) // 3 byte character
			{
				if (by & 0x10)
				{
					ch = 0; // We only support up to 16 bit chars
				}
				else
				{
					ch = (by & 0x0f) << 12;
					if (++i == value.end())
						break;
					by = *i;
					if ((by & 0xc0) != 0x80)
					{
						ch = 0; // Bad position for this char!
					}
					else
					{
						ch |= (by & 0x3f) << 6;
						if (++i == value.end())
							break;
						by = *i;
						if ((by & 0xc0) != 0x80)
						{
							ch = 0; // Bad position for this char!
						}
						else
						{
							ch |= (by & 0x3f);
						}
					}
				}
			}
			else // 2 byte character
			{
				ch = (by & 0x1f) << 6;
				if (++i == value.end())
					break;
				by = *i;
				if ((by & 0xc0) != 0x80)
				{
					ch = 0; // Bad position for this char!
				}
				else
				{
					ch |= (by & 0x3f);
				}
			}
		}
		else // 1 byte character
		{
			ch = by;
		}
		temp += ch;
	}
	return temp;
}


// ----------------------------------------------------------------------------
// Widen 8 bits to 16 bit string - for use with filenames and element names
wstring WidenString(const string& value)
{
	wstring temp;
	for (string::const_iterator i = value.begin(); i != value.end(); ++i)
	{
		temp += *i;
	}
	return temp;
}

// ----------------------------------------------------------------------------
// Narrow 16 bits to 8 bit string - for use with Win32 API function calls
string NarrowString(const wstring& value)
{
	string temp;
	for (wstring::const_iterator i = value.begin(); i != value.end(); ++i)
	{
		if (*i < 256) // Don't include characters outside code page 1252
			temp += *i;
	}
	return temp;
}

// ----------------------------------------------------------------------------
// Dodgy routine to make things upper-case, for comparing filenames safely
// or XML element names
wstring MakeUpper(const wstring& value)
{
	wstring outstr;
	int i, len;

	// Make it upper case
	len = value.length();
	for (i = 0; i < len; ++i)
	{
		// !!! does not handle accented characters or EBCDIC
		if (value[i] >= 'a' && value[i] <= 'z')
			outstr += ((value[i] - 'a') + 'A');
		else
			outstr += value[i];
	}
	return outstr;
}

// ----------------------------------------------------------------------------
// Special name compare (handles embedded numbers)
int CompareFileNames(LPCTSTR str1, LPCTSTR str2)
{
	long val1, val2;
	int ch1, ch2;

	while (*str1 && *str2)
	{
		if (isdigit(*str1) && isdigit(*str2))
		{
			// Do numeric compare
			val1 = 0;
			while (isdigit(*str1))
			{
				val1 = val1 * 10 + (*str1) - '0';
				str1++;
				if (val1 > 99999)
					break;
			}
			val2 = 0;
			while (isdigit(*str2))
			{
				val2 = val2 * 10 + (*str2) - '0';
				str2++;
				if (val2 > 99999)
					break;
			}
			if (val1 < val2)
				return -1;
			else if (val1 > val2)
				return 1;
		}
		else
		{
			// Normal string compare
			ch1 = toupper(*str1++);
			ch2 = toupper(*str2++);
			if (ch1 < ch2)
				return -1;
			else if (ch1 > ch2)
				return 1;
		}
	}
	// Run out of strings to compare
	if (*str1 == *str2) // Both have finished
		return 0;
	else if (*str1) // Second string ran out first
		return 1;
	else
		return -1;
}

// ----------------------------------------------------------------------------
// CFileName class
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
void CFileName::Clear()
{
	m_Drive.resize(0); // .clear when STL is supported properly by Microsoft
	m_Dir.resize(0); // .clear when STL is supported properly by Microsoft
	m_Name.resize(0); // .clear when STL is supported properly by Microsoft
	m_Ext.resize(0); // .clear when STL is supported properly by Microsoft
}

// ----------------------------------------------------------------------------
void CFileName::Set(const string& filename)
{
	Set(WidenString(filename));
}

// ----------------------------------------------------------------------------
void CFileName::Set(const wstring& filename)
{
	wstring S(filename);

	// Remove the extension
	{
		int len = S.length();
		int lastslash = S.find_last_of('\\');
		int lastdot = S.find_last_of('.');
		if ((lastdot >= 0) && (lastdot < len) && (lastdot > lastslash))
		{
			m_Ext = S.substr(lastdot);
			S = S.substr(0, lastdot);
		}
	}
	// Remove the drive
	if (S.length() > 2)
	{
		if (S[0] == '\\' && S[1] == '\\')
		{
			// UNC path name e.g. \\COMPUTER\SHARED
			// ...
		}
		else if (S[1] == ':')
		{
			// Drive name e.g. D:
			m_Drive = S.substr(0, 2);
			S = S.substr(2);
		}
	}
	// Remove the path
	{
		int len = S.length();
		int lastslash = S.find_last_of('\\');
		if ((lastslash >= 0) && (lastslash + 1 < len))
		{
			m_Name = S.substr(lastslash + 1);
			m_Dir = S.substr(0, lastslash + 1);
		}
	}
}

// ----------------------------------------------------------------------------
void CFileName::AddDir(const wstring& newdir)
{
	if (newdir.compare(L".\\") == 0)
		; // Do nothing
	else if (newdir.compare(L"..\\") == 0)
	{
		// Go up a directory
		int i, len;

		len = m_Dir.length();
		if (len > 1)
		{
			m_Dir = m_Dir.substr(0, len - 1);
		}
		for (i = len - 2; i >= 0; --i)
		{
			if (m_Dir[i] == '\\')
				break;
			else
				m_Dir = m_Dir.substr(0, i);
		}
	}
	else
	{
		m_Dir += newdir;
	}
}

// ----------------------------------------------------------------------------
void CFileName::MakeAbsolute()
{
	wchar_t basepath[_MAX_PATH];

	_wgetcwd(basepath, _MAX_PATH - 2);
	if (basepath[wcslen(basepath)-1] != '\\')
		wcscat(basepath, L"\\");

	wstring newpath = Get();
	Set(CFileName::MakeAbsolute(basepath, newpath));
}

// ----------------------------------------------------------------------------
// Static functions
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
wstring CFileName::MakeRelative(const wstring& basepath, const wstring& newpath)
{
	int i, j, k;
	bool match;
	wchar_t temp1[_MAX_PATH];
	wchar_t temp2[_MAX_PATH];
	wchar_t outstr[_MAX_PATH];

	// Are the filenames the same?
	match = false;
	for (k = basepath.length() - 1; k > 0; --k)
		if (basepath[k] == '\\')
			break;
	for (j = 0; j < _MAX_PATH; ++j)
	{
		if (basepath[j] != newpath[j])
		{
			match = false;
			break;
		}
		else if ((basepath[j] == '.') && (j > k)) // Matching name
		{
			match = true;
			break;
		}
		else if (basepath[j] == 0) // Identical
		{
			match = false;
			break;
		}
	}
	if (match) // Matching paths and filenames
	{
		wcscpy(outstr, L"#");
		wcscat(outstr, &newpath[j]);
	}
	else if (basepath[0] != newpath[0]) // Drive letters are different
	{
		wcscpy(outstr, newpath.c_str());
	}
	else
	{
		// Find the paths
		wcscpy(temp1, basepath.c_str());
		for (i = wcslen(temp1) - 1; i >= 0; --i)
			if (temp1[i] == '\\')
				break;
			else
				temp1[i] = 0;

		wcscpy(temp2, newpath.c_str());
		for (i = wcslen(temp2) - 1; i >= 0; --i)
			if (temp2[i] == '\\')
				break;
			else
				temp2[i] = 0;

		// Are the paths the same?
		wcscpy(outstr, L"");
		if (wcscmp(temp1, temp2) == 0) // Paths are the same
		{
			j = wcslen(temp1);
		}
		else // Paths are different
		{
			j = 2; // Drives are the same
			// Find different bits
			for (i = 0; i < wcslen(temp1); ++i)
			{
				if ((temp1[i] == '\\') && (temp2[i] == '\\'))
					j = i + 1;
				else
					if (temp1[i] != temp2[i])
						break;
			}
			if (j > 3)
			{
				for (i = j; i < wcslen(temp1); ++i)
					if (temp1[i] == '\\')
						wcscat(outstr, L"..\\");
			}
			else
				j = 2;
		}
		wcscat(outstr, newpath.substr(j).c_str());
	}
	wstring temp(outstr);
	return temp;
}

// ----------------------------------------------------------------------------
template<class stringtype>
	wstring MakeAbsoluteHelper(const wstring& basepath, const stringtype& offset)
{
	int i, j;
	wstring basefilename;
	wstring outstr;

	if (offset.length() == 0) // Empty offset
	{
		; // Empty answer
	}
	else if ((offset[0] && offset[1] == ':') // Drive letter makes it absolute
		|| (offset[0] == '\\' && offset[1] == '\\')) // UNC makes it absolute
	{
		// Copy the offset as the full path
		for (i = 0; i < offset.length(); ++i)
			outstr += offset[i];
	}
	else
	{
		// Find path of the file
		j = 0;
		i = basepath.find_last_of('\\');

		// Find the filename
		if (i > 0)
		{
			outstr = basepath.substr(0, i+1);
			basefilename = basepath.substr(i+1);
		}
		i = basefilename.find_last_of('.');
		if (i > 0)
		{
			basefilename = basefilename.substr(0, i);
		}
		else
		{
			basefilename.empty();
		}

		// Is it from the root directory?
		if (offset[0] == '\\')
		{
			outstr = outstr.substr(0, 2); // Leave only the drive letter
		}
		else
		{
			// Parse the ..\ and .\ paths
			for (;;)
			{
				if ((offset[j] == '.') && (offset[j+1] == '.') &&
				  	(offset[j+2] == '\\'))
				{
					int len = outstr.length();
					if (len > 1)
					{
						outstr = outstr.substr(0, len - 1);
						i = outstr.find_last_of('\\', len - 1);
						if (i > 0)
						{
							outstr = outstr.substr(0, i+1);
						}
					}
					j += 3;
				}
				else if ((offset[j] == '.') && (offset[j+1] == '\\'))
				{
					j += 2;
				}
				else
					break;
			}
		}

		// Add the remaining offset to the name
		while (offset[j])
		{
			if (offset[j] == '#') // Same filename
			{
				outstr.append(basefilename);
			}
			else
			{
				outstr += offset[j];
			}
			++j;
		}
	}

	// Make it upper case
	return MakeUpper(outstr);
}

// ----------------------------------------------------------------------------
wstring CFileName::MakeAbsolute(const wstring& basepath, const wstring& offset)
{
	return MakeAbsoluteHelper(basepath, offset);
}

// ----------------------------------------------------------------------------
wstring CFileName::MakeAbsolute(const wstring& basepath, const string& offset)
{
	return MakeAbsoluteHelper(basepath, offset);
}

// ----------------------------------------------------------------------------
// Do the extensions match?
bool CFileName::HasExt(const wstring& ext) const
{
	return (MakeUpper(m_Ext).compare(MakeUpper(ext)) == 0);
}

// ----------------------------------------------------------------------------
// Do the extensions match?
bool CFileName::HasExt(const string& ext) const
{
	return HasExt(WidenString(ext));
}

// ----------------------------------------------------------------------------
long CFileName::FileSize(const wstring& filename)
{
	long filelength = -1;
	int filehandle = ::_open(NarrowString(filename).c_str(), O_RDONLY | O_BINARY);
	if (filehandle >= 0)
	{
		filelength = _filelength(filehandle);
		close(filehandle);
	}
	return filelength;
}

// ----------------------------------------------------------------------------
bool CFileName::FileLocked(const wstring& filename)
{
	bool ok = false;
	int filehandle = ::_open(NarrowString(filename).c_str(), O_RDWR | O_APPEND | O_BINARY);
	if (filehandle >= 0)
	{
		ok = true;
		close(filehandle);
	}
	return ok;
}

// ----------------------------------------------------------------------------
bool CFileName::FileExists(const wstring& filename)
{
	bool ok = false;
	int filehandle = ::_open(NarrowString(filename).c_str(), O_RDONLY | O_BINARY);
	if (filehandle >= 0)
	{
		ok = true;
		close(filehandle);
	}
	return ok;
}

// ----------------------------------------------------------------------------
bool CFileName::HasWildcard(const wstring& filename)
{
	return (filename.find_first_of(L"*?") != wstring.npos);
}
