// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     Iff.cpp
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 16 Oct 2001 by Carl Muller
//
// This file implements a file class that supports fast reading and
// writing of binary files, with special support for IFF format files.
// (See the Electronic Arts Inc. / Commodore Amiga Ltd.
// Interchange File Format 1985 specification for more details)
// (Microsoft has loosely based their RIFF file format on the IFF specification)
// ----------------------------------------------------------------------------

#include "stdafx.h"
#include "filename.h"
#include "binarystream.h"
#include "bufferedfile.h"
#include "iff.h"
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>


// ----------------------------------------------------------------------------
// IFF file
// ----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
// Create an IFF file variable
IFFfile::IFFfile() :
	m_Align(true),
	m_Intel(false)
{
}


// ----------------------------------------------------------------------------
// Destroy a IFF file variable
IFFfile::~IFFfile()
{
}

// ----------------------------------------------------------------------------
// Open an IFF file for writing
HRESULT IFFfile::OpenWrite(const CFileName& filename, int newbufsize)
{
	HRESULT hr = BufferedFile::OpenWrite(filename, newbufsize);
	m_PosnStack.clear();
	m_Align = true;
	m_Intel = false;
	return hr;
}


// ----------------------------------------------------------------------------
// Open an IFF file for reading
HRESULT IFFfile::OpenRead(const CFileName& filename, int newbufsize)
{
	HRESULT hr = BufferedFile::OpenRead(filename, newbufsize);
	m_PosnStack.clear();
	m_Align = true;
	m_Intel = false;
	return hr;
}


// ----------------------------------------------------------------------------
void IFFfile::Form(DWORD formname)
{
	WriteLong(CHUNK_FORM);
	Push(GetPos());
	WriteLong(0);
	WriteLong(formname);
}


// ----------------------------------------------------------------------------
void IFFfile::Riff(DWORD formname)
{
	m_Align = false;
	m_Intel = true;

	WriteLong(CHUNK_RIFF);
	Push(GetPos());
	WriteLong(0);
	WriteLong(formname);
}


// ----------------------------------------------------------------------------
void IFFfile::EndForm()
{
	long pos1, pos2, len;

	pos2 = GetPos();
	pos1 = Pop();
	len = pos2 - pos1 - 4L;
	SetPos(pos1);
	if (m_Intel)
	{
		WriteIntelLong(len);
	}
	else
	{
		WriteLong(len);
	}
	SetPos(pos2);
	if (m_Align)
	{
		if (pos2 & 1)
		{
			WriteByte(0);
			pos2++;
		}
	}
}


// ----------------------------------------------------------------------------
void IFFfile::Chunk(DWORD chunkname)
{
	WriteLong(chunkname);
	Push(GetPos());
	WriteLong(0);
}


// ----------------------------------------------------------------------------
void IFFfile::EndChunk()
{
	long pos1, pos2, len;

	pos2 = GetPos();
	pos1 = Pop();
	len = pos2 - pos1 - 4L;
	SetPos(pos1);
	if (m_Intel)
	{
		WriteIntelLong(len);
	}
	else
	{
		WriteLong(len);
	}
	SetPos(pos2);
	if (m_Align)
	{
		if (pos2 & 1)
		{
			WriteByte(0);
			pos2++;
		}
	}
}
