// ----------------------------------------------------------------------------
// CHALK (c) 2000, 2001 Carl Muller.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of version 2 of the GNU General Public License
// as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program, as the file license.txt; if not, write to
// the Free Software Foundation, Inc.,
// 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// ----------------------------------------------------------------------------
//
// FILE:     Iff.h
// CREATED:  28 Apr 2000 by Carl Muller
// MODIFIED: 21 Mar 2001 by Carl Muller
//
// This file defines a file class that supports fast reading and
// writing of binary files, with special support for IFF format files.
// (See the Electronic Arts Inc. / Commodore Amiga Ltd.
// Interchange File Format 1985 specification for more details)
// (Microsoft has loosely based their RIFF file format on the IFF specification)
// ----------------------------------------------------------------------------

// Standard bitmap enumerations
enum IffMasking {
	mskNone = 0,
	mskHasMask = 1,
	mskHasTransparentColor = 2,
	mskLasso = 3
};

enum IffCompression {
	cmpNone = 0,
	cmpByteRun1 = 1
};

// Construct an IFF chunk name (68000 format DWORD)
#define MAKE_IFF(a,b,c,d) (DWORD) ((DWORD)(a)<<24|(DWORD)(b)<<16|(DWORD)(c)<<8|(DWORD)(d))

// Master file format chunks
const DWORD CHUNK_FORM = MAKE_IFF('F','O','R','M');
const DWORD CHUNK_RIFF = MAKE_IFF('R','I','F','F');
const DWORD CHUNK_RIFX = MAKE_IFF('R','I','F','X');

// Standard bitmap chunks
const DWORD CHUNK_ILBM = MAKE_IFF('I','L','B','M');
const DWORD CHUNK_PBM  = MAKE_IFF('P','B','M',' ');
const DWORD CHUNK_CMAP = MAKE_IFF('C','M','A','P');
const DWORD CHUNK_CRNG = MAKE_IFF('C','R','N','G');
const DWORD CHUNK_BMHD = MAKE_IFF('B','M','H','D');
const DWORD CHUNK_BODY = MAKE_IFF('B','O','D','Y');

// Standard sound sample chunks
const DWORD CHUNK_8SVX = MAKE_IFF('8','S','V','X');
const DWORD CHUNK_VHDR = MAKE_IFF('V','H','D','R');
const DWORD CHUNK_AIFF = MAKE_IFF('A','I','F','F');
const DWORD CHUNK_SSND = MAKE_IFF('S','S','N','D');

// Custom bitmap chunks
const DWORD CHUNK_SYST = MAKE_IFF('S','Y','S','T');
const DWORD CHUNK_OFST = MAKE_IFF('O','F','S','T');

// Custom tiled map chunks
const DWORD CHUNK_TMAP = MAKE_IFF('T','M','A','P');
const DWORD CHUNK_THDR = MAKE_IFF('T','H','D','R');
const DWORD CHUNK_COLR = MAKE_IFF('C','O','L','R');
const DWORD CHUNK_ATTR = MAKE_IFF('A','T','T','R');
const DWORD CHUNK_KIDS = MAKE_IFF('K','I','D','S');

// Custom placed object chunks
const DWORD CHUNK_BOXS = MAKE_IFF('B','O','X','S');
const DWORD CHUNK_FRAK = MAKE_IFF('F','R','A','K');
const DWORD CHUNK_POSN = MAKE_IFF('P','O','S','N');
const DWORD CHUNK_LINE = MAKE_IFF('L','I','N','E');
const DWORD CHUNK_PATH = MAKE_IFF('P','A','T','H');
const DWORD CHUNK_GRID = MAKE_IFF('G','R','I','D');
const DWORD CHUNK_ANIM = MAKE_IFF('A','N','I','M');
const DWORD CHUNK_DEFN = MAKE_IFF('D','E','F','N');
const DWORD CHUNK_DKID = MAKE_IFF('D','K','I','D');

const DWORD CHUNK_PXXP = MAKE_IFF('P','X','X','P');
const DWORD CHUNK_PHDR = MAKE_IFF('P','H','D','R');
const DWORD CHUNK_GRAB = MAKE_IFF('G','R','A','B');
const DWORD CHUNK_FIXD = MAKE_IFF('F','I','X','D');


// ----------------------------------------------------------------------------
class IFFfile : public BufferedFile
{
public:
	IFFfile();
	~IFFfile();
	HRESULT OpenWrite(const CFileName& filename, int newbufsize = 4096); // 0 = OK
	HRESULT OpenRead(const CFileName& filename, int newbufsize = 4096);  // 0 = OK

	void Form(DWORD formname);
	void Riff(DWORD formname);
	void EndForm();
	void Chunk(DWORD formname);
	void EndChunk();

private:
	IFFfile(const IFFfile &s);  // dummy copy constructor
	const IFFfile& operator= (const IFFfile&s);  // dummy assignment operator

	void Push(long it) { m_PosnStack.push_back(it); }
	long Pop()
	{
		long pos = m_PosnStack.back();
		m_PosnStack.pop_back();
		return pos;
	}

private:
	bool m_Align, m_Intel; // Support RIFF as well as IFF
	vector<long> m_PosnStack;
};
