; --------------------------------------
; SUGARLOAF DEMO
; 2003-08-31
;
; Freeware for all uses
;
; Simple technology demo of the big-endian 6502
; emulator and simulated XGS GPU.
;
; Written by kiwi
; Photo taken by kiwi
; Ball object from royalty-free clip art
;
; Note that the collision detection does not take
; angles into account even though the data supports
; this - the balls simply reverse direction on
; hitting the mountain.
; --------------------------------------

; --------------------------------------
; API CONSTANTS

; Hook function
API_Hook:               .EQ $FFFF

; General API Functions
API_Exit:               .EQ $01
API_Get_Rand:           .EQ $02
API_Get_Key_State:      .EQ $03

; GPU Functions
GPU_Load_Sprites:       .EQ $0F
GPU_Set_Color:          .EQ $04
GPU_Draw_Point:         .EQ $05
GPU_Draw_Line:          .EQ $06
GPU_Draw_String:        .EQ $07 ; debug
GPU_Draw_Sprite:        .EQ $08
GPU_Draw_Sprite_16x16:  .EQ $09
GPU_Draw_Screen:        .EQ $0A
GPU_Lock_Video:         .EQ $0B
GPU_Unlock_Video:       .EQ $0C
GPU_Clear_Screen:       .EQ $0D
GPU_Flip:               .EQ $0E

; Directions
KEY_UP:                 .EQ $C8
KEY_DOWN:               .EQ $D0
KEY_LEFT:               .EQ $CB
KEY_RIGHT:              .EQ $CD
KEY_SPACE:              .EQ $39

; --------------------------------------
; Demo constants

; Object allocation
OBJNUM_MIN:             .EQ 0
OBJNUM_MAX:             .EQ 8   ; Requires manual support

; Main character
BALL_WIDTH:             .EQ 16
BALL_HEIGHT:            .EQ 16
BALL_WIDTH_HALF:        .EQ 8
BALL_HEIGHT_HALF:       .EQ 8

; Game World
MAX_SCREEN:             .EQ 0
MAX_TILES:              .EQ 64
SCREEN_WIDTH:           .EQ 16 ; In tiles
SCREEN_HEIGHT:          .EQ 16 ; In tiles

GRAVITY:                .EQ 5


; --------------------------------------
; ZERO PAGE DATA 


; Reserve 128 bytes to make my emulator easier to write
; by not clashing with other hardware

           .ORG $80

key_map:    .DB $FF, $00	; Emulator opposite endian from 6502
temp_ptr:
temp_ptr_h: .DB 0	; Emulator opposite endian from 6502
temp_ptr_l: .DB 0
temp_x_l:   .DB 0
temp_x_h:   .DB 0
temp_y_l:   .DB 0
temp_y_h:   .DB 0


; --------------------------------------
; DATA 

        .ORG $600

; Global parameter passing area
; Note that the API (and processor) is big-endian unlike the 6502

api_func:   .DB 0
p_0:        .DB 0
p_1:        .DB 0
p_2:        .DB 0
p_3:        .DB 0
p_x_0:      .DB 0
p_y_0:      .DB 0
p_x_1:      .DB 0
p_y_1:      .DB 0
p_r_0:      .DB 0
p_g_0:      .DB 0
p_b_0:      .DB 0

; Global return value passing area
r_0:        .DB 0
r_1:        .DB 0
r_2:        .DB 0
r_3:        .DB 0

; --------------------------------------
; Temporary registers
tr_0:       .DB 0
tr_1:       .DB 0
tr_2:       .DB 0
tr_3:       .DB 0

; Misc. flags
key_press:      .DB 0

; --------------------------------------
; Object information
; Because the 6502 has only 8 bit index registers,
; it is better to use separate arrays of variables
; rather than an array of structures
;
; Note: we REALLY need a .DS command, or assemble-time maths support

obj_x:          .DB 32,64,96,128,160,192,240,150
obj_y:          .DB 32,32,32,32,32,32,32,50
obj_x_l:        .DB 0,0,0,0,0,0,0,0
obj_y_l:        .DB 0,0,0,0,0,0,0,0
obj_xv:         .DB 0,1,$ff,0,0,0,$ff,0
obj_yv:         .DB 0,0,0,0,0,0,0,0
obj_xv_l:       .DB 0,0,$80,0,$80,$40,$c0,0
obj_yv_l:       .DB 0,0,0,0,0,0,0,0

; Game world status
curr_screen:    .DB 0


; --------------------------------------
; CODE 
; --------------------------------------

        .ORG $A00

; --------------------------------------
PROGRAM_START:
        ; Setup 6502 to known state
;       SEI
;       CLD
;       LDX #$ff
;       TXS

        ; Load tile bitmaps into the
        ; GPU sprite bank
        LDA #tile_base.h
        STA p_0
        LDA #tile_base.l
        STA p_1
        LDA #MAX_TILES
        STA p_2
        LDA #GPU_Load_Sprites
        STA api_func
        JSR API_Hook

main_loop:

        ; Draw the tile map background
        LDA curr_screen
        CLC
        ADC #screen_base.h
        STA p_0
        LDA #screen_base.l
        STA p_1
        LDA #SCREEN_WIDTH
        STA p_x_0
        LDA #SCREEN_HEIGHT
        STA p_y_0
        LDA #GPU_Draw_Screen
        STA api_func
        JSR API_Hook

        ; Lock the video buffer for sprite drawing
        LDA #GPU_Lock_Video
        STA api_func
        JSR API_Hook

        ; Draw balls
        LDX #OBJNUM_MIN
draw_loop:
        LDA obj_x,X
        SEC
        SBC #BALL_WIDTH_HALF
        STA p_x_0
        LDA obj_y,X
        SEC
        SBC #BALL_HEIGHT_HALF
		CMP #224
		BCS dont_crash	; Emulator crashes if sprites are too low
		CMP #16
		BCC dont_crash  ; Emulator might crash if sprites are too high
        STA p_y_0
        LDA #spr_ball_0.h
        STA p_0
        LDA #spr_ball_0.l
        STA p_1
        LDA #GPU_Draw_Sprite_16x16
        STA api_func
        JSR API_Hook
dont_crash:
        INX
        ; CPX instruction causes emulator to crash!
        TXA
        CMP #OBJNUM_MAX
        BCC draw_loop

        ; Unlock the video buffer from sprite drawing
        LDA #GPU_Unlock_Video
        STA api_func
        JSR API_Hook

        ; Blit the frame buffer to the screen
		; (Hopefully this synchronises the frame)
        LDA #GPU_Flip
        STA api_func
        JSR API_Hook

        ; Clear the new frame buffer to
        ; the blue sky color
        LDA #%00101011
        STA p_0
        LDA #%10111111
        STA p_1
        LDA #GPU_Clear_Screen
        STA api_func
        JSR API_Hook

        ; Grab the keyboard status
        LDA #API_Get_Key_State
        STA api_func
        JSR API_Hook

        LDA #0
        STA key_press

        ; Check for the exit key (esc)
        LDY #$01
        LDA ( key_map ), Y
        BEQ skip_exit

        ; Terminate the program
        LDA #API_Exit
        STA api_func
        JSR API_Hook
skip_exit:

        ; Move the objects
        LDX #OBJNUM_MIN
move_loop:
        ; Move horizontally
        LDA obj_x_l,X
        STA temp_x_l
        CLC
        ADC obj_xv_l,X
        STA obj_x_l,X
        LDA obj_x,X
        STA temp_x_h
        ADC obj_xv,X
        STA obj_x,X

        ; Add gravity
        LDA obj_yv_l,X
        CLC
        ADC #GRAVITY
        STA obj_yv_l,X
        LDA obj_yv,X
        ADC #0
        STA obj_yv,X

        ; Move vertically
        LDA obj_y_l,X
        STA temp_y_l
        CLC
        ADC obj_yv_l,X
        STA obj_y_l,X
        LDA obj_y,X
        STA temp_y_h
        ADC obj_yv,X
        STA obj_y,X

        ; Check for collision
        JSR check_collision
        BCC no_collision

        ; Reverse movement
		LDA temp_x_h
		STA obj_x,X
		LDA temp_x_l
		STA obj_x_l,X
		LDA temp_y_h
		STA obj_y,X
		LDA temp_y_l
		STA obj_y_l,X

        ; Bounce backwards
		; The "SBC" instruction is wrong in the emulator
;        LDA #0
;        SEC
;        SBC obj_yv_l,X
;        STA obj_yv_l,X
;        LDA #0
;        SBC obj_yv,X
;        STA obj_yv,X

		LDA obj_yv_l,X
		EOR #$ff
		SEC
		ADC #0
		STA obj_yv_l,X
		LDA obj_yv,X
		EOR #$ff
		ADC #0
		STA obj_yv,X

;        LDA #0
;        SEC
;        SBC obj_xv_l,X
;        STA obj_xv_l,X
;        LDA #0
;        SBC obj_xv,X
;        STA obj_xv,X

		LDA obj_xv_l,X
		EOR #$ff
		SEC
		ADC #0
		STA obj_xv_l,X
		LDA obj_xv,X
		EOR #$ff
		ADC #0
		STA obj_xv,X

no_collision:
        INX
        ; CPX instruction causes emulator to crash!
        TXA
        CMP #OBJNUM_MAX
        BCC jmove_loop	; Jump past -128 bounds causes emulator to crash!

        JMP main_loop

jmove_loop:
		JMP move_loop


; --------------------------------------
; Check for collision
; Input: X = index number of object
; Output: Carry set if collided, clear otherwise
;         A = the angle of the scenery collided with
; Preserves X, Destroys Y
check_collision:
        ; Assumes the screen is 32*32 chars of 8*8 pixels

        ; Load the character number into A
        ; A = collision_map[(x/8)+(y/8)*32]
        lda #0
        sta temp_ptr_h
        lda obj_y,X
        and #$F8
        asl
        rol temp_ptr_h
        asl
        rol temp_ptr_h
        adc #collision_map.l
        sta temp_ptr_l
        lda temp_ptr_h
        adc #collision_map.h
        sta temp_ptr_h
        lda obj_x,X
        lsr
        lsr
        lsr
        tay
        lda (temp_ptr),Y

        ; Load the angle into A
        ; A = collision_tiles[a*64+(x&7)+(y&7)*8]
        ldy #0
        sty temp_ptr_h
        asl
        rol temp_ptr_h
        asl
        rol temp_ptr_h
        asl
        rol temp_ptr_h
        asl
        rol temp_ptr_h
        asl
        rol temp_ptr_h
        asl
        rol temp_ptr_h
        sta temp_ptr_l

        lda obj_y,X
        and #7
        asl
        asl
        asl
        ora temp_ptr_l
        clc
        adc #collision_tiles.l
        sta temp_ptr_l
        lda temp_ptr_h
        adc #collision_tiles.h
        sta temp_ptr_h
        lda obj_x,X
        and #7
        tay
        lda (temp_ptr),Y

        ; Check the result
        cmp #1 ; Carry set if solid
        rts



; --------------------------------------
; Data
; --------------------------------------

        .ORG $4000

; --------------------------------------
; Collision map for the screen (8*8 tiles)
collision_map:
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$01,$02,$03,$04
	.DB $05,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$06,$07,$08,$09,$0A,$0B
	.DB $0C,$0D,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$0E,$0F,$10,$11,$12,$13
	.DB $14,$15,$16,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$17,$18,$19,$1A,$00,$00,$00
	.DB $1B,$1C,$1D,$1E,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$1F,$20,$21,$22,$00,$00,$00,$00
	.DB $23,$24,$25,$26,$27,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$28,$29,$2A,$2B,$2C,$00,$00,$00,$00
	.DB $00,$2D,$2E,$2F,$30,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$31,$32,$33,$34,$35,$00,$00,$00,$00,$00
	.DB $00,$00,$36,$37,$38,$1E,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$39,$3A,$3B,$3C,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$3D,$3E,$3F,$40,$41,$00,$00,$00,$00,$00,$00,$00,$00

	.DB $00,$00,$00,$00,$00,$00,$42,$43,$44,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$2D,$45,$46,$47,$48,$49,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$4A,$4B,$4C,$4D,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$4E,$4F,$50,$51,$52,$53,$54,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$55,$56,$57,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$58,$59,$5A,$5B,$5C,$5D,$5E,$5F,$00,$00
	.DB $00,$00,$00,$00,$4A,$4B,$4C,$60,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$61,$62,$63,$64,$65,$66,$67
	.DB $00,$00,$00,$00,$55,$56,$68,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$69,$6A,$6B,$6C,$6D
	.DB $00,$00,$00,$6E,$6F,$70,$71,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$72,$73,$74
	.DB $00,$00,$75,$76,$77,$78,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$79
	.DB $7A,$7B,$7C,$7D,$7E,$7F,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

	.DB $80,$80,$80,$81,$82,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

; --------------------------------------
; Collision tiles (0=free, 01..ff = angle north..east..etc.)
collision_tiles:
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$FA,$FA,$00,$00,$00,$00,$F0,$F5,$FA,$FA

	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$FA,$FA,$FA,$FA,$FA
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA

	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$FA,$FA
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA

	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$01,$00,$00,$00,$FA,$FA,$FA,$FB,$01,$07,$0D,$12
	.DB $FA,$FA,$FA,$FA,$01,$08,$0F,$14,$FA,$FA,$FA,$FA,$01,$0A,$11,$18
	.DB $FA,$FA,$FA,$FA,$01,$0B,$14,$1C,$FA,$FA,$FA,$FA,$01,$0F,$19,$1F
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $19,$1E,$00,$00,$00,$00,$00,$00,$1D,$1F,$1F,$00,$00,$00,$00,$00
	.DB $1F,$1F,$1F,$1F,$00,$00,$00,$00,$1F,$1F,$1F,$1F,$1F,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$EC,$00,$00,$00,$00,$00,$00,$E4,$E9
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$EC,$EC,$00,$00,$00,$00,$00,$EC,$EC,$EC
	.DB $00,$00,$00,$EC,$EC,$EC,$EC,$EC,$00,$EC,$EC,$EC,$EC,$EC,$EC,$EC
	.DB $EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$EC
	.DB $00,$00,$EC,$EC,$EE,$F3,$FA,$FA,$EC,$EC,$EC,$EC,$EC,$F1,$F8,$FA
	.DB $EC,$EC,$EC,$EC,$EC,$EE,$F7,$FA,$EC,$EC,$EC,$EC,$EC,$EC,$F3,$FA
	.DB $EC,$EC,$EC,$EC,$EC,$EC,$EE,$FA,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$FA
	.DB $EC,$EC,$EC,$EC,$EC,$EC,$EC,$FA,$EC,$EC,$EC,$EC,$EC,$EC,$6C,$6C
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$7A,$7A,$7A
	.DB $7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA
	.DB $FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$FA,$7A
	.DB $FA,$FA,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A
	.DB $7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A
	.DB $FA,$FA,$FA,$FA,$01,$14,$1F,$1F,$FA,$FA,$FA,$FA,$01,$1F,$1F,$1F
	.DB $FA,$FA,$FA,$FA,$1F,$1F,$1F,$1F,$7A,$7A,$7A,$7A,$9F,$9F,$1F,$1F
	.DB $7A,$7A,$7A,$9F,$9F,$9F,$9F,$1F,$7A,$7A,$9F,$9F,$9F,$9F,$9F,$9F
	.DB $7A,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F
	.DB $1F,$1F,$1F,$1F,$1F,$1F,$00,$00,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$00
	.DB $1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F
	.DB $1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F,$1F
	.DB $9F,$1F,$1F,$1F,$1F,$1F,$1F,$21,$9F,$9F,$1F,$1F,$1F,$1F,$21,$24
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $1F,$00,$00,$00,$00,$00,$00,$00,$1F,$1F,$00,$00,$00,$00,$00,$00
	.DB $1F,$21,$24,$00,$00,$00,$00,$00,$21,$24,$24,$24,$00,$00,$00,$00
	.DB $24,$24,$24,$24,$24,$00,$00,$00,$24,$24,$24,$24,$24,$24,$00,$00
	.DB $00,$00,$00,$00,$00,$DD,$E1,$E5,$00,$00,$00,$00,$00,$D8,$DC,$E1
	.DB $00,$00,$00,$D3,$D3,$D3,$D7,$DB,$00,$00,$00,$D3,$D3,$D3,$D3,$D3
	.DB $00,$00,$D3,$D3,$D3,$D3,$D3,$D3,$00,$00,$D3,$D3,$D3,$D3,$D3,$D3
	.DB $00,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$00,$D3,$D3,$D3,$D3,$D3,$D3,$D3
	.DB $EA,$EC,$EC,$EC,$EC,$EC,$EC,$EC,$E6,$EC,$EC,$EC,$EC,$EC,$EC,$EC
	.DB $E1,$E9,$EC,$EC,$EC,$EC,$EC,$EC,$D8,$E1,$EC,$EC,$EC,$EC,$EC,$6C
	.DB $D3,$D3,$E1,$EC,$EC,$6C,$6C,$6C,$D3,$D3,$D3,$EC,$6C,$6C,$6C,$6C
	.DB $D3,$D3,$D3,$53,$6C,$6C,$6C,$6C,$D3,$D3,$D3,$53,$53,$6C,$6C,$6C
	.DB $EC,$EC,$EC,$EC,$6C,$6C,$6C,$6C,$EC,$EC,$6C,$6C,$6C,$6C,$6C,$6C
	.DB $EC,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C
	.DB $6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C
	.DB $6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$6C,$00,$00
	.DB $7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A
	.DB $7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A
	.DB $6C,$7A,$7A,$7A,$7A,$7A,$00,$00,$6C,$76,$71,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $7A,$7A,$7A,$7A,$7A,$7A,$7A,$9F,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A
	.DB $7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$7A,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F,$9F
	.DB $7A,$00,$9F,$9F,$9F,$9F,$9F,$9F,$00,$00,$00,$9F,$9F,$9F,$9F,$9F
	.DB $00,$00,$00,$00,$9F,$9F,$9F,$9F,$00,$00,$00,$00,$00,$9F,$9F,$9F
	.DB $00,$00,$00,$00,$00,$00,$9F,$A4,$00,$00,$00,$00,$00,$00,$00,$A4
	.DB $9F,$9F,$9F,$1F,$1F,$21,$24,$24,$9F,$9F,$9F,$9F,$24,$24,$24,$24
	.DB $9F,$9F,$9F,$A4,$A4,$24,$24,$24,$9F,$9F,$A4,$A4,$A4,$A4,$24,$24
	.DB $9F,$A4,$A4,$A4,$A4,$A4,$A4,$24,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4
	.DB $A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4
	.DB $24,$24,$24,$24,$24,$24,$00,$00,$24,$24,$24,$24,$24,$24,$24,$00
	.DB $24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24
	.DB $24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24,$24
	.DB $A4,$24,$24,$24,$24,$24,$24,$24,$A4,$A4,$24,$24,$24,$24,$24,$24
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$24,$00,$00,$00,$00,$00,$00,$00
	.DB $24,$24,$00,$00,$00,$00,$00,$00,$24,$24,$24,$00,$00,$00,$00,$00
	.DB $24,$24,$21,$21,$00,$00,$00,$00,$24,$21,$21,$21,$21,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$D3,$00,$00,$00,$00,$00,$00,$00,$D3
	.DB $00,$00,$00,$00,$00,$00,$00,$D3,$00,$00,$00,$00,$00,$E6,$D3,$D3
	.DB $00,$00,$00,$00,$E6,$E6,$D3,$D3,$00,$00,$00,$E6,$E6,$E6,$E6,$D3
	.DB $D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3
	.DB $D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3
	.DB $D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$D3
	.DB $D3,$D3,$D3,$D3,$D3,$D3,$D3,$53,$D3,$D3,$D3,$D3,$D3,$D3,$D3,$53
	.DB $D3,$D3,$53,$53,$53,$6C,$6C,$6C,$D3,$D3,$53,$53,$53,$53,$6C,$6C
	.DB $D3,$53,$53,$53,$53,$53,$6C,$6C,$D3,$53,$53,$53,$53,$53,$53,$53
	.DB $53,$53,$53,$53,$53,$53,$53,$53,$53,$53,$53,$53,$53,$53,$53,$53
	.DB $53,$53,$53,$53,$53,$53,$53,$00,$53,$53,$53,$53,$53,$53,$53,$00
	.DB $6C,$6C,$6C,$6C,$00,$00,$00,$00,$6C,$6C,$00,$00,$00,$00,$00,$00
	.DB $6C,$5E,$00,$00,$00,$00,$00,$00,$53,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $A4,$A4,$A4,$A4,$A4,$A4,$A4,$A4,$00,$A4,$A4,$A4,$A4,$A4,$A4,$A4
	.DB $00,$00,$A4,$A4,$A4,$A4,$A4,$A4,$00,$00,$00,$A4,$A4,$A4,$A4,$A4
	.DB $00,$00,$00,$00,$A4,$A4,$A4,$A4,$00,$00,$00,$00,$A4,$A4,$A4,$A4
	.DB $00,$00,$00,$00,$00,$A4,$A4,$A4,$00,$00,$00,$00,$00,$00,$A4,$A4
	.DB $A4,$A4,$A4,$24,$24,$24,$24,$24,$A4,$A4,$A4,$A4,$24,$24,$24,$21
	.DB $A4,$A4,$A4,$A4,$A4,$24,$21,$21,$A4,$A4,$A4,$A4,$A4,$21,$21,$21
	.DB $A4,$A4,$A4,$A4,$A1,$A1,$21,$21,$A4,$A4,$A4,$A1,$A1,$A1,$A1,$21
	.DB $A4,$A4,$A1,$A1,$A1,$A1,$A1,$A1,$A4,$A1,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $21,$21,$21,$21,$21,$21,$00,$00,$21,$21,$21,$21,$21,$21,$21,$00
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$A1,$21,$21,$21,$21,$21,$21,$21
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$21,$00,$00,$00,$00,$00,$00,$00
	.DB $21,$21,$00,$00,$00,$00,$00,$00,$21,$21,$21,$00,$00,$00,$00,$00
	.DB $21,$21,$21,$21,$00,$00,$00,$00,$21,$21,$21,$21,$21,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$E6,$00,$00,$00,$00,$00,$00,$E6,$E6
	.DB $00,$00,$00,$00,$00,$E6,$E6,$E6,$00,$00,$00,$00,$00,$E4,$E6,$E6
	.DB $00,$00,$00,$E0,$E0,$E1,$E6,$E6,$00,$00,$E0,$E0,$E0,$E0,$E1,$E6
	.DB $00,$00,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6
	.DB $E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6
	.DB $E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6
	.DB $E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$E6,$66
	.DB $E6,$D3,$D3,$D3,$D3,$D3,$53,$53,$E6,$E6,$E6,$D3,$D3,$D3,$53,$53
	.DB $E6,$E6,$E6,$E6,$E6,$D3,$53,$53,$E6,$E6,$E6,$E6,$66,$66,$61,$54
	.DB $E6,$E6,$E6,$66,$66,$66,$66,$61,$E6,$E6,$66,$66,$66,$66,$66,$66
	.DB $E6,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66
	.DB $53,$53,$53,$53,$53,$53,$00,$00,$53,$53,$53,$53,$53,$53,$00,$00
	.DB $53,$53,$53,$53,$53,$00,$00,$00,$53,$53,$53,$53,$53,$00,$00,$00
	.DB $59,$54,$53,$53,$4D,$00,$00,$00,$61,$5C,$58,$54,$52,$00,$00,$00
	.DB $66,$61,$5D,$59,$00,$00,$00,$00,$66,$66,$61,$5E,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$A4,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $00,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$00,$00,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $00,$00,$00,$A1,$A1,$A1,$A1,$A1,$00,$00,$00,$00,$A1,$A1,$A1,$A1
	.DB $00,$00,$00,$00,$00,$A1,$A1,$A1,$00,$00,$00,$00,$00,$00,$A1,$A1
	.DB $A1,$A1,$21,$21,$21,$21,$21,$21,$A1,$A1,$A1,$21,$21,$21,$21,$21
	.DB $A1,$A1,$A1,$A1,$21,$21,$21,$21,$A1,$A1,$A1,$A1,$A1,$21,$21,$21
	.DB $A1,$A1,$A1,$A1,$A1,$A1,$21,$21,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$30
	.DB $A1,$A1,$A1,$A1,$A1,$B0,$B0,$B0,$A1,$A1,$A1,$B0,$B0,$B0,$B0,$B0
	.DB $21,$21,$21,$21,$21,$25,$00,$00,$21,$21,$21,$21,$26,$2A,$00,$00
	.DB $21,$21,$21,$27,$2B,$2F,$30,$00,$21,$21,$2A,$2F,$30,$30,$30,$00
	.DB $21,$2F,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30
	.DB $30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30,$30
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$30,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$E0,$00,$00,$00,$00,$00,$00,$E0,$E0
	.DB $00,$00,$00,$00,$00,$00,$E0,$E0,$00,$00,$00,$00,$00,$E0,$E0,$E0
	.DB $00,$00,$00,$00,$E0,$E0,$E0,$E0,$00,$00,$00,$E0,$E0,$E0,$E0,$E0
	.DB $00,$E0,$E0,$E0,$E0,$E0,$E0,$E1,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0
	.DB $E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0
	.DB $E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0
	.DB $E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$E0,$60
	.DB $E6,$E6,$E6,$E6,$E6,$E6,$66,$66,$E1,$E6,$E6,$E6,$E6,$66,$66,$66
	.DB $E0,$E1,$E6,$E6,$66,$66,$66,$66,$E0,$E0,$E6,$66,$66,$66,$66,$66
	.DB $E0,$E0,$60,$66,$66,$66,$66,$66,$E0,$60,$60,$60,$66,$66,$66,$66
	.DB $60,$60,$60,$60,$60,$66,$66,$66,$60,$60,$60,$60,$60,$60,$66,$66
	.DB $66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66
	.DB $66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$66,$00
	.DB $66,$66,$66,$66,$66,$66,$00,$00,$66,$66,$66,$66,$00,$00,$00,$00
	.DB $66,$66,$66,$00,$00,$00,$00,$00,$66,$66,$00,$00,$00,$00,$00,$00
	.DB $66,$66,$65,$00,$00,$00,$00,$00,$66,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$A1,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $A1,$B0,$B0,$B0,$B0,$B0,$B0,$B0,$A1,$B0,$B0,$B0,$B0,$B0,$B0,$B0
	.DB $00,$B0,$B0,$B0,$B0,$B0,$B0,$B0,$00,$00,$B0,$B0,$B0,$B0,$B0,$B0
	.DB $00,$00,$B0,$B0,$B0,$B0,$B0,$B0,$00,$00,$00,$B0,$B0,$B0,$B0,$B0
	.DB $00,$00,$00,$B0,$B0,$B0,$B0,$B0,$00,$00,$00,$00,$B0,$B0,$B0,$B0
	.DB $B0,$30,$30,$30,$30,$30,$30,$30,$B0,$30,$30,$30,$30,$30,$30,$30
	.DB $B0,$B0,$30,$30,$30,$30,$30,$30,$B0,$B0,$30,$30,$30,$30,$30,$30
	.DB $B0,$B0,$B0,$30,$30,$30,$30,$30,$B0,$B0,$B0,$30,$30,$30,$30,$30
	.DB $B0,$B0,$B0,$B0,$30,$30,$30,$30,$B0,$B0,$B0,$B0,$30,$30,$30,$30
	.DB $30,$00,$00,$00,$00,$00,$00,$00,$30,$30,$00,$00,$00,$00,$00,$00
	.DB $30,$30,$00,$00,$00,$00,$00,$00,$30,$30,$30,$00,$00,$00,$00,$00
	.DB $30,$30,$30,$00,$00,$00,$00,$00,$30,$30,$30,$30,$00,$00,$00,$00
	.DB $30,$30,$30,$30,$00,$00,$00,$00,$30,$30,$30,$21,$21,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$C5,$C5
	.DB $00,$00,$E0,$E0,$E0,$E0,$E0,$E0,$00,$E0,$E0,$E0,$E0,$E0,$E0,$E0
	.DB $00,$DD,$E0,$E0,$E0,$E0,$E0,$E0,$00,$D8,$DC,$E0,$E0,$E0,$E0,$E0
	.DB $D1,$D3,$D7,$DB,$E0,$E0,$E0,$E0,$CC,$CE,$D0,$D3,$D8,$E0,$E0,$E0
	.DB $C6,$C7,$C9,$CA,$CE,$D3,$E0,$E0,$C5,$C5,$C5,$C5,$C5,$C5,$C5,$E0
	.DB $E0,$E0,$E0,$E0,$E0,$E0,$60,$60,$E0,$E0,$E0,$E0,$E0,$60,$60,$60
	.DB $E0,$E0,$E0,$E0,$60,$60,$60,$60,$E0,$E0,$E0,$60,$60,$60,$60,$60
	.DB $E0,$E0,$60,$60,$60,$60,$60,$60,$E0,$60,$60,$60,$60,$60,$60,$60
	.DB $60,$60,$60,$60,$60,$60,$60,$60,$60,$60,$60,$60,$60,$60,$60,$60
	.DB $60,$60,$60,$60,$60,$60,$60,$66,$60,$60,$60,$60,$60,$60,$60,$65
	.DB $60,$60,$60,$60,$60,$60,$00,$00,$60,$60,$60,$60,$60,$00,$00,$00
	.DB $60,$60,$60,$60,$00,$00,$00,$00,$60,$60,$60,$00,$00,$00,$00,$00
	.DB $60,$60,$00,$00,$00,$00,$00,$00,$60,$60,$00,$00,$00,$00,$00,$00
	.DB $66,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$B0,$B0,$B0,$B0,$00,$00,$00,$00,$00,$B0,$B0,$B0
	.DB $00,$00,$00,$00,$00,$B0,$B0,$B0,$00,$00,$00,$00,$00,$00,$B0,$B0
	.DB $00,$00,$00,$00,$00,$00,$B0,$B0,$00,$00,$00,$00,$00,$00,$B0,$B0
	.DB $00,$00,$00,$00,$00,$00,$00,$B0,$00,$00,$00,$00,$00,$00,$00,$B0
	.DB $B0,$B0,$B0,$B0,$B0,$30,$30,$30,$B0,$B0,$B0,$B0,$B0,$30,$30,$30
	.DB $B0,$B0,$B0,$B0,$B0,$B0,$30,$30,$B0,$B0,$B0,$B0,$B0,$B0,$30,$21
	.DB $B0,$B0,$B0,$B0,$B0,$B0,$21,$21,$B0,$B0,$B0,$B0,$AE,$A1,$A1,$21
	.DB $B0,$B0,$AE,$A9,$A1,$A1,$A1,$A1,$AE,$AA,$A6,$A1,$A1,$A1,$A1,$A1
	.DB $30,$30,$21,$21,$21,$21,$00,$00,$30,$21,$21,$21,$21,$21,$21,$00
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$A1,$21,$21,$21,$21,$21,$21,$21
	.DB $00,$00,$00,$00,$00,$00,$C5,$C5,$00,$00,$00,$00,$00,$00,$C5,$C5
	.DB $00,$00,$00,$00,$00,$00,$C5,$C5,$00,$00,$00,$00,$00,$00,$C5,$C5
	.DB $00,$00,$00,$00,$00,$00,$C5,$C5,$00,$00,$00,$00,$00,$00,$C5,$C5
	.DB $00,$00,$00,$00,$00,$00,$C5,$C5,$00,$00,$00,$00,$00,$00,$C5,$C5
	.DB $C5,$C5,$C5,$C5,$C5,$C5,$C5,$45,$C5,$C5,$C5,$C5,$C5,$C5,$C5,$45
	.DB $C5,$C5,$C5,$C5,$C5,$C5,$C5,$45,$C5,$C5,$C5,$C5,$C5,$C5,$C5,$45
	.DB $C5,$C5,$C5,$C5,$C5,$C5,$C5,$45,$C5,$C5,$C5,$C5,$C5,$C5,$C5,$45
	.DB $C5,$C5,$C5,$C5,$C5,$C5,$C5,$45,$C5,$C5,$C5,$C5,$C5,$C5,$C5,$45
	.DB $45,$60,$60,$60,$60,$60,$60,$60,$45,$45,$60,$60,$60,$60,$60,$60
	.DB $45,$45,$45,$60,$60,$60,$60,$00,$45,$45,$45,$45,$60,$60,$00,$00
	.DB $45,$45,$45,$45,$45,$45,$00,$00,$45,$45,$45,$45,$45,$45,$00,$00
	.DB $45,$45,$45,$45,$45,$45,$00,$00,$45,$45,$45,$45,$45,$45,$00,$00
	.DB $60,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $A9,$A5,$A1,$A1,$A1,$A1,$A1,$A1,$A4,$A1,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $00,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$00,$00,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $00,$00,$00,$A1,$A1,$A1,$A1,$A1,$00,$00,$00,$00,$A1,$A1,$A1,$A1
	.DB $00,$00,$00,$00,$00,$A1,$A1,$A1,$00,$00,$00,$00,$00,$00,$A1,$A1
	.DB $A1,$A1,$21,$21,$21,$21,$21,$21,$A1,$A1,$A1,$21,$21,$21,$21,$21
	.DB $A1,$A1,$A1,$A1,$21,$21,$21,$21,$A1,$A1,$A1,$A1,$A1,$21,$21,$21
	.DB $A1,$A1,$A1,$A1,$A1,$A1,$21,$21,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$21
	.DB $A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $21,$21,$21,$21,$21,$21,$00,$00,$21,$21,$21,$21,$21,$21,$21,$00
	.DB $21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$21,$15
	.DB $21,$21,$21,$21,$21,$21,$15,$15,$21,$21,$21,$21,$21,$21,$15,$15
	.DB $21,$21,$21,$21,$21,$15,$15,$15,$A1,$21,$21,$21,$21,$15,$15,$15
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$15,$00,$00,$00,$00,$00,$00,$00
	.DB $15,$15,$15,$00,$00,$00,$00,$00,$15,$15,$15,$15,$15,$00,$00,$00
	.DB $15,$15,$15,$15,$15,$15,$15,$00,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$15,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$C5,$C5,$00,$00,$00,$00,$00,$D4,$C5,$C5
	.DB $00,$00,$00,$00,$00,$D4,$D4,$D4,$00,$00,$00,$00,$D4,$D4,$D4,$D4
	.DB $00,$00,$00,$00,$D4,$D4,$D4,$D4,$00,$00,$00,$D4,$D4,$D4,$D4,$D4
	.DB $00,$00,$00,$D4,$D4,$D4,$D4,$D4,$00,$00,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $C5,$C5,$C5,$C5,$C5,$C5,$C5,$45,$C5,$C5,$C5,$C5,$C5,$C5,$C5,$45
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$C5,$45,$D4,$D4,$D4,$D4,$D4,$D4,$54,$54
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$54,$54,$D4,$D4,$D4,$D4,$D4,$54,$54,$54
	.DB $D4,$D4,$D4,$D4,$D4,$54,$54,$54,$D4,$D4,$D4,$D4,$54,$54,$54,$54
	.DB $45,$45,$45,$45,$45,$45,$00,$00,$45,$45,$45,$45,$45,$45,$00,$00
	.DB $45,$45,$45,$45,$45,$45,$00,$00,$54,$4F,$4B,$4A,$48,$47,$00,$00
	.DB $54,$54,$54,$51,$4F,$4D,$00,$00,$54,$54,$54,$54,$54,$52,$00,$00
	.DB $54,$54,$54,$54,$59,$00,$00,$00,$54,$54,$54,$54,$5E,$00,$00,$00
	.DB $A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $00,$A1,$A1,$A1,$A1,$A1,$A1,$A1,$00,$00,$A1,$A1,$A1,$A1,$A1,$A1
	.DB $00,$00,$00,$A1,$A1,$A1,$A1,$A1,$00,$00,$00,$00,$A1,$A1,$A1,$A1
	.DB $00,$00,$00,$00,$00,$A1,$A1,$9C,$00,$00,$00,$00,$00,$00,$9D,$98
	.DB $A1,$A1,$21,$21,$15,$15,$15,$15,$A1,$A1,$A1,$15,$15,$15,$15,$15
	.DB $A1,$A1,$A1,$95,$95,$15,$15,$15,$A1,$A1,$95,$95,$95,$95,$95,$15
	.DB $A1,$98,$95,$95,$95,$95,$95,$95,$9B,$95,$95,$95,$95,$95,$95,$95
	.DB $97,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95
	.DB $15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $95,$15,$15,$15,$15,$15,$15,$15,$95,$95,$95,$15,$15,$15,$15,$15
	.DB $95,$95,$95,$95,$95,$15,$15,$15,$95,$95,$95,$95,$95,$95,$95,$15
	.DB $15,$15,$15,$00,$00,$00,$00,$00,$15,$15,$15,$15,$15,$00,$00,$00
	.DB $15,$15,$15,$15,$15,$15,$15,$00,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $15,$15,$00,$00,$00,$00,$00,$00,$15,$15,$15,$15,$00,$00,$00,$00
	.DB $15,$15,$15,$15,$15,$15,$00,$00,$15,$15,$15,$15,$15,$15,$15,$15
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$D4
	.DB $00,$00,$00,$00,$00,$00,$00,$D4,$00,$00,$00,$00,$00,$00,$D4,$D4
	.DB $00,$00,$D4,$D4,$D4,$D4,$D4,$D4,$00,$D4,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $00,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $D4,$D4,$D4,$D4,$54,$54,$54,$54,$D4,$D4,$D4,$54,$54,$54,$54,$54
	.DB $D4,$D4,$D4,$54,$54,$54,$54,$54,$D4,$D4,$54,$54,$54,$54,$54,$54
	.DB $D4,$D4,$54,$54,$54,$54,$54,$54,$D4,$54,$54,$54,$54,$54,$54,$54
	.DB $D4,$54,$54,$54,$54,$54,$54,$54,$54,$54,$54,$54,$54,$54,$54,$54
	.DB $54,$54,$54,$65,$00,$00,$00,$00,$54,$54,$54,$00,$00,$00,$00,$00
	.DB $54,$54,$00,$00,$00,$00,$00,$00,$54,$54,$00,$00,$00,$00,$00,$00
	.DB $54,$00,$00,$00,$00,$00,$00,$00,$54,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$00,$00,$95,$95,$95,$95,$95,$95
	.DB $00,$00,$00,$00,$95,$95,$95,$95,$00,$00,$00,$00,$00,$00,$95,$95
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$00,$00,$95,$95,$95,$95,$95,$95
	.DB $00,$00,$00,$95,$95,$95,$95,$95,$00,$00,$00,$00,$00,$95,$95,$95
	.DB $95,$15,$15,$15,$15,$15,$15,$15,$95,$95,$95,$15,$15,$15,$15,$15
	.DB $95,$95,$95,$95,$95,$15,$15,$15,$95,$95,$95,$95,$95,$95,$95,$15
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95,$95
	.DB $15,$15,$15,$15,$15,$15,$15,$06,$15,$15,$15,$15,$15,$15,$15,$06
	.DB $15,$15,$15,$15,$15,$15,$15,$06,$15,$15,$15,$15,$15,$15,$15,$06
	.DB $95,$15,$15,$15,$15,$15,$15,$06,$95,$95,$95,$15,$15,$15,$15,$06
	.DB $95,$95,$95,$95,$95,$15,$15,$06,$95,$95,$95,$95,$95,$95,$06,$06
	.DB $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
	.DB $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
	.DB $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
	.DB $06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06,$06
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$07,$0D,$12,$00,$00,$00,$00,$00
	.DB $08,$0F,$14,$18,$18,$00,$00,$00,$0A,$11,$18,$18,$18,$18,$18,$00
	.DB $0B,$14,$18,$18,$18,$18,$18,$18,$0F,$18,$18,$18,$18,$18,$18,$18
	.DB $14,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $18,$00,$00,$00,$00,$00,$00,$00,$18,$18,$00,$00,$00,$00,$00,$00
	.DB $18,$18,$18,$18,$00,$00,$00,$00,$18,$18,$18,$18,$18,$18,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$D4,$D4,$00,$00,$00,$00,$00,$D4,$D4,$D4
	.DB $00,$00,$00,$00,$00,$D4,$D4,$D4,$00,$00,$00,$00,$D4,$D4,$D4,$D4
	.DB $00,$00,$00,$00,$D4,$D4,$D4,$D4,$00,$00,$00,$D4,$D4,$D4,$D4,$D4
	.DB $00,$00,$00,$D4,$D4,$D4,$D4,$D4,$00,$00,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$54
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$D4,$54,$D4,$D4,$D4,$D4,$D4,$D4,$54,$54
	.DB $D4,$D4,$D4,$D4,$D4,$D4,$54,$54,$D4,$D4,$D4,$D4,$D4,$54,$54,$54
	.DB $D4,$D4,$D4,$D4,$D4,$54,$54,$54,$D4,$D4,$D4,$D4,$54,$54,$54,$54
	.DB $54,$54,$54,$54,$54,$54,$54,$00,$54,$54,$54,$54,$54,$54,$54,$00
	.DB $54,$54,$54,$54,$54,$54,$00,$00,$54,$54,$54,$54,$54,$54,$00,$00
	.DB $54,$54,$54,$54,$54,$00,$00,$00,$54,$54,$54,$54,$54,$00,$00,$00
	.DB $54,$54,$54,$54,$00,$00,$00,$00,$54,$54,$54,$54,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$95,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $95,$95,$95,$95,$95,$95,$95,$95,$00,$95,$95,$95,$95,$95,$95,$95
	.DB $00,$00,$00,$95,$95,$95,$95,$95,$00,$00,$00,$00,$00,$95,$95,$95
	.DB $00,$00,$00,$00,$00,$00,$00,$95,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $95,$95,$95,$95,$95,$95,$86,$86,$95,$95,$95,$95,$95,$93,$86,$86
	.DB $95,$95,$95,$95,$95,$8E,$86,$86,$95,$95,$95,$95,$93,$8A,$86,$86
	.DB $95,$95,$95,$95,$90,$89,$86,$86,$00,$95,$95,$93,$8E,$87,$86,$86
	.DB $00,$00,$00,$91,$8C,$86,$86,$86,$00,$00,$00,$00,$00,$00,$86,$86
	.DB $86,$86,$86,$86,$86,$86,$86,$18,$86,$86,$86,$86,$86,$86,$86,$98
	.DB $86,$86,$86,$86,$86,$86,$98,$98,$86,$86,$86,$86,$86,$86,$98,$98
	.DB $86,$86,$86,$86,$86,$98,$98,$98,$86,$86,$86,$86,$98,$98,$98,$98
	.DB $86,$86,$86,$86,$98,$98,$98,$98,$86,$86,$86,$86,$98,$98,$98,$98
	.DB $18,$18,$18,$18,$18,$18,$18,$18,$98,$18,$18,$18,$18,$18,$18,$18
	.DB $98,$98,$98,$18,$18,$18,$18,$18,$98,$98,$98,$98,$18,$18,$18,$18
	.DB $98,$98,$98,$98,$98,$98,$18,$18,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $18,$18,$18,$18,$18,$18,$18,$00,$18,$18,$18,$18,$18,$18,$18,$18
	.DB $18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18
	.DB $18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18
	.DB $98,$18,$18,$18,$18,$18,$18,$18,$98,$98,$98,$18,$18,$18,$18,$18
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$18,$00,$00,$00,$00,$00,$00,$00
	.DB $18,$18,$00,$00,$00,$00,$00,$00,$18,$18,$18,$18,$00,$00,$00,$00
	.DB $18,$18,$18,$18,$18,$18,$00,$00,$18,$18,$18,$18,$18,$18,$18,$00
	.DB $18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $18,$00,$00,$00,$00,$00,$00,$00,$18,$18,$18,$00,$00,$00,$00,$00
	.DB $54,$54,$54,$00,$00,$00,$00,$00,$54,$54,$54,$00,$00,$00,$00,$00
	.DB $54,$54,$00,$00,$00,$00,$00,$00,$54,$54,$00,$00,$00,$00,$00,$00
	.DB $54,$00,$00,$00,$00,$00,$00,$00,$54,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$98,$98,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $00,$98,$98,$98,$98,$98,$98,$98,$00,$00,$00,$98,$98,$98,$98,$98
	.DB $00,$00,$00,$00,$98,$98,$98,$98,$00,$00,$00,$00,$00,$00,$98,$98
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $98,$98,$98,$98,$18,$18,$18,$18,$98,$98,$98,$98,$98,$98,$18,$18
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$00,$98,$98,$98,$98,$98,$98,$98
	.DB $18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18,$18
	.DB $18,$18,$18,$18,$18,$18,$18,$18,$98,$18,$18,$18,$18,$18,$18,$18
	.DB $98,$98,$98,$18,$18,$18,$18,$18,$98,$98,$98,$98,$98,$18,$18,$18
	.DB $98,$98,$98,$98,$98,$98,$18,$18,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $18,$18,$18,$18,$00,$00,$00,$00,$18,$18,$18,$18,$11,$18,$00,$00
	.DB $18,$18,$18,$18,$11,$11,$11,$11,$18,$18,$18,$18,$11,$11,$11,$11
	.DB $18,$18,$18,$11,$11,$11,$11,$11,$18,$18,$18,$11,$11,$11,$11,$11
	.DB $18,$18,$11,$11,$11,$11,$11,$11,$18,$18,$11,$11,$11,$11,$11,$11
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$11,$11,$00,$00,$00,$00,$00,$00
	.DB $11,$11,$11,$11,$11,$00,$00,$00,$11,$11,$11,$11,$11,$11,$11,$00
	.DB $11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $11,$11,$00,$00,$00,$00,$00,$00,$11,$11,$11,$11,$00,$00,$00,$00
	.DB $54,$54,$54,$54,$54,$54,$54,$00,$54,$54,$54,$54,$54,$54,$54,$00
	.DB $54,$54,$54,$54,$54,$54,$00,$00,$54,$54,$54,$54,$54,$54,$00,$00
	.DB $54,$54,$54,$54,$54,$00,$00,$00,$54,$54,$54,$54,$54,$00,$00,$00
	.DB $54,$54,$54,$54,$54,$00,$00,$00,$54,$54,$54,$54,$00,$00,$00,$00
	.DB $00,$00,$00,$98,$98,$98,$98,$98,$00,$00,$00,$00,$00,$98,$98,$98
	.DB $00,$00,$00,$00,$00,$00,$98,$98,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98,$98
	.DB $00,$98,$98,$98,$98,$98,$98,$93,$00,$00,$00,$98,$98,$98,$97,$91
	.DB $00,$00,$00,$00,$9D,$98,$93,$91,$00,$00,$00,$00,$00,$00,$91,$91
	.DB $98,$11,$11,$11,$11,$11,$11,$11,$98,$91,$91,$91,$11,$11,$11,$11
	.DB $93,$91,$91,$91,$91,$91,$11,$11,$91,$91,$91,$91,$91,$91,$91,$91
	.DB $91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91
	.DB $91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91
	.DB $11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11
	.DB $11,$11,$11,$11,$11,$11,$11,$11,$91,$11,$11,$11,$11,$11,$11,$11
	.DB $91,$91,$91,$11,$11,$11,$11,$11,$91,$91,$91,$91,$91,$91,$11,$11
	.DB $91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91
	.DB $11,$11,$11,$11,$11,$11,$00,$00,$11,$11,$11,$11,$11,$11,$11,$11
	.DB $11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11
	.DB $11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11,$11
	.DB $11,$11,$11,$11,$11,$11,$11,$11,$91,$91,$91,$11,$11,$11,$11,$11
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$DF,$00,$00,$00,$00,$00,$00,$DF,$DF
	.DB $00,$00,$00,$00,$00,$DF,$DF,$DF,$00,$00,$00,$00,$DF,$DF,$DF,$DF
	.DB $00,$00,$D4,$D4,$D4,$D4,$D4,$D4,$00,$00,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $00,$D4,$D4,$D4,$D4,$D4,$D4,$D4,$DF,$D4,$D4,$D4,$D4,$D4,$D4,$D4
	.DB $DF,$DF,$D4,$D4,$D4,$D4,$D4,$D4,$DF,$DF,$DF,$DF,$D4,$D4,$D4,$D4
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$D4,$D4,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF
	.DB $D4,$D4,$D4,$D4,$54,$54,$54,$54,$D4,$D4,$D4,$54,$54,$54,$54,$54
	.DB $D4,$D4,$D4,$54,$54,$54,$54,$54,$D4,$D4,$54,$54,$54,$54,$54,$54
	.DB $D4,$D4,$54,$54,$54,$54,$54,$54,$D4,$54,$54,$54,$54,$54,$54,$54
	.DB $D4,$54,$54,$54,$54,$54,$54,$54,$D4,$54,$54,$54,$54,$54,$54,$54
	.DB $54,$54,$54,$54,$00,$00,$00,$00,$54,$54,$54,$00,$00,$00,$00,$00
	.DB $54,$54,$54,$00,$00,$00,$00,$00,$54,$54,$00,$00,$00,$00,$00,$00
	.DB $54,$54,$00,$00,$00,$00,$00,$00,$54,$00,$00,$00,$00,$00,$00,$00
	.DB $54,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$91,$91,$91,$91,$91,$91,$91,$00,$00,$00,$00,$91,$91,$91,$91
	.DB $00,$00,$00,$00,$00,$00,$91,$91,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91
	.DB $91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91,$91
	.DB $00,$00,$00,$91,$91,$91,$91,$91,$00,$00,$00,$00,$00,$91,$91,$91
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $91,$91,$91,$91,$91,$11,$11,$11,$91,$91,$91,$91,$91,$91,$91,$11
	.DB $91,$91,$91,$91,$91,$91,$91,$81,$91,$91,$91,$91,$91,$91,$91,$81
	.DB $91,$91,$91,$91,$91,$91,$8E,$81,$91,$91,$91,$91,$91,$91,$8A,$81
	.DB $91,$91,$91,$91,$91,$90,$89,$81,$00,$00,$91,$91,$91,$8E,$87,$81
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$DF
	.DB $00,$00,$00,$00,$00,$00,$DF,$DF,$00,$00,$00,$00,$00,$DF,$DF,$DF
	.DB $00,$00,$00,$00,$DF,$DF,$DF,$DF,$00,$00,$00,$DF,$DF,$DF,$DF,$DF
	.DB $00,$00,$DF,$DF,$DF,$DF,$DF,$DF,$00,$DF,$DF,$DF,$DF,$DF,$DF,$DF
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$5F
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$5F,$5F,$DF,$DF,$DF,$DF,$DF,$5F,$5F,$5F
	.DB $DF,$DF,$DF,$DF,$5F,$5F,$5F,$5F,$DF,$DF,$DF,$5F,$5F,$5F,$5F,$5F
	.DB $DF,$DF,$5F,$5F,$5F,$5F,$5F,$5F,$DF,$5F,$5F,$5F,$5F,$5F,$5F,$5F
	.DB $5F,$5F,$54,$54,$54,$54,$54,$54,$5F,$5F,$5F,$59,$54,$54,$54,$4D
	.DB $5F,$5F,$5F,$5F,$5C,$58,$54,$52,$5F,$5F,$5F,$5F,$5F,$5D,$59,$00
	.DB $5F,$5F,$5F,$5F,$5F,$5F,$5E,$00,$5F,$5F,$5F,$5F,$5F,$65,$00,$00
	.DB $5F,$5F,$5F,$5F,$00,$00,$00,$00,$5F,$5F,$5F,$5F,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$91,$8C,$86,$81,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$00,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $00,$01,$00,$01,$DF,$DF,$DF,$DF,$01,$01,$01,$01,$DF,$DF,$DF,$DF
	.DB $01,$01,$01,$DF,$DF,$DF,$DF,$DF,$01,$01,$01,$01,$DF,$DF,$DF,$DF
	.DB $01,$01,$01,$01,$01,$DF,$DF,$DF,$01,$01,$01,$01,$01,$01,$DF,$DF
	.DB $01,$01,$01,$01,$01,$01,$01,$DF,$01,$01,$01,$01,$01,$01,$01,$01
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF,$DF
	.DB $DF,$DF,$DF,$DF,$DF,$DF,$DF,$5F,$DF,$DF,$DF,$DF,$DF,$DF,$5F,$5F
	.DB $DF,$DF,$DF,$DF,$DF,$5F,$5F,$5F,$DF,$DF,$DF,$DF,$5F,$5F,$5F,$5F
	.DB $DF,$DF,$DF,$5F,$5F,$5F,$5F,$5F,$DF,$DF,$5F,$5F,$5F,$5F,$5F,$5F
	.DB $DF,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F
	.DB $5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F,$5F
	.DB $5F,$5F,$5F,$5F,$5F,$5F,$5F,$00,$5F,$5F,$5F,$5F,$5F,$5F,$00,$00
	.DB $5F,$5F,$5F,$5F,$5F,$00,$00,$00,$5F,$5F,$5F,$5F,$00,$00,$00,$00
	.DB $5F,$5F,$5F,$00,$00,$00,$00,$00,$5F,$5F,$00,$00,$00,$00,$00,$00
	.DB $5F,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $01,$01,$01,$01,$01,$01,$01,$01,$81,$81,$81,$81,$81,$81,$81,$81
	.DB $81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81
	.DB $81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81
	.DB $81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81,$81
	.DB $01,$DF,$5F,$5F,$5F,$5F,$5F,$5F,$81,$81,$61,$5F,$5F,$5F,$5F,$5F
	.DB $81,$81,$6F,$61,$5F,$5F,$5F,$5F,$81,$81,$74,$6A,$61,$5F,$5F,$5F
	.DB $81,$81,$78,$6F,$67,$61,$5F,$5F,$81,$81,$79,$72,$6B,$66,$61,$5E
	.DB $81,$81,$7B,$74,$6F,$6A,$65,$00,$81,$81,$7C,$76,$71,$00,$00,$00
	.DB $5F,$5F,$5F,$5F,$00,$00,$00,$00,$5F,$5F,$5F,$00,$00,$00,$00,$00
	.DB $5F,$5F,$00,$00,$00,$00,$00,$00,$5F,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00



; --------------------------------------
; Screen map (16*16 pixel tiles in a map of 16*16 bytes)
; (API offsets screen data by $2E)
screen_base:
	.DB $2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$2E,$2E,$2E,$2F,$30,$31,$2E,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$2E,$2E,$32,$33,$34,$35,$31,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$2E,$36,$37,$38,$39,$3A,$3B,$2E,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$3C,$3D,$3E,$3F,$40,$41,$42,$43,$2E,$2E,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$44,$45,$46,$47,$48,$49,$4A,$4B,$4C,$4D,$2E,$2E,$2E
	.DB $2E,$2E,$2E,$4E,$4F,$50,$51,$52,$53,$54,$55,$56,$55,$4C,$4D,$2E
	.DB $2E,$2E,$57,$58,$59,$5A,$5B,$5C,$56,$53,$5D,$5E,$5F,$60,$4B,$4C
	.DB $2E,$61,$62,$63,$64,$65,$66,$67,$68,$69,$6A,$6B,$6C,$5E,$5F,$60
	.DB $6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D
	.DB $6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D
	.DB $6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D
	.DB $6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D,$6D


; --------------------------------------
; Sprites (16*16 pixels)
spr_ball_0:
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$45,$29,$8E,$73,$51,$8C
	.DB $10,$84,$4D,$6B,$65,$29,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$24,$21,$9A,$D6,$5D,$EF,$9E,$F7,$3C,$E7
	.DB $1C,$E7,$79,$CE,$96,$B5,$51,$8C,$86,$31,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$49,$4A,$3C,$E7,$BE,$F7,$BE,$F7,$5D,$EF,$1C,$E7
	.DB $1C,$E7,$79,$CE,$F7,$BD,$75,$AD,$30,$84,$28,$42,$00,$00,$00,$00
	.DB $00,$00,$C3,$18,$1C,$E7,$9E,$F7,$7D,$EF,$7D,$EF,$1C,$E7,$5D,$EF
	.DB $9A,$D6,$79,$CE,$38,$C6,$B6,$B5,$D3,$9C,$6D,$6B,$24,$21,$00,$00
	.DB $00,$00,$79,$CE,$7D,$EF,$7D,$EF,$DF,$FF,$BE,$F7,$DF,$FF,$9A,$D6
	.DB $FB,$DE,$18,$C6,$D7,$BD,$F7,$BD,$34,$A5,$30,$84,$8A,$52,$00,$00
	.DB $82,$10,$FB,$DE,$7D,$EF,$BE,$F7,$FF,$FF,$BE,$F7,$9E,$F7,$9A,$D6
	.DB $59,$CE,$59,$CE,$55,$AD,$F3,$9C,$D3,$9C,$EF,$7B,$AA,$52,$04,$21
	.DB $2C,$63,$1C,$E7,$3C,$E7,$9E,$F7,$BE,$F7,$DF,$FF,$7D,$EF,$9E,$F7
	.DB $18,$C6,$38,$C6,$55,$AD,$F3,$9C,$71,$8C,$EF,$7B,$0C,$63,$69,$4A
	.DB $51,$8C,$3C,$E7,$7D,$EF,$5D,$EF,$5D,$EF,$DB,$DE,$BA,$D6,$38,$C6
	.DB $96,$B5,$18,$C6,$F3,$9C,$14,$A5,$D3,$9C,$CF,$7B,$4D,$6B,$8A,$52
	.DB $51,$8C,$DB,$DE,$FB,$DE,$FB,$DE,$3C,$E7,$3C,$E7,$9A,$D6,$59,$CE
	.DB $55,$AD,$14,$A5,$14,$A5,$CF,$7B,$CF,$7B,$4D,$6B,$EB,$5A,$69,$4A
	.DB $AE,$73,$BA,$D6,$BA,$D6,$79,$CE,$38,$C6,$18,$C6,$B6,$B5,$18,$C6
	.DB $D7,$BD,$B2,$94,$F3,$9C,$30,$84,$AE,$73,$8E,$73,$EB,$5A,$49,$4A
	.DB $86,$31,$18,$C6,$DB,$DE,$BA,$D6,$9A,$D6,$59,$CE,$55,$AD,$D3,$9C
	.DB $D3,$9C,$14,$A5,$CF,$7B,$EF,$7B,$4D,$6B,$EB,$5A,$28,$42,$65,$29
	.DB $00,$00,$71,$8C,$75,$AD,$B6,$B5,$96,$B5,$B6,$B5,$B6,$B5,$14,$A5
	.DB $D3,$9C,$CF,$7B,$AE,$73,$4D,$6B,$0C,$63,$AA,$52,$86,$31,$00,$00
	.DB $00,$00,$24,$21,$92,$94,$55,$AD,$34,$A5,$F3,$9C,$51,$8C,$51,$8C
	.DB $92,$94,$AE,$73,$6D,$6B,$4D,$6B,$CB,$5A,$C7,$39,$45,$29,$00,$00
	.DB $00,$00,$00,$00,$E7,$39,$CF,$7B,$71,$8C,$D3,$9C,$51,$8C,$AE,$73
	.DB $AE,$73,$6D,$6B,$EB,$5A,$AA,$52,$A6,$31,$A6,$31,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$E3,$18,$CB,$5A,$4D,$6B,$8E,$73,$4D,$6B
	.DB $2C,$63,$CB,$5A,$E7,$39,$C3,$18,$24,$21,$00,$00,$00,$00,$00,$00
	.DB $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$24,$21,$28,$42,$E7,$39
	.DB $E7,$39,$E7,$39,$E3,$18,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

; --------------------------------------
; Tiles (16*16 pixel tiles of 16 bit pixels)
tile_base:
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15

	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$57,$85,$35,$85,$51,$74
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$D9,$8D
	.DB $98,$8D,$77,$8D,$76,$8D,$54,$8D,$B2,$84,$ED,$73,$4A,$63,$08,$5B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$55,$85
	.DB $B2,$7C,$4F,$74,$4E,$7C,$6E,$84,$6D,$8C,$CB,$7B,$48,$6B,$65,$52

	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$1C,$8E,$BE,$9E,$BD,$9E,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$D8,$95,$77,$8D,$F4,$7C,$B4,$74
	.DB $F5,$7C,$14,$7D,$34,$85,$96,$95,$96,$8D,$14,$85,$F4,$7C,$36,$7D
	.DB $57,$7D,$F4,$7C,$92,$7C,$D2,$84,$12,$95,$8F,$8C,$AB,$73,$26,$42
	.DB $29,$6B,$8E,$94,$92,$B5,$92,$AD,$4D,$84,$29,$5B,$0A,$53,$30,$74
	.DB $AD,$63,$2A,$53,$2A,$5B,$8A,$6B,$8A,$73,$48,$6B,$E7,$62,$C6,$6A
	.DB $28,$7B,$C9,$8B,$2A,$94,$09,$94,$47,$73,$27,$6B,$0C,$84,$2A,$5B
	.DB $86,$4A,$25,$42,$45,$4A,$C6,$52,$07,$5B,$07,$5B,$C7,$52,$66,$52
	.DB $C8,$62,$08,$6B,$28,$6B,$27,$6B,$E5,$62,$E6,$62,$8A,$6B,$E8,$52
	.DB $C6,$62,$C6,$5A,$65,$4A,$A6,$4A,$6A,$5B,$4A,$5B,$A8,$42,$28,$3A
	.DB $89,$4A,$88,$4A,$66,$42,$A6,$4A,$E6,$52,$E7,$52,$28,$5B,$6A,$5B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $56,$8D,$98,$8D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $76,$9D,$56,$8D,$D9,$95,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $13,$95,$D3,$84,$97,$95,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $CE,$73,$CE,$6B,$14,$8D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$BB,$85,$FA,$95
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$DC,$8D,$DB,$8D,$B9,$8D,$B7,$95
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$DC,$85,$DB,$8D,$B9,$95,$77,$95,$54,$95
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$DB,$8D,$DA,$95,$98,$95,$54,$9D,$F0,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$DB,$8D,$D9,$95,$76,$9D,$33,$9D,$8E,$94
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$D9,$A5,$75,$A5,$31,$A5,$0F,$A5,$4B,$8C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$99,$95,$97,$9D,$33,$9D,$8D,$8C,$E9,$83,$67,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$B9,$95,$56,$95,$13,$9D,$F0,$9C,$8C,$94,$E8,$83,$E7,$83
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$D8,$9D,$F3,$8C,$4E,$84,$4C,$8C,$AB,$9C,$CA,$A4,$69,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$96,$9D,$4F,$84,$48,$6B,$47,$73,$E9,$8B,$6A,$9C,$29,$94
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $B6,$A5,$D1,$8C,$AA,$6B,$E6,$62,$E6,$6A,$27,$73,$68,$7B,$E6,$6A
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$B7,$9D
	.DB $12,$95,$2C,$7C,$88,$6B,$06,$63,$C6,$62,$A7,$62,$88,$6A,$C5,$49
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$34,$8D
	.DB $AF,$84,$0B,$74,$88,$6B,$06,$63,$65,$5A,$06,$52,$E7,$51,$28,$5A
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$FA,$95,$97,$8D,$B2,$84,$2C,$84
	.DB $0A,$84,$0A,$84,$EA,$83,$EA,$83,$2B,$8C,$EB,$83,$08,$63,$A8,$5A
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$76,$8D,$91,$7C,$CE,$73,$AA,$73
	.DB $89,$7B,$88,$7B,$68,$73,$68,$73,$68,$73,$28,$6B,$87,$5A,$06,$4A
	.DB $5F,$15,$B8,$8D,$75,$85,$34,$8D,$B0,$8C,$AC,$73,$AB,$7B,$0C,$8C
	.DB $EB,$83,$89,$7B,$28,$6B,$E7,$62,$E7,$62,$A7,$5A,$66,$52,$46,$4A
	.DB $77,$8D,$F3,$7C,$4F,$74,$6E,$84,$EB,$83,$68,$73,$0B,$94,$EB,$8B
	.DB $EC,$8B,$69,$73,$C7,$62,$08,$6B,$AA,$7B,$8B,$73,$2A,$6B,$88,$52
	.DB $F3,$84,$0D,$74,$49,$63,$68,$73,$47,$73,$27,$73,$2B,$9C,$CB,$8B
	.DB $2D,$9C,$8A,$7B,$A6,$5A,$49,$73,$2D,$8C,$CC,$7B,$E9,$5A,$E6,$39
	.DB $4E,$84,$AA,$73,$27,$6B,$26,$6B,$26,$6B,$07,$6B,$AA,$83,$EB,$8B
	.DB $6E,$A4,$CB,$8B,$08,$6B,$0C,$84,$F0,$A4,$4E,$84,$4B,$63,$C6,$31
	.DB $EB,$83,$A8,$83,$47,$73,$06,$6B,$28,$6B,$E8,$5A,$08,$63,$28,$73
	.DB $AA,$8B,$28,$73,$28,$73,$AE,$9C,$B3,$BD,$73,$AD,$33,$A5,$49,$42
	.DB $88,$7B,$87,$83,$26,$73,$C5,$62,$08,$63,$E8,$52,$87,$4A,$6A,$7B
	.DB $89,$83,$C6,$6A,$E7,$6A,$6D,$94,$31,$A5,$52,$A5,$D5,$B5,$AA,$4A
	.DB $48,$6B,$2A,$63,$89,$52,$88,$52,$87,$52,$48,$6B,$E6,$5A,$89,$73
	.DB $30,$A5,$49,$6B,$86,$52,$AB,$73,$B3,$B5,$73,$AD,$B5,$B5,$C9,$52
	.DB $C5,$5A,$E8,$62,$E9,$62,$C8,$5A,$A7,$52,$07,$63,$06,$63,$8D,$94
	.DB $10,$A5,$0C,$84,$69,$73,$49,$6B,$11,$A5,$D4,$BD,$F5,$BD,$AD,$73
	.DB $86,$73,$C9,$7B,$AE,$9C,$6D,$94,$2C,$8C,$A9,$7B,$48,$73,$EB,$83
	.DB $4D,$94,$10,$AD,$6D,$94,$08,$63,$8F,$9C,$15,$CE,$93,$B5,$90,$94
	.DB $49,$94,$29,$8C,$CD,$A4,$AD,$9C,$AD,$A4,$89,$7B,$E6,$6A,$C6,$62
	.DB $EB,$8B,$D3,$C5,$6E,$94,$A6,$5A,$CC,$83,$B3,$BD,$32,$AD,$B0,$9C
	.DB $AB,$A4,$49,$94,$6A,$94,$E9,$8B,$2A,$8C,$27,$73,$86,$62,$A6,$62
	.DB $0C,$8C,$D3,$C5,$EC,$83,$E8,$62,$E9,$62,$6F,$94,$73,$B5,$4F,$8C
	.DB $6C,$9C,$2E,$B5,$2D,$AD,$8B,$9C,$C9,$83,$C6,$6A,$86,$62,$86,$5A
	.DB $49,$73,$D0,$A4,$AB,$7B,$6E,$94,$2A,$6B,$6B,$73,$12,$AD,$8C,$73
	.DB $4A,$7B,$CD,$A4,$2E,$AD,$CC,$A4,$47,$73,$45,$5A,$66,$5A,$25,$4A
	.DB $87,$52,$0D,$84,$2A,$6B,$B4,$B5,$8F,$94,$6B,$6B,$CD,$7B,$6B,$6B
	.DB $89,$62,$8A,$7B,$EA,$83,$6B,$94,$C5,$62,$44,$52,$C8,$62,$C8,$5A
	.DB $C8,$5A,$CC,$7B,$67,$52,$D4,$BD,$B4,$B5,$0E,$84,$A9,$52,$AC,$73
	.DB $87,$5A,$06,$42,$E5,$41,$87,$52,$08,$63,$07,$5B,$07,$5B,$A6,$4A
	.DB $86,$4A,$C7,$52,$A7,$52,$05,$42,$46,$4A,$E8,$5A,$A7,$5A,$08,$63
	.DB $26,$4A,$06,$4A,$06,$42,$66,$4A,$C7,$5A,$07,$63,$48,$63,$69,$63
	.DB $28,$5B,$49,$63,$29,$63,$87,$4A,$46,$4A,$66,$4A,$25,$42,$29,$63
	.DB $26,$4A,$E6,$41,$E5,$41,$25,$42,$86,$52,$07,$5B,$48,$63,$28,$5B
	.DB $E8,$52,$29,$63,$4A,$6B,$E8,$5A,$67,$52,$66,$4A,$26,$4A,$A7,$52
	.DB $C5,$39,$44,$29,$43,$29,$A4,$31,$26,$42,$C7,$5A,$28,$63,$49,$63
	.DB $E8,$5A,$E8,$5A,$09,$63,$A8,$52,$E6,$41,$A4,$31,$A4,$31,$E4,$39
	.DB $23,$21,$E3,$18,$03,$21,$02,$21,$64,$29,$46,$42,$A7,$52,$08,$5B
	.DB $A7,$52,$67,$4A,$87,$52,$67,$4A,$C5,$39,$84,$31,$C5,$39,$E5,$39
	.DB $24,$21,$66,$29,$A7,$31,$E3,$18,$A2,$10,$44,$29,$C5,$31,$25,$42
	.DB $45,$42,$05,$42,$26,$42,$47,$4A,$E6,$39,$E6,$39,$67,$4A,$88,$52
	.DB $66,$29,$C8,$31,$09,$3A,$25,$21,$83,$10,$04,$19,$03,$21,$05,$42
	.DB $87,$52,$46,$4A,$06,$42,$07,$42,$A6,$31,$65,$29,$C6,$39,$88,$4A
	.DB $26,$19,$26,$19,$A9,$31,$47,$29,$26,$21,$46,$29,$E3,$18,$42,$29
	.DB $05,$42,$06,$42,$C5,$39,$07,$3A,$C6,$39,$85,$29,$C6,$39,$E5,$39
	.DB $24,$21,$C6,$39,$10,$84,$E8,$39,$C4,$10,$87,$29,$25,$21,$24,$21
	.DB $A6,$39,$06,$42,$E5,$41,$C4,$39,$C4,$31,$A4,$31,$84,$29,$28,$3A
	.DB $A5,$31,$A6,$31,$EB,$5A,$65,$29,$04,$21,$E8,$39,$A3,$10,$E4,$20
	.DB $45,$31,$A6,$39,$C5,$39,$C5,$39,$E5,$39,$C5,$31,$A5,$29,$E7,$31
	.DB $06,$42,$85,$31,$C6,$39,$03,$21,$E7,$39,$2C,$63,$04,$21,$C4,$18
	.DB $05,$21,$45,$29,$85,$31,$A5,$31,$A5,$31,$A5,$29,$85,$29,$23,$19
	.DB $E6,$41,$64,$31,$24,$29,$23,$21,$48,$4A,$AE,$73,$86,$31,$E4,$20
	.DB $E5,$20,$05,$21,$45,$29,$65,$29,$65,$29,$44,$21,$43,$21,$84,$29
	.DB $84,$39,$64,$29,$23,$29,$03,$21,$C6,$39,$AA,$5A,$45,$29,$25,$21
	.DB $04,$21,$05,$21,$25,$21,$45,$21,$25,$21,$24,$21,$64,$21,$A4,$29
	.DB $64,$31,$43,$29,$64,$29,$44,$21,$64,$29,$A6,$31,$44,$29,$45,$29
	.DB $04,$21,$04,$19,$45,$21,$46,$21,$25,$21,$65,$21,$C5,$31,$A7,$4A
	.DB $C5,$39,$64,$29,$85,$31,$85,$29,$A6,$31,$24,$21,$85,$29,$65,$29
	.DB $04,$19,$E4,$18,$25,$21,$25,$21,$05,$19,$85,$21,$27,$3A,$A7,$4A
	.DB $68,$4A,$85,$29,$A5,$31,$A5,$31,$E7,$39,$C2,$10,$65,$29,$64,$29
	.DB $03,$19,$E3,$18,$25,$19,$05,$19,$E4,$10,$65,$21,$27,$3A,$E8,$52
	.DB $C8,$52,$A7,$4A,$89,$42,$2C,$53,$D4,$84,$B9,$A5,$38,$8D,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $46,$42,$26,$3A,$68,$42,$CB,$4A,$8F,$63,$94,$7C,$5F,$15,$5F,$15
	.DB $7A,$7D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $E4,$39,$26,$3A,$26,$3A,$48,$3A,$8A,$42,$0E,$53,$16,$85,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $25,$42,$A7,$4A,$E5,$31,$06,$32,$89,$42,$49,$3A,$6F,$53,$98,$8D
	.DB $5F,$15,$5F,$15,$9A,$7D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $87,$4A,$A7,$4A,$E5,$31,$67,$42,$EA,$52,$27,$3A,$69,$3A,$11,$64
	.DB $57,$85,$5F,$15,$79,$7D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $87,$4A,$25,$42,$46,$42,$C8,$52,$E8,$52,$A8,$4A,$89,$42,$CB,$42
	.DB $73,$74,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $46,$42,$83,$29,$66,$42,$A7,$52,$87,$4A,$6A,$6B,$0A,$5B,$69,$42
	.DB $4E,$5B,$D6,$84,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $05,$3A,$41,$21,$45,$42,$45,$42,$65,$4A,$EC,$7B,$2A,$63,$28,$42
	.DB $E9,$31,$2F,$5B,$38,$8D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $25,$19,$65,$21,$48,$42,$C8,$4A,$69,$63,$C9,$6B,$48,$63,$47,$42
	.DB $86,$29,$C7,$29,$AF,$63,$77,$9D,$B9,$95,$5F,$15,$5F,$15,$5F,$15
	.DB $C6,$31,$E6,$31,$A9,$4A,$AB,$6B,$0B,$74,$CA,$73,$A9,$6B,$67,$4A
	.DB $E7,$39,$65,$21,$08,$32,$F1,$6B,$57,$8D,$5F,$15,$5F,$15,$5F,$15
	.DB $E6,$31,$06,$3A,$87,$4A,$EC,$73,$0B,$74,$47,$5B,$28,$63,$26,$42
	.DB $85,$31,$03,$19,$25,$19,$29,$3A,$0D,$4B,$33,$6C,$78,$8D,$5F,$15
	.DB $67,$42,$C8,$52,$29,$5B,$0C,$7C,$2B,$7C,$68,$63,$E7,$5A,$C5,$39
	.DB $65,$31,$23,$21,$24,$21,$66,$29,$A7,$29,$AB,$42,$53,$74,$16,$85
	.DB $E4,$31,$A6,$4A,$28,$5B,$48,$63,$68,$63,$27,$63,$24,$42,$E5,$39
	.DB $E7,$41,$44,$29,$A2,$18,$24,$21,$08,$42,$AA,$4A,$EC,$52,$31,$74
	.DB $66,$42,$48,$5B,$AA,$73,$48,$63,$69,$6B,$8A,$6B,$87,$52,$C6,$39
	.DB $86,$31,$03,$21,$C2,$18,$E2,$20,$44,$29,$07,$42,$0B,$5B,$50,$7C
	.DB $A6,$4A,$E6,$52,$27,$5B,$E7,$5A,$A7,$52,$87,$52,$67,$4A,$C6,$31
	.DB $45,$21,$04,$21,$65,$29,$03,$21,$40,$10,$23,$29,$0A,$63,$0E,$7C
	.DB $28,$5B,$E6,$52,$C6,$52,$07,$5B,$66,$4A,$C4,$39,$68,$4A,$C7,$31
	.DB $E8,$31,$C7,$31,$A6,$31,$85,$31,$23,$29,$23,$29,$C5,$41,$C9,$5A
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$F8,$A5
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$F9,$AD,$95,$9D
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$7A,$AE,$D7,$9D,$13,$95,$D2,$94,$8C,$7B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$D7,$9D,$12,$8D,$2E,$7C,$8B,$73,$6B,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$18,$9E,$54,$95,$B0,$84,$EB,$73,$27,$63,$C6,$5A
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$D7,$9D,$D1,$8C,$4D,$84,$0A,$7C,$67,$6B,$05,$63
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$19,$A6,$54,$9D,$2D,$7C,$89,$73,$67,$6B,$46,$6B,$08,$7C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5A,$AE,$B6,$A5,$B1,$8C,$EC,$7B,$07,$6B,$E5,$62,$67,$73,$E8,$7B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $17,$A6,$F2,$8C,$2E,$84,$0D,$8C,$08,$73,$85,$62,$89,$7B,$27,$6B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$18,$9E
	.DB $F6,$A5,$4F,$7C,$AC,$7B,$2D,$94,$09,$7B,$45,$5A,$69,$7B,$48,$6B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$D8,$95,$95,$95,$CB,$73
	.DB $0B,$84,$EB,$7B,$E8,$5A,$C8,$4A,$E8,$52,$66,$42,$26,$42,$24,$42
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$D8,$95,$54,$8D,$B0,$84,$4D,$84
	.DB $8A,$73,$6A,$6B,$C8,$52,$87,$4A,$A8,$4A,$25,$3A,$66,$4A,$A6,$52
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$34,$8D,$11,$8D,$0D,$7C,$EC,$83
	.DB $08,$63,$AB,$73,$2A,$63,$87,$4A,$67,$42,$A4,$29,$66,$4A,$C7,$5A
	.DB $5F,$15,$5F,$15,$5F,$15,$95,$9D,$D0,$8C,$CF,$9C,$0B,$84,$49,$73
	.DB $09,$6B,$B0,$9C,$0E,$84,$88,$52,$06,$3A,$42,$21,$05,$42,$A7,$5A
	.DB $5F,$15,$5F,$15,$17,$A6,$6F,$7C,$4D,$8C,$4B,$94,$C9,$8B,$8A,$7B
	.DB $0C,$8C,$52,$AD,$2E,$84,$A8,$52,$27,$42,$63,$29,$C5,$41,$C8,$62
	.DB $5F,$15,$F6,$9D,$93,$9D,$EB,$73,$EB,$83,$A9,$83,$88,$83,$4D,$9C
	.DB $10,$AD,$F0,$A4,$6B,$73,$A8,$52,$68,$52,$E5,$41,$C5,$41,$2A,$73
	.DB $B6,$9D,$32,$8D,$CF,$8C,$2C,$84,$CA,$83,$69,$7B,$68,$83,$2B,$94
	.DB $71,$B5,$6D,$94,$09,$6B,$E9,$62,$47,$52,$C5,$41,$84,$39,$AC,$83
	.DB $B0,$84,$2D,$74,$CB,$73,$6D,$8C,$8A,$7B,$49,$7B,$49,$7B,$27,$73
	.DB $50,$B5,$2C,$8C,$4A,$73,$4A,$6B,$E5,$41,$64,$39,$43,$31,$0E,$94
	.DB $0E,$94,$CD,$8B,$CC,$8B,$0C,$8C,$EE,$A4,$E9,$73,$E9,$7B,$67,$6B
	.DB $06,$63,$EB,$83,$4E,$94,$D1,$A4,$89,$5A,$23,$31,$A5,$39,$6C,$73
	.DB $8B,$83,$09,$73,$E8,$6A,$4D,$94,$91,$B5,$6C,$8C,$88,$73,$89,$73
	.DB $07,$63,$A7,$5A,$E9,$62,$2A,$6B,$07,$4A,$23,$31,$89,$5A,$0E,$8C
	.DB $27,$6B,$48,$73,$E7,$62,$AE,$9C,$D2,$BD,$EF,$A4,$EB,$83,$89,$73
	.DB $28,$6B,$E4,$41,$88,$5A,$47,$52,$27,$4A,$44,$31,$AD,$7B,$F2,$AC
	.DB $46,$6B,$A8,$7B,$27,$6B,$EF,$A4,$51,$B5,$8E,$9C,$CB,$83,$AA,$7B
	.DB $8A,$7B,$67,$52,$0D,$8C,$2B,$73,$A9,$62,$44,$31,$4F,$94,$73,$B5
	.DB $66,$6B,$67,$6B,$06,$63,$AD,$9C,$AE,$9C,$AB,$83,$4A,$7B,$2D,$8C
	.DB $0C,$84,$6B,$73,$53,$B5,$0E,$8C,$EA,$62,$63,$31,$8F,$9C,$72,$B5
	.DB $87,$6B,$47,$63,$E6,$5A,$CA,$7B,$EB,$8B,$CB,$8B,$6E,$A4,$CF,$A4
	.DB $0C,$8C,$CB,$83,$11,$AD,$CD,$83,$88,$5A,$C4,$41,$6F,$94,$B3,$BD
	.DB $89,$6B,$68,$63,$07,$5B,$48,$6B,$EB,$8B,$2C,$9C,$B2,$CD,$AE,$A4
	.DB $8A,$7B,$AA,$7B,$EC,$83,$4A,$73,$87,$5A,$C8,$62,$4E,$94,$72,$B5
	.DB $AA,$73,$48,$5B,$07,$5B,$48,$6B,$6D,$9C,$2C,$9C,$0F,$BD,$2C,$94
	.DB $07,$63,$89,$73,$08,$6B,$09,$6B,$C8,$62,$AB,$7B,$2D,$8C,$AF,$9C
	.DB $45,$4A,$EB,$83,$2C,$94,$CB,$83,$E8,$6A,$C7,$62,$66,$5A,$66,$52
	.DB $C7,$62,$A7,$5A,$A8,$5A,$93,$BD,$36,$CE,$32,$AD,$EE,$83,$88,$52
	.DB $E7,$62,$CA,$83,$2C,$8C,$69,$7B,$28,$73,$A6,$5A,$25,$4A,$25,$4A
	.DB $46,$52,$A7,$5A,$86,$52,$EC,$83,$52,$B5,$15,$CE,$32,$AD,$A8,$5A
	.DB $28,$6B,$69,$73,$CA,$83,$28,$6B,$48,$73,$A6,$5A,$E3,$41,$25,$4A
	.DB $E4,$41,$A7,$5A,$25,$4A,$25,$4A,$0D,$8C,$F4,$C5,$B0,$9C,$8F,$94
	.DB $E8,$6A,$28,$6B,$89,$7B,$89,$7B,$48,$6B,$E7,$5A,$24,$42,$25,$4A
	.DB $E5,$41,$46,$4A,$E4,$41,$C3,$41,$CB,$7B,$F0,$A4,$6A,$73,$CF,$9C
	.DB $E8,$6A,$69,$7B,$69,$7B,$6D,$94,$08,$63,$48,$6B,$66,$4A,$C5,$39
	.DB $E6,$41,$C5,$41,$A4,$39,$45,$4A,$0C,$84,$AE,$9C,$EB,$83,$AB,$7B
	.DB $4A,$7B,$8A,$83,$29,$73,$AE,$9C,$E8,$62,$6A,$6B,$47,$4A,$C5,$39
	.DB $07,$42,$84,$31,$84,$31,$05,$42,$6A,$73,$2C,$8C,$10,$A5,$69,$6B
	.DB $2E,$94,$8B,$83,$C8,$6A,$6E,$94,$8B,$7B,$0D,$7C,$88,$4A,$E7,$39
	.DB $A7,$39,$44,$29,$85,$31,$63,$31,$45,$4A,$28,$6B,$8E,$94,$6A,$73
	.DB $F1,$B4,$AC,$83,$C9,$6A,$4E,$94,$90,$9C,$F1,$9C,$2B,$63,$E8,$39
	.DB $25,$29,$04,$21,$E7,$41,$43,$29,$A3,$31,$65,$52,$8A,$73,$8E,$94
	.DB $4B,$73,$6F,$94,$8C,$7B,$8F,$9C,$33,$B5,$6F,$94,$2A,$6B,$6B,$6B
	.DB $27,$42,$C6,$31,$86,$31,$C6,$39,$67,$4A,$EB,$7B,$48,$6B,$48,$6B
	.DB $B0,$9C,$52,$B5,$ED,$83,$B0,$9C,$F1,$A4,$0E,$8C,$0E,$84,$90,$94
	.DB $E9,$5A,$48,$4A,$65,$29,$85,$31,$46,$4A,$8A,$73,$68,$6B,$C6,$5A
	.DB $93,$BD,$31,$AD,$6A,$73,$2D,$8C,$AF,$9C,$2D,$8C,$F1,$A4,$90,$94
	.DB $A9,$52,$49,$52,$25,$29,$24,$29,$47,$52,$8A,$73,$0B,$84,$CA,$7B
	.DB $31,$AD,$8E,$9C,$28,$6B,$CB,$83,$10,$A5,$AF,$9C,$12,$AD,$6F,$94
	.DB $68,$52,$AA,$5A,$25,$31,$E3,$28,$06,$4A,$E8,$62,$AA,$7B,$CF,$9C
	.DB $CF,$9C,$CF,$A4,$EC,$83,$CB,$83,$92,$BD,$72,$B5,$D0,$A4,$C9,$62
	.DB $24,$31,$08,$4A,$25,$31,$04,$29,$68,$5A,$EC,$83,$10,$A5,$6D,$94
	.DB $10,$AD,$71,$B5,$4D,$94,$49,$73,$72,$B5,$D4,$C5,$F2,$AC,$CD,$83
	.DB $89,$52,$EC,$6A,$C7,$41,$04,$29,$A6,$41,$6B,$73,$AF,$9C,$AF,$9C
	.DB $92,$BD,$92,$BD,$0C,$8C,$29,$6B,$93,$BD,$94,$BD,$33,$AD,$CD,$7B
	.DB $0B,$63,$6D,$73,$6A,$52,$66,$31,$85,$39,$CC,$83,$F0,$A4,$93,$BD
	.DB $92,$BD,$92,$BD,$4D,$94,$2E,$8C,$37,$D6,$54,$B5,$B1,$9C,$ED,$7B
	.DB $6C,$6B,$AF,$7B,$EC,$62,$A7,$39,$44,$31,$AC,$7B,$AF,$9C,$B3,$BD
	.DB $47,$4A,$C6,$39,$E7,$39,$E2,$18,$85,$29,$C2,$08,$65,$21,$46,$29
	.DB $E4,$18,$E4,$18,$25,$21,$04,$21,$44,$29,$E6,$39,$67,$4A,$88,$4A
	.DB $27,$42,$27,$42,$C6,$39,$85,$29,$A5,$29,$23,$19,$44,$19,$25,$21
	.DB $C4,$18,$E4,$18,$E4,$18,$C3,$18,$03,$21,$A5,$31,$C5,$39,$88,$4A
	.DB $6B,$73,$2A,$63,$A5,$31,$A5,$29,$44,$21,$C6,$29,$86,$29,$25,$21
	.DB $05,$21,$05,$21,$04,$21,$C3,$18,$44,$21,$C5,$31,$C5,$31,$67,$4A
	.DB $4E,$8C,$8F,$8C,$4B,$63,$68,$4A,$65,$29,$65,$29,$E3,$18,$E4,$18
	.DB $05,$21,$25,$29,$24,$21,$03,$21,$85,$31,$26,$42,$06,$3A,$67,$4A
	.DB $EC,$83,$4E,$8C,$4F,$8C,$47,$4A,$85,$29,$45,$21,$E4,$18,$C4,$18
	.DB $C4,$18,$05,$21,$E4,$18,$E3,$18,$85,$29,$06,$42,$06,$42,$26,$42
	.DB $8B,$73,$8B,$73,$4F,$8C,$A6,$31,$65,$29,$25,$21,$46,$29,$E5,$18
	.DB $C4,$18,$C4,$18,$C3,$18,$C2,$10,$44,$21,$E6,$39,$06,$3A,$25,$42
	.DB $49,$6B,$09,$63,$B1,$9C,$28,$42,$86,$31,$05,$21,$C5,$18,$26,$21
	.DB $C4,$18,$C4,$18,$C3,$18,$C2,$18,$23,$21,$E5,$39,$46,$42,$26,$42
	.DB $6A,$73,$67,$52,$CE,$7B,$C6,$39,$46,$29,$67,$31,$47,$29,$47,$29
	.DB $C4,$18,$A3,$10,$C3,$18,$C2,$18,$23,$21,$06,$3A,$87,$4A,$66,$4A
	.DB $A7,$5A,$A9,$5A,$4E,$6B,$E9,$41,$68,$31,$46,$29,$05,$21,$25,$21
	.DB $04,$19,$E4,$18,$C3,$18,$82,$10,$C3,$10,$65,$29,$28,$42,$AA,$52
	.DB $EC,$7B,$AD,$73,$CB,$5A,$06,$21,$26,$21,$67,$29,$66,$29,$25,$21
	.DB $04,$19,$E4,$18,$C3,$18,$C3,$10,$C3,$18,$45,$29,$C7,$39,$07,$42
	.DB $11,$A5,$6C,$6B,$C8,$39,$06,$21,$87,$29,$66,$21,$04,$19,$05,$19
	.DB $E4,$18,$C4,$18,$C4,$18,$E4,$18,$04,$19,$25,$21,$65,$29,$C6,$39
	.DB $31,$A5,$48,$4A,$45,$29,$87,$31,$87,$29,$04,$11,$04,$19,$04,$19
	.DB $E5,$18,$C4,$18,$C4,$18,$E4,$18,$04,$21,$65,$29,$A6,$31,$07,$42
	.DB $AF,$9C,$C6,$39,$25,$21,$66,$29,$E4,$18,$03,$19,$A6,$31,$04,$19
	.DB $05,$19,$E4,$18,$C4,$18,$C3,$18,$24,$21,$A6,$31,$07,$42,$07,$3A
	.DB $D0,$A4,$E7,$41,$45,$29,$05,$21,$E4,$20,$A6,$31,$E6,$39,$66,$29
	.DB $66,$29,$25,$21,$C3,$18,$C3,$18,$44,$21,$E6,$39,$68,$4A,$E5,$39
	.DB $D1,$A4,$A6,$39,$46,$31,$C4,$20,$46,$31,$A9,$52,$07,$42,$07,$42
	.DB $C7,$31,$65,$29,$E2,$18,$E2,$18,$44,$29,$E6,$39,$47,$42,$05,$3A
	.DB $2E,$8C,$24,$31,$66,$31,$83,$18,$25,$29,$4B,$6B,$EA,$62,$89,$4A
	.DB $28,$42,$65,$29,$E2,$18,$E2,$18,$64,$29,$E5,$39,$05,$3A,$46,$42
	.DB $67,$4A,$66,$4A,$E8,$5A,$45,$42,$45,$42,$24,$42,$45,$42,$E3,$39
	.DB $E4,$39,$84,$29,$45,$29,$46,$29,$C4,$18,$E4,$18,$07,$42,$4A,$6B
	.DB $47,$4A,$46,$42,$C8,$52,$66,$4A,$A7,$52,$86,$4A,$A7,$52,$C4,$31
	.DB $E4,$39,$85,$29,$65,$29,$46,$29,$C4,$18,$C3,$18,$E6,$41,$08,$63
	.DB $46,$42,$05,$3A,$66,$4A,$E4,$39,$46,$42,$25,$42,$26,$42,$87,$4A
	.DB $47,$42,$85,$29,$04,$19,$05,$19,$A3,$10,$E3,$18,$E5,$41,$CB,$7B
	.DB $46,$42,$05,$3A,$05,$42,$63,$29,$C5,$39,$E5,$39,$E6,$39,$2B,$63
	.DB $CA,$52,$A6,$29,$04,$19,$25,$21,$04,$19,$03,$21,$C5,$39,$49,$6B
	.DB $66,$4A,$46,$42,$46,$42,$84,$29,$27,$42,$68,$4A,$89,$52,$2B,$5B
	.DB $CA,$52,$C6,$29,$44,$19,$A6,$29,$A6,$31,$64,$29,$C4,$39,$AB,$7B
	.DB $67,$4A,$46,$42,$46,$42,$84,$29,$27,$42,$68,$4A,$89,$52,$8C,$6B
	.DB $2B,$5B,$C6,$31,$23,$19,$A5,$29,$06,$32,$26,$42,$87,$52,$6E,$94
	.DB $67,$4A,$26,$42,$06,$3A,$64,$29,$07,$42,$27,$42,$C6,$39,$0E,$7C
	.DB $ED,$73,$A8,$4A,$A4,$21,$C4,$29,$06,$32,$46,$3A,$A8,$52,$87,$52
	.DB $67,$4A,$06,$42,$06,$3A,$85,$31,$68,$4A,$28,$42,$A6,$31,$2E,$84
	.DB $8F,$8C,$CC,$6B,$C8,$4A,$87,$42,$25,$32,$C4,$29,$05,$3A,$63,$29
	.DB $A6,$31,$84,$31,$84,$31,$64,$29,$68,$4A,$8C,$73,$6B,$73,$2C,$8C
	.DB $8D,$94,$49,$6B,$06,$42,$44,$29,$E4,$20,$65,$29,$85,$29,$47,$42
	.DB $65,$29,$85,$31,$A5,$31,$64,$29,$68,$4A,$2B,$6B,$E9,$62,$2C,$8C
	.DB $4C,$8C,$29,$63,$06,$42,$65,$31,$24,$29,$C7,$39,$28,$42,$68,$4A
	.DB $A5,$31,$E6,$39,$C6,$39,$A5,$31,$68,$4A,$0A,$63,$A8,$5A,$07,$63
	.DB $28,$6B,$66,$4A,$C5,$39,$65,$29,$E3,$20,$44,$29,$85,$31,$C5,$39
	.DB $27,$42,$68,$4A,$06,$42,$C5,$39,$67,$4A,$E9,$5A,$A7,$52,$C6,$5A
	.DB $E7,$5A,$25,$42,$E6,$41,$A6,$39,$04,$21,$44,$29,$65,$29,$26,$42
	.DB $67,$4A,$A8,$52,$47,$4A,$25,$42,$67,$4A,$C8,$5A,$C7,$5A,$69,$6B
	.DB $28,$63,$25,$42,$A5,$31,$85,$31,$24,$21,$85,$31,$C6,$39,$67,$4A
	.DB $67,$4A,$C8,$52,$C8,$5A,$C7,$52,$C7,$52,$E7,$5A,$08,$63,$48,$6B
	.DB $28,$63,$05,$3A,$84,$31,$65,$29,$25,$21,$65,$29,$65,$29,$47,$42
	.DB $66,$4A,$A7,$52,$E7,$5A,$07,$5B,$E7,$5A,$07,$5B,$68,$6B,$27,$63
	.DB $28,$63,$46,$42,$A5,$31,$C6,$31,$86,$31,$86,$29,$24,$21,$0A,$5B
	.DB $65,$42,$65,$4A,$85,$4A,$C6,$52,$A5,$4A,$C5,$52,$88,$6B,$07,$63
	.DB $08,$63,$E4,$39,$23,$21,$24,$21,$45,$21,$86,$29,$44,$19,$0B,$5B
	.DB $4D,$84,$2C,$7C,$4D,$7C,$54,$95,$D8,$9D,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $2C,$84,$2B,$84,$CB,$6B,$2E,$74,$92,$7C,$36,$85,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $8D,$9C,$8D,$94,$0B,$7C,$CB,$73,$AD,$6B,$31,$74,$77,$95,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $A9,$7B,$C9,$83,$CA,$7B,$EB,$7B,$AC,$73,$AC,$6B,$10,$6C,$D8,$95
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $AA,$83,$89,$7B,$89,$7B,$CA,$7B,$6A,$73,$E8,$5A,$A9,$42,$F4,$7C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $4E,$94,$0C,$8C,$8A,$7B,$08,$63,$A7,$5A,$46,$42,$C6,$31,$6E,$5B
	.DB $36,$8D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $E9,$62,$4A,$73,$E9,$62,$87,$52,$A8,$52,$67,$4A,$C6,$31,$E8,$31
	.DB $6F,$5B,$57,$95,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $47,$4A,$2A,$63,$2B,$6B,$0A,$63,$2B,$63,$C9,$52,$85,$29,$04,$19
	.DB $A8,$29,$D2,$6B,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $26,$3A,$4A,$63,$2D,$7C,$CD,$73,$EE,$7B,$89,$52,$49,$4A,$06,$21
	.DB $89,$29,$69,$19,$D6,$7C,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $A8,$52,$E8,$5A,$C8,$52,$4A,$6B,$94,$AD,$30,$8C,$08,$42,$45,$29
	.DB $46,$21,$67,$19,$D1,$63,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $67,$4A,$A6,$52,$25,$4A,$29,$6B,$77,$D6,$53,$B5,$E6,$41,$06,$42
	.DB $C6,$31,$08,$32,$ED,$42,$B9,$95,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $87,$52,$A6,$52,$65,$52,$86,$52,$B0,$A4,$2E,$94,$A5,$41,$E9,$5A
	.DB $87,$4A,$89,$42,$8A,$3A,$D5,$7C,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $87,$52,$25,$4A,$24,$4A,$C3,$39,$2A,$73,$AC,$83,$C9,$62,$E9,$62
	.DB $67,$4A,$E6,$29,$E7,$21,$B0,$53,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $C8,$52,$04,$3A,$E4,$39,$C4,$39,$2A,$6B,$8D,$7B,$0B,$6B,$27,$4A
	.DB $65,$31,$C3,$10,$25,$11,$8C,$32,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $ED,$73,$87,$4A,$05,$3A,$E5,$39,$C9,$52,$AA,$52,$08,$42,$A7,$39
	.DB $25,$29,$26,$21,$27,$19,$CA,$21,$54,$6C,$5F,$15,$5F,$15,$5F,$15
	.DB $EE,$73,$26,$3A,$84,$21,$C5,$29,$68,$42,$CB,$52,$CC,$5A,$88,$39
	.DB $89,$39,$8D,$52,$CA,$31,$69,$19,$EF,$3A,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$7B,$AE
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$59,$B6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5A,$B6,$58,$B6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$D6,$AD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$BE,$17,$B6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$7A,$BE,$58,$C6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$79,$C6,$58,$C6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$C6,$79,$C6,$16,$C6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$C6,$58,$C6,$F5,$BD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$BE,$38,$C6,$F5,$BD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$79,$BE,$17,$BE,$F5,$BD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$54,$A5
	.DB $8F,$A4,$CC,$83,$D0,$A4,$AB,$7B,$8A,$7B,$C6,$5A,$68,$73,$27,$6B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$F8,$A5,$4F,$8C
	.DB $2E,$94,$6F,$9C,$ED,$8B,$31,$AD,$6A,$73,$C7,$5A,$08,$63,$49,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$39,$B6,$75,$9D,$D1,$8C,$6B,$73
	.DB $ED,$8B,$6E,$9C,$6A,$73,$15,$CE,$AB,$7B,$A7,$5A,$C8,$62,$8A,$7B
	.DB $5F,$15,$5B,$AE,$3A,$B6,$95,$A5,$B1,$94,$2E,$84,$CB,$7B,$8A,$7B
	.DB $0C,$8C,$EB,$83,$0C,$8C,$F3,$C5,$4D,$8C,$A6,$52,$E7,$62,$69,$73
	.DB $5F,$15,$BB,$BE,$F7,$AD,$2E,$84,$09,$6B,$28,$73,$69,$7B,$2B,$8C
	.DB $EA,$83,$A8,$7B,$0E,$A5,$6F,$AD,$8C,$94,$E5,$5A,$C5,$5A,$27,$63
	.DB $9A,$B6,$1A,$D7,$77,$C6,$4D,$8C,$08,$6B,$28,$73,$48,$7B,$CA,$83
	.DB $06,$6B,$67,$73,$8C,$94,$8C,$94,$E9,$7B,$06,$5B,$43,$4A,$06,$5B
	.DB $D6,$AD,$35,$BE,$F4,$BD,$6D,$94,$89,$7B,$28,$73,$C7,$62,$A6,$62
	.DB $A6,$5A,$C6,$5A,$48,$6B,$07,$63,$08,$63,$C6,$52,$44,$4A,$E6,$5A
	.DB $B0,$8C,$2C,$7C,$0B,$7C,$48,$6B,$07,$6B,$E8,$6A,$45,$52,$E5,$49
	.DB $E9,$62,$26,$4A,$67,$52,$A4,$31,$E9,$62,$A7,$52,$E8,$5A,$86,$52
	.DB $4F,$7C,$72,$AD,$69,$6B,$85,$4A,$68,$6B,$06,$5B,$E7,$5A,$25,$3A
	.DB $66,$42,$25,$3A,$A7,$4A,$A7,$4A,$6A,$63,$E7,$52,$A6,$52,$07,$5B
	.DB $90,$8C,$AF,$94,$29,$6B,$07,$63,$89,$73,$89,$6B,$E7,$5A,$E7,$5A
	.DB $A7,$52,$65,$42,$28,$5B,$08,$5B,$08,$5B,$65,$42,$65,$4A,$A6,$52
	.DB $94,$B5,$11,$A5,$EC,$83,$8A,$7B,$29,$6B,$AB,$7B,$49,$6B,$28,$6B
	.DB $65,$52,$44,$4A,$69,$6B,$8A,$73,$28,$63,$A6,$52,$07,$5B,$85,$4A
	.DB $D5,$BD,$93,$BD,$8F,$9C,$4D,$94,$6A,$7B,$4A,$73,$49,$73,$EB,$83
	.DB $07,$6B,$C6,$5A,$89,$7B,$CA,$7B,$27,$63,$A5,$52,$C6,$5A,$E7,$5A
	.DB $32,$AD,$52,$B5,$AF,$A4,$51,$B5,$6E,$9C,$E9,$62,$08,$63,$AA,$83
	.DB $07,$6B,$E7,$62,$48,$73,$CA,$7B,$68,$73,$48,$6B,$48,$63,$C6,$52
	.DB $32,$B5,$72,$BD,$F0,$B4,$D3,$C5,$6E,$9C,$A7,$5A,$28,$6B,$E7,$6A
	.DB $A5,$62,$C6,$62,$07,$6B,$CA,$7B,$CA,$7B,$0A,$84,$EA,$7B,$E6,$5A
	.DB $D4,$C5,$71,$BD,$30,$B5,$F3,$C5,$EB,$83,$C7,$5A,$8A,$73,$48,$73
	.DB $E7,$62,$28,$6B,$07,$63,$A9,$73,$48,$6B,$88,$6B,$68,$6B,$C9,$73
	.DB $D3,$C5,$4C,$94,$CE,$A4,$13,$CE,$EB,$83,$49,$6B,$8A,$73,$08,$6B
	.DB $86,$5A,$07,$63,$08,$63,$A9,$7B,$07,$63,$48,$6B,$68,$6B,$A8,$73
	.DB $88,$73,$E6,$62,$E6,$62,$E5,$62,$0A,$8C,$4A,$94,$4B,$94,$07,$6B
	.DB $07,$6B,$07,$73,$E7,$6A,$65,$5A,$45,$5A,$29,$7B,$4E,$9C,$31,$B5
	.DB $08,$6B,$A6,$5A,$C6,$62,$A6,$5A,$07,$6B,$27,$73,$48,$73,$28,$6B
	.DB $A7,$62,$66,$5A,$66,$5A,$66,$5A,$66,$5A,$A7,$6A,$09,$73,$EC,$8B
	.DB $E8,$62,$E8,$62,$29,$6B,$6A,$73,$A7,$62,$86,$5A,$C7,$62,$87,$5A
	.DB $87,$5A,$66,$52,$46,$52,$05,$4A,$05,$4A,$26,$52,$46,$5A,$E9,$72
	.DB $E7,$62,$69,$73,$EB,$83,$6D,$94,$28,$6B,$E7,$62,$C7,$62,$66,$52
	.DB $88,$52,$87,$52,$26,$4A,$C4,$39,$C5,$41,$E5,$49,$06,$4A,$88,$5A
	.DB $E6,$5A,$89,$73,$C9,$7B,$2B,$8C,$48,$6B,$07,$6B,$C7,$62,$09,$63
	.DB $C8,$5A,$46,$4A,$E5,$39,$C5,$39,$E5,$41,$E6,$41,$C5,$41,$26,$4A
	.DB $06,$5B,$47,$6B,$27,$6B,$47,$6B,$E6,$62,$E6,$62,$E6,$62,$8A,$73
	.DB $29,$6B,$C8,$5A,$46,$4A,$C4,$39,$A4,$39,$C5,$39,$06,$4A,$46,$4A
	.DB $E7,$5A,$E7,$5A,$28,$6B,$C7,$5A,$C6,$5A,$C6,$5A,$28,$6B,$AA,$7B
	.DB $8A,$73,$69,$6B,$C8,$5A,$05,$42,$83,$31,$E4,$39,$67,$52,$87,$52
	.DB $A7,$52,$C7,$5A,$8A,$73,$E8,$62,$C8,$62,$C7,$5A,$6A,$73,$2C,$8C
	.DB $89,$7B,$27,$6B,$08,$63,$C7,$5A,$46,$4A,$25,$42,$46,$4A,$87,$52
	.DB $08,$63,$E6,$5A,$C6,$5A,$27,$6B,$07,$6B,$C6,$62,$C6,$62,$2B,$94
	.DB $67,$7B,$06,$6B,$EA,$8B,$68,$73,$E7,$62,$C3,$39,$05,$4A,$E6,$49
	.DB $08,$63,$48,$6B,$28,$63,$E7,$62,$C6,$5A,$E7,$62,$48,$73,$47,$73
	.DB $88,$7B,$E9,$83,$4B,$94,$27,$6B,$65,$52,$82,$31,$84,$39,$64,$31
	.DB $A6,$52,$A6,$52,$A6,$52,$C6,$5A,$C6,$5A,$C7,$5A,$E7,$62,$07,$6B
	.DB $A9,$7B,$EA,$83,$EB,$83,$48,$6B,$28,$6B,$46,$4A,$43,$29,$44,$29
	.DB $C6,$52,$E6,$5A,$E7,$5A,$07,$5B,$28,$63,$48,$6B,$28,$6B,$07,$6B
	.DB $A9,$7B,$89,$73,$48,$73,$28,$6B,$EB,$7B,$8B,$73,$A4,$31,$E3,$18
	.DB $E6,$5A,$68,$63,$48,$63,$85,$52,$A5,$52,$68,$6B,$A9,$73,$07,$63
	.DB $A9,$7B,$A9,$7B,$27,$6B,$C7,$5A,$8A,$73,$EC,$7B,$E5,$39,$E2,$18
	.DB $C5,$52,$27,$63,$06,$5B,$44,$4A,$A5,$52,$68,$6B,$88,$73,$EA,$83
	.DB $0A,$8C,$CA,$83,$68,$73,$07,$6B,$29,$6B,$EC,$7B,$C5,$39,$44,$21
	.DB $27,$63,$47,$6B,$67,$6B,$E5,$5A,$67,$73,$2A,$8C,$0A,$8C,$6B,$9C
	.DB $2A,$94,$0A,$8C,$EA,$83,$69,$73,$28,$6B,$4E,$8C,$A9,$5A,$44,$29
	.DB $67,$6B,$E9,$7B,$C8,$7B,$C5,$5A,$46,$6B,$6B,$94,$AC,$A4,$09,$8C
	.DB $2A,$94,$8C,$9C,$6B,$9C,$27,$73,$65,$52,$CF,$9C,$6F,$94,$64,$31
	.DB $B3,$C5,$AF,$A4,$AC,$7B,$B0,$9C,$93,$B5,$98,$D6,$D4,$B5,$11,$A5
	.DB $D0,$9C,$F2,$9C,$28,$4A,$28,$4A,$23,$21,$C4,$39,$4E,$8C,$52,$AD
	.DB $B3,$C5,$B4,$C5,$6F,$9C,$F1,$A4,$93,$B5,$36,$CE,$11,$A5,$F4,$BD
	.DB $11,$A5,$2F,$8C,$A6,$39,$C7,$39,$44,$29,$84,$31,$AC,$73,$4E,$8C
	.DB $73,$BD,$F5,$CD,$4E,$9C,$6F,$9C,$52,$B5,$15,$CE,$11,$AD,$F4,$C5
	.DB $D4,$BD,$12,$A5,$AA,$5A,$86,$31,$03,$21,$A5,$39,$8F,$94,$4B,$6B
	.DB $B0,$A4,$93,$C5,$ED,$93,$2E,$94,$31,$B5,$14,$D6,$72,$BD,$35,$CE
	.DB $F4,$BD,$B1,$9C,$0C,$6B,$A6,$39,$45,$31,$64,$29,$6B,$73,$6C,$73
	.DB $0E,$8C,$73,$BD,$8F,$A4,$8F,$A4,$AF,$AC,$71,$BD,$B3,$C5,$55,$D6
	.DB $15,$C6,$91,$9C,$4D,$73,$86,$39,$24,$29,$23,$29,$C9,$5A,$88,$52
	.DB $4B,$73,$D0,$A4,$4E,$9C,$0D,$94,$AB,$83,$6D,$9C,$D3,$CD,$B3,$C5
	.DB $93,$B5,$2F,$8C,$0C,$6B,$25,$29,$E3,$20,$A5,$39,$8B,$73,$E9,$62
	.DB $87,$52,$6A,$73,$08,$6B,$49,$73,$28,$73,$0C,$94,$D3,$CD,$B3,$BD
	.DB $4E,$94,$27,$4A,$66,$39,$25,$31,$A6,$39,$47,$4A,$E9,$62,$6F,$8C
	.DB $C4,$39,$25,$4A,$25,$4A,$49,$73,$AA,$83,$4C,$94,$92,$BD,$B3,$BD
	.DB $4E,$94,$48,$52,$66,$39,$25,$31,$65,$39,$67,$52,$29,$6B,$A7,$52
	.DB $C6,$41,$44,$39,$07,$52,$88,$5A,$A7,$62,$EB,$8B,$6D,$9C,$4D,$94
	.DB $11,$AD,$2A,$6B,$63,$31,$47,$4A,$89,$52,$CF,$73,$F0,$7B,$C7,$31
	.DB $64,$31,$24,$29,$E6,$41,$26,$4A,$A4,$41,$25,$52,$45,$52,$28,$6B
	.DB $2D,$8C,$E8,$62,$C4,$39,$C9,$5A,$2B,$63,$D2,$9C,$75,$AD,$E8,$39
	.DB $65,$31,$85,$31,$69,$52,$88,$5A,$06,$42,$E5,$41,$C3,$39,$86,$52
	.DB $AA,$7B,$29,$63,$67,$52,$E9,$5A,$A8,$52,$0F,$7C,$71,$8C,$89,$52
	.DB $E3,$18,$24,$29,$E7,$39,$48,$4A,$06,$42,$C4,$39,$C4,$39,$04,$42
	.DB $C7,$5A,$A7,$52,$66,$4A,$A7,$52,$06,$42,$6C,$6B,$CE,$7B,$68,$4A
	.DB $A2,$10,$C3,$18,$25,$21,$85,$31,$85,$31,$43,$29,$42,$29,$46,$42
	.DB $66,$4A,$25,$42,$05,$3A,$06,$42,$A5,$31,$6C,$73,$8C,$73,$C5,$39
	.DB $C3,$18,$04,$21,$45,$29,$A6,$31,$E6,$39,$83,$29,$83,$29,$06,$3A
	.DB $E5,$31,$C5,$31,$E6,$39,$C5,$31,$43,$29,$A9,$5A,$27,$4A,$06,$42
	.DB $A2,$10,$E3,$18,$04,$21,$85,$31,$06,$42,$A4,$31,$C5,$31,$23,$19
	.DB $03,$19,$64,$21,$C6,$31,$A6,$31,$65,$31,$CA,$5A,$C6,$39,$A9,$5A
	.DB $A2,$18,$C3,$18,$A2,$18,$24,$29,$E6,$39,$A4,$31,$E5,$39,$E7,$31
	.DB $65,$21,$66,$29,$86,$29,$24,$21,$45,$29,$2C,$6B,$48,$52,$69,$52
	.DB $F1,$9C,$64,$31,$C3,$20,$E3,$20,$C5,$39,$E4,$39,$49,$6B,$67,$5A
	.DB $C6,$49,$24,$31,$A2,$20,$04,$29,$E8,$49,$49,$52,$E6,$41,$E5,$39
	.DB $ED,$7B,$48,$4A,$66,$31,$04,$29,$28,$4A,$C4,$39,$25,$4A,$67,$52
	.DB $A5,$39,$E3,$20,$62,$18,$82,$10,$25,$29,$A6,$39,$85,$31,$26,$42
	.DB $EA,$5A,$AB,$5A,$26,$29,$A3,$18,$08,$42,$C5,$39,$A4,$31,$A4,$31
	.DB $23,$21,$03,$21,$04,$21,$E4,$18,$24,$21,$85,$29,$A5,$31,$26,$42
	.DB $8D,$73,$CC,$5A,$26,$29,$C5,$18,$66,$31,$A5,$31,$84,$31,$43,$21
	.DB $C2,$10,$E3,$10,$45,$21,$04,$19,$C3,$10,$44,$21,$A5,$31,$0A,$5B
	.DB $8D,$73,$C8,$39,$26,$29,$87,$31,$04,$21,$C6,$39,$27,$42,$0A,$53
	.DB $27,$3A,$C6,$29,$C6,$29,$65,$21,$44,$21,$07,$3A,$EA,$52,$D1,$94
	.DB $0E,$84,$45,$29,$25,$21,$A7,$31,$04,$21,$26,$42,$0A,$5B,$A9,$4A
	.DB $A5,$29,$03,$19,$23,$19,$E3,$18,$E3,$18,$85,$31,$47,$42,$D0,$9C
	.DB $B0,$94,$07,$42,$C7,$39,$65,$29,$64,$29,$46,$42,$08,$5B,$88,$4A
	.DB $84,$29,$03,$21,$44,$21,$44,$29,$64,$29,$A5,$31,$E5,$39,$08,$5B
	.DB $A8,$52,$64,$29,$C6,$39,$03,$21,$27,$42,$86,$4A,$C7,$52,$4A,$6B
	.DB $06,$42,$23,$21,$03,$21,$23,$29,$64,$31,$C4,$39,$C4,$39,$62,$29
	.DB $45,$21,$24,$21,$64,$29,$E6,$39,$E6,$39,$E5,$39,$4A,$6B,$0A,$63
	.DB $84,$31,$E2,$18,$65,$29,$66,$29,$04,$21,$E3,$18,$04,$21,$E2,$18
	.DB $24,$21,$24,$21,$44,$29,$27,$42,$A5,$31,$84,$29,$88,$52,$E5,$39
	.DB $C5,$39,$C5,$39,$A6,$39,$A6,$31,$45,$29,$E4,$18,$E3,$18,$64,$29
	.DB $65,$29,$65,$29,$03,$19,$E6,$39,$63,$29,$A4,$31,$A8,$52,$66,$4A
	.DB $A8,$52,$67,$4A,$07,$42,$07,$42,$86,$31,$E4,$18,$03,$19,$E2,$18
	.DB $E6,$39,$48,$4A,$23,$21,$C5,$39,$64,$29,$26,$42,$A7,$52,$29,$63
	.DB $C7,$5A,$05,$3A,$27,$42,$CA,$5A,$E8,$39,$C3,$18,$E4,$18,$03,$21
	.DB $27,$4A,$8C,$73,$06,$42,$26,$42,$C5,$39,$66,$4A,$05,$42,$45,$4A
	.DB $24,$42,$83,$31,$06,$42,$0B,$5B,$49,$4A,$E4,$18,$C3,$10,$45,$29
	.DB $C9,$5A,$0E,$84,$88,$52,$26,$42,$26,$42,$A7,$52,$25,$42,$C3,$39
	.DB $25,$42,$C4,$39,$64,$29,$E7,$39,$08,$42,$46,$29,$E4,$18,$25,$21
	.DB $C9,$5A,$6B,$6B,$67,$52,$E5,$39,$66,$4A,$A7,$52,$66,$4A,$86,$52
	.DB $87,$52,$06,$3A,$24,$21,$04,$21,$67,$29,$67,$29,$05,$21,$C3,$18
	.DB $E6,$41,$68,$52,$47,$52,$E5,$41,$66,$4A,$45,$42,$E4,$39,$E8,$5A
	.DB $26,$42,$A5,$31,$85,$31,$45,$29,$46,$29,$26,$21,$E4,$18,$66,$29
	.DB $66,$4A,$65,$42,$07,$5B,$C5,$4A,$47,$5B,$47,$5B,$89,$63,$6E,$8C
	.DB $4A,$6B,$27,$42,$43,$21,$C2,$10,$85,$29,$E2,$18,$E6,$39,$8C,$73
	.DB $66,$4A,$65,$42,$E6,$52,$A5,$4A,$28,$5B,$27,$5B,$28,$63,$F0,$A4
	.DB $CC,$7B,$C9,$5A,$85,$29,$23,$21,$C7,$31,$85,$29,$48,$4A,$4B,$6B
	.DB $09,$5B,$08,$5B,$86,$4A,$24,$3A,$08,$5B,$08,$5B,$86,$4A,$51,$AD
	.DB $4E,$8C,$AC,$73,$C6,$31,$24,$21,$65,$29,$24,$21,$86,$31,$48,$4A
	.DB $E9,$5A,$86,$4A,$45,$42,$45,$42,$A7,$52,$87,$4A,$46,$42,$6A,$73
	.DB $2A,$6B,$CD,$7B,$07,$3A,$65,$29,$45,$21,$45,$21,$66,$31,$45,$29
	.DB $C8,$52,$A3,$31,$25,$42,$C8,$52,$46,$4A,$26,$42,$67,$52,$25,$4A
	.DB $67,$52,$0E,$7C,$A9,$52,$A6,$31,$66,$29,$66,$29,$87,$31,$25,$29
	.DB $4D,$84,$AB,$73,$86,$52,$66,$4A,$26,$42,$67,$52,$67,$52,$05,$4A
	.DB $87,$52,$70,$8C,$AD,$73,$28,$42,$A7,$31,$46,$29,$66,$29,$86,$31
	.DB $8E,$8C,$51,$A5,$49,$63,$86,$52,$67,$4A,$C9,$5A,$06,$42,$C4,$39
	.DB $47,$52,$90,$94,$D2,$94,$2C,$63,$AB,$52,$C7,$31,$A7,$31,$86,$31
	.DB $86,$52,$0B,$84,$EB,$7B,$EC,$7B,$E9,$5A,$A8,$52,$E5,$41,$26,$4A
	.DB $C9,$5A,$12,$A5,$F7,$BD,$10,$7C,$4D,$63,$86,$29,$04,$21,$44,$29
	.DB $A4,$31,$87,$52,$0D,$84,$31,$A5,$AC,$7B,$2A,$6B,$05,$42,$26,$4A
	.DB $6B,$73,$6F,$94,$F6,$C5,$74,$B5,$92,$94,$46,$21,$88,$29,$06,$19
	.DB $80,$08,$05,$3A,$B0,$94,$8F,$94,$F1,$A4,$E9,$62,$47,$52,$05,$42
	.DB $2D,$8C,$72,$B5,$B8,$DE,$B4,$BD,$2F,$84,$E7,$39,$45,$21,$46,$19
	.DB $E6,$39,$6C,$6B,$4B,$6B,$ED,$7B,$ED,$83,$A9,$5A,$C9,$62,$C8,$5A
	.DB $0C,$84,$6E,$94,$31,$AD,$AF,$9C,$8C,$73,$2A,$63,$C6,$31,$A6,$29
	.DB $03,$21,$23,$21,$47,$4A,$AD,$7B,$27,$4A,$0A,$63,$6C,$73,$E9,$62
	.DB $2A,$6B,$2D,$8C,$F0,$AC,$F1,$A4,$4F,$94,$90,$94,$2B,$63,$27,$3A
	.DB $85,$31,$27,$42,$8C,$73,$06,$42,$AD,$7B,$33,$AD,$4B,$73,$A8,$5A
	.DB $26,$52,$2E,$94,$90,$9C,$6F,$9C,$2E,$94,$B1,$9C,$91,$94,$85,$31
	.DB $40,$08,$88,$4A,$AC,$73,$88,$52,$E9,$62,$2A,$6B,$4A,$6B,$8C,$7B
	.DB $67,$5A,$ED,$8B,$0D,$94,$4E,$94,$4F,$9C,$B1,$A4,$F2,$A4,$CE,$7B
	.DB $03,$21,$06,$3A,$83,$31,$0D,$84,$8F,$9C,$8B,$73,$C8,$62,$EC,$83
	.DB $49,$73,$EB,$8B,$8E,$A4,$30,$B5,$51,$B5,$31,$B5,$8F,$9C,$B0,$94
	.DB $07,$3A,$02,$19,$46,$4A,$4A,$6B,$CB,$7B,$6A,$73,$C7,$62,$A5,$5A
	.DB $C5,$62,$43,$52,$A4,$62,$E5,$6A,$47,$7B,$8C,$9C,$2C,$8C,$11,$A5
	.DB $4A,$6B,$09,$63,$C3,$39,$A6,$52,$69,$6B,$CB,$73,$CC,$73,$0A,$5B
	.DB $75,$A5,$8F,$6B,$90,$63,$2B,$32,$89,$19,$95,$74,$5F,$15,$5F,$15
	.DB $C9,$5A,$46,$52,$63,$31,$46,$4A,$8A,$6B,$8A,$6B,$E8,$52,$6B,$6B
	.DB $37,$C6,$92,$8C,$0D,$5B,$8C,$42,$CA,$21,$0F,$4B,$5F,$15,$5F,$15
	.DB $C6,$41,$64,$31,$02,$21,$84,$31,$4B,$6B,$0A,$5B,$27,$42,$88,$4A
	.DB $12,$9D,$B2,$94,$29,$3A,$E8,$31,$27,$19,$47,$11,$B1,$5B,$5F,$15
	.DB $24,$29,$03,$29,$24,$29,$65,$31,$6D,$6B,$EB,$5A,$27,$42,$88,$52
	.DB $4E,$8C,$12,$9D,$AA,$52,$C8,$31,$26,$21,$47,$19,$A9,$21,$F6,$84
	.DB $25,$29,$45,$31,$07,$4A,$AA,$5A,$EF,$83,$EB,$5A,$E7,$39,$2A,$63
	.DB $12,$A5,$F2,$9C,$EB,$5A,$6A,$4A,$87,$29,$A8,$29,$E5,$10,$4F,$53
	.DB $24,$29,$65,$31,$69,$52,$CF,$83,$30,$84,$69,$4A,$C6,$31,$C9,$52
	.DB $B5,$B5,$8D,$73,$A7,$31,$4A,$4A,$47,$21,$27,$19,$E6,$10,$C9,$29
	.DB $03,$29,$24,$29,$27,$4A,$EF,$83,$4C,$6B,$27,$42,$CA,$52,$CA,$5A
	.DB $75,$AD,$EC,$5A,$46,$29,$C8,$31,$47,$29,$26,$19,$47,$19,$06,$09
	.DB $E3,$20,$23,$29,$A5,$39,$8C,$73,$88,$52,$68,$4A,$91,$8C,$6D,$6B
	.DB $F3,$9C,$0D,$63,$C8,$39,$E6,$18,$27,$21,$27,$21,$C5,$08,$E5,$08
	.DB $27,$19,$A9,$29,$88,$29,$29,$4A,$4D,$7B,$2B,$7B,$4B,$83,$8C,$83
	.DB $33,$AD,$AD,$7B,$45,$29,$87,$31,$E5,$18,$26,$21,$E6,$18,$47,$29
	.DB $26,$19,$C8,$31,$86,$29,$A5,$31,$27,$52,$88,$62,$6B,$83,$ED,$8B
	.DB $F2,$A4,$6D,$73,$C7,$39,$67,$29,$47,$21,$06,$21,$07,$21,$E5,$20
	.DB $AA,$4A,$4C,$5B,$E7,$39,$63,$29,$67,$52,$29,$73,$8B,$83,$2E,$94
	.DB $0F,$8C,$EB,$62,$E8,$41,$06,$21,$68,$29,$C5,$18,$27,$29,$27,$29
	.DB $C6,$29,$27,$3A,$89,$4A,$06,$42,$84,$39,$87,$5A,$4E,$94,$B1,$9C
	.DB $4C,$73,$6A,$52,$C8,$39,$26,$21,$88,$29,$C5,$18,$27,$21,$26,$29
	.DB $66,$29,$25,$21,$04,$19,$24,$21,$A6,$39,$CA,$62,$6C,$7B,$33,$AD
	.DB $0B,$6B,$69,$4A,$66,$29,$67,$29,$67,$29,$26,$21,$26,$21,$05,$21
	.DB $A7,$31,$62,$08,$46,$29,$87,$31,$C4,$20,$A7,$41,$2D,$73,$D2,$A4
	.DB $0B,$6B,$49,$4A,$25,$21,$87,$29,$46,$21,$66,$21,$05,$19,$04,$19
	.DB $4C,$6B,$65,$29,$E4,$18,$E4,$18,$E4,$20,$66,$39,$45,$31,$8D,$7B
	.DB $0B,$6B,$E7,$41,$45,$29,$65,$21,$66,$21,$C7,$29,$65,$21,$A6,$29
	.DB $ED,$7B,$E6,$39,$85,$29,$65,$29,$E4,$20,$25,$31,$E3,$28,$68,$5A
	.DB $0B,$6B,$C6,$39,$A6,$31,$45,$21,$C7,$29,$28,$32,$08,$32,$2B,$53
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $D5,$84,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $CD,$42,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $47,$11,$D1,$5B,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $E5,$18,$8B,$4A,$F4,$94,$19,$AE,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $E5,$20,$26,$21,$87,$29,$CC,$4A,$56,$95,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $E6,$20,$E5,$20,$C4,$18,$26,$19,$CC,$4A,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $A4,$18,$C4,$20,$26,$29,$E4,$18,$C4,$10,$6A,$42,$73,$7C,$9C,$B6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $E5,$20,$A3,$18,$A3,$18,$E4,$20,$66,$29,$C7,$31,$29,$3A,$B3,$7C
	.DB $35,$8D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $86,$31,$65,$29,$44,$29,$A6,$31,$07,$3A,$E6,$31,$C6,$29,$2B,$53
	.DB $EF,$6B,$D7,$A5,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $07,$3A,$47,$42,$88,$4A,$A8,$4A,$06,$32,$C5,$29,$67,$3A,$29,$5B
	.DB $2A,$5B,$ED,$73,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $4B,$5B,$09,$53,$29,$5B,$AB,$63,$6A,$5B,$09,$4B,$49,$53,$C7,$4A
	.DB $E7,$52,$EC,$73,$87,$4A,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$19,$AE,$34,$9D,$57,$C6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$BE,$17,$B6,$B8,$D6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$B9,$CE,$97,$CE
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$BA,$CE,$B9,$CE,$56,$CE
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$99,$C6,$77,$CE,$D4,$BD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$99,$C6,$57,$CE,$31,$AD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$C6,$78,$C6,$36,$C6,$B2,$BD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$59,$BE,$37,$BE,$15,$C6,$55,$D6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$1C,$CF,$D9,$D6,$77,$D6,$55,$D6,$54,$D6,$D1,$C5
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$57,$C6,$15,$C6,$F4,$CD,$B2,$C5,$0B,$94
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$DB,$CE,$D9,$DE,$D8,$DE,$F3,$C5,$8D,$A4,$06,$6B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$99,$C6,$16,$C6,$93,$BD,$8D,$9C,$68,$7B,$27,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$FC,$CE,$58,$C6,$F1,$A4,$2D,$8C,$6D,$9C,$CE,$AC,$EA,$8B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$FB,$D6,$99,$CE,$52,$AD,$6E,$9C,$CE,$AC,$50,$BD,$6C,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$3E,$CF
	.DB $DC,$CE,$DB,$D6,$98,$D6,$93,$BD,$AE,$A4,$CA,$8B,$27,$73,$CA,$8B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $DC,$CE,$FB,$D6,$78,$CE,$B3,$BD,$10,$B5,$2B,$9C,$27,$7B,$C5,$6A
	.DB $93,$B5,$10,$A5,$71,$BD,$54,$D6,$EA,$83,$EA,$83,$85,$5A,$44,$52
	.DB $48,$73,$CB,$83,$49,$7B,$E7,$62,$07,$6B,$07,$6B,$E7,$62,$48,$6B
	.DB $76,$D6,$75,$D6,$34,$D6,$91,$BD,$07,$6B,$28,$6B,$07,$63,$A6,$62
	.DB $28,$73,$49,$73,$C7,$62,$85,$5A,$E7,$62,$69,$73,$89,$7B,$89,$73
	.DB $F4,$C5,$B2,$BD,$50,$B5,$CE,$A4,$AA,$7B,$CA,$7B,$0B,$84,$E7,$62
	.DB $28,$73,$48,$73,$07,$6B,$A6,$5A,$A6,$5A,$C6,$5A,$07,$63,$27,$6B
	.DB $92,$BD,$AE,$A4,$A9,$83,$48,$73,$A9,$7B,$89,$73,$89,$73,$28,$6B
	.DB $68,$73,$89,$73,$68,$73,$E7,$62,$65,$52,$45,$4A,$85,$52,$85,$5A
	.DB $34,$CE,$D2,$CD,$4B,$94,$68,$7B,$EA,$83,$EA,$83,$CA,$7B,$89,$73
	.DB $88,$73,$48,$6B,$07,$63,$85,$52,$64,$4A,$C6,$5A,$48,$6B,$C6,$5A
	.DB $54,$D6,$B5,$E6,$90,$C5,$EE,$AC,$CD,$9C,$AD,$94,$EA,$7B,$68,$6B
	.DB $48,$6B,$07,$63,$C6,$5A,$65,$4A,$65,$4A,$C6,$5A,$48,$6B,$08,$63
	.DB $13,$D6,$12,$D6,$70,$BD,$70,$BD,$0A,$8C,$0B,$84,$A5,$52,$27,$63
	.DB $27,$63,$06,$5B,$E6,$5A,$C6,$5A,$C6,$52,$A6,$52,$86,$52,$C7,$5A
	.DB $33,$D6,$B0,$C5,$4F,$BD,$90,$BD,$88,$7B,$4C,$84,$48,$63,$47,$63
	.DB $27,$63,$06,$5B,$E6,$52,$27,$5B,$48,$63,$28,$63,$A6,$52,$65,$4A
	.DB $90,$BD,$D1,$C5,$B1,$C5,$ED,$AC,$0A,$8C,$67,$7B,$C5,$6A,$27,$6B
	.DB $C6,$5A,$28,$6B,$48,$6B,$E7,$5A,$28,$63,$07,$63,$24,$42,$24,$42
	.DB $AD,$A4,$2F,$B5,$0E,$AD,$4B,$94,$A9,$83,$26,$73,$A5,$62,$07,$6B
	.DB $07,$63,$E7,$62,$28,$6B,$69,$6B,$07,$63,$65,$4A,$45,$4A,$28,$63
	.DB $A9,$83,$EA,$8B,$A9,$83,$68,$7B,$68,$7B,$06,$6B,$64,$5A,$07,$63
	.DB $07,$63,$65,$52,$86,$52,$48,$6B,$C6,$5A,$44,$4A,$08,$63,$08,$63
	.DB $88,$83,$68,$7B,$06,$6B,$C6,$62,$E6,$62,$E6,$6A,$07,$6B,$28,$6B
	.DB $E7,$62,$65,$4A,$85,$52,$E7,$5A,$85,$52,$65,$4A,$E7,$5A,$65,$4A
	.DB $0B,$8C,$A9,$7B,$27,$6B,$C6,$62,$64,$52,$C6,$62,$EA,$83,$48,$6B
	.DB $C7,$5A,$C7,$5A,$C7,$5A,$65,$4A,$65,$4A,$65,$4A,$04,$3A,$45,$42
	.DB $CA,$8B,$E6,$62,$C6,$62,$48,$6B,$E6,$62,$A5,$5A,$48,$6B,$69,$73
	.DB $A6,$52,$A6,$52,$86,$52,$E3,$39,$44,$42,$C7,$52,$65,$4A,$A6,$52
	.DB $68,$7B,$64,$52,$44,$52,$48,$6B,$48,$6B,$A5,$5A,$E6,$5A,$49,$6B
	.DB $A6,$52,$45,$4A,$45,$4A,$24,$42,$24,$42,$A6,$4A,$08,$5B,$49,$63
	.DB $68,$7B,$07,$6B,$85,$52,$E7,$62,$E7,$62,$E7,$62,$AA,$7B,$E7,$5A
	.DB $C7,$5A,$85,$4A,$E7,$5A,$28,$63,$24,$42,$82,$29,$65,$4A,$69,$6B
	.DB $07,$6B,$C9,$83,$68,$7B,$64,$5A,$67,$7B,$CC,$A4,$2A,$94,$C9,$83
	.DB $EA,$8B,$49,$7B,$49,$7B,$C8,$6A,$68,$62,$2F,$8C,$AD,$7B,$44,$21
	.DB $89,$7B,$88,$7B,$88,$7B,$68,$7B,$2A,$94,$2A,$94,$A8,$83,$CA,$83
	.DB $C7,$6A,$E7,$6A,$6A,$7B,$2A,$73,$AD,$83,$4C,$73,$28,$4A,$24,$21
	.DB $AA,$7B,$48,$73,$0A,$8C,$4B,$94,$CD,$A4,$C9,$8B,$C9,$83,$CE,$A4
	.DB $29,$73,$45,$52,$8B,$83,$0A,$73,$2F,$94,$CE,$8B,$28,$4A,$25,$29
	.DB $69,$73,$27,$6B,$4C,$94,$0B,$8C,$8D,$A4,$C9,$8B,$4C,$9C,$CF,$A4
	.DB $2E,$8C,$26,$52,$D1,$AC,$4C,$7B,$8D,$83,$EF,$8B,$66,$39,$86,$31
	.DB $E7,$62,$E7,$62,$2C,$94,$68,$7B,$6C,$9C,$6C,$9C,$CE,$A4,$52,$B5
	.DB $6F,$94,$27,$52,$70,$9C,$69,$5A,$AA,$62,$0C,$73,$25,$31,$E8,$41
	.DB $A6,$5A,$85,$5A,$AA,$83,$AA,$83,$EF,$AC,$50,$BD,$72,$BD,$15,$CE
	.DB $AD,$7B,$27,$4A,$EB,$6A,$65,$39,$0C,$73,$69,$5A,$08,$4A,$E7,$41
	.DB $C7,$5A,$45,$52,$E7,$6A,$EB,$8B,$10,$B5,$51,$BD,$92,$C5,$B4,$BD
	.DB $0A,$63,$47,$52,$CA,$62,$C6,$49,$2C,$7B,$C6,$49,$E7,$49,$85,$31
	.DB $08,$63,$65,$52,$25,$52,$CB,$8B,$4D,$9C,$2C,$94,$F0,$AC,$D4,$BD
	.DB $AD,$73,$C6,$41,$27,$52,$23,$31,$E7,$49,$85,$41,$A6,$41,$65,$31
	.DB $49,$6B,$24,$4A,$A7,$5A,$AB,$7B,$AB,$7B,$6E,$94,$8B,$7B,$F1,$A4
	.DB $70,$94,$44,$29,$A7,$39,$E5,$18,$C5,$18,$67,$29,$26,$21,$06,$19
	.DB $E7,$62,$04,$42,$04,$4A,$E7,$62,$49,$73,$AB,$83,$8F,$9C,$32,$AD
	.DB $EE,$83,$45,$29,$66,$31,$05,$21,$88,$31,$67,$29,$C9,$39,$6A,$4A
	.DB $04,$42,$65,$52,$24,$4A,$A7,$5A,$29,$73,$8A,$7B,$15,$CE,$B4,$B5
	.DB $CE,$7B,$A6,$39,$46,$29,$C4,$18,$C8,$39,$A8,$39,$8F,$73,$6E,$73
	.DB $04,$42,$28,$63,$E8,$62,$A7,$5A,$66,$52,$66,$5A,$72,$B5,$73,$AD
	.DB $EE,$7B,$28,$4A,$08,$42,$E4,$20,$67,$31,$09,$4A,$B3,$9C,$EB,$62
	.DB $A6,$52,$49,$6B,$CB,$7B,$AB,$7B,$E8,$62,$29,$6B,$11,$AD,$D0,$9C
	.DB $8C,$73,$27,$4A,$2C,$6B,$87,$39,$25,$29,$4A,$52,$92,$9C,$EB,$62
	.DB $C7,$5A,$86,$52,$CB,$7B,$CF,$9C,$2D,$8C,$8F,$9C,$52,$AD,$11,$A5
	.DB $EA,$62,$A5,$39,$6D,$73,$A7,$39,$25,$29,$6A,$52,$31,$8C,$8D,$7B
	.DB $C7,$5A,$45,$42,$AB,$7B,$F0,$9C,$0D,$84,$AB,$7B,$0D,$8C,$73,$B5
	.DB $47,$4A,$27,$4A,$EF,$83,$86,$39,$25,$29,$E8,$41,$8E,$7B,$0B,$6B
	.DB $A7,$52,$C8,$5A,$6E,$8C,$72,$B5,$4E,$8C,$6A,$73,$2E,$8C,$11,$A5
	.DB $C5,$39,$4B,$73,$34,$AD,$08,$4A,$46,$31,$E4,$20,$CB,$5A,$48,$4A
	.DB $E3,$18,$C2,$10,$E3,$18,$E3,$18,$A2,$18,$E3,$28,$86,$39,$64,$29
	.DB $84,$29,$26,$42,$67,$4A,$05,$3A,$C5,$39,$66,$4A,$E8,$5A,$C8,$5A
	.DB $E3,$18,$C3,$10,$E4,$18,$E3,$18,$C3,$18,$E3,$20,$45,$31,$65,$29
	.DB $85,$29,$C5,$31,$C5,$31,$63,$29,$63,$29,$05,$42,$87,$52,$A7,$52
	.DB $C3,$18,$A3,$10,$C3,$10,$E4,$18,$A3,$18,$A3,$18,$E4,$28,$65,$31
	.DB $86,$31,$85,$31,$23,$29,$E2,$20,$23,$21,$A4,$31,$05,$42,$25,$42
	.DB $04,$21,$A3,$10,$C4,$10,$E4,$18,$C4,$18,$A3,$18,$C4,$20,$46,$31
	.DB $C7,$39,$A6,$39,$44,$31,$44,$29,$64,$29,$63,$29,$63,$31,$C4,$39
	.DB $66,$29,$04,$19,$E4,$18,$05,$19,$E4,$18,$C4,$18,$C4,$18,$04,$21
	.DB $C7,$39,$08,$42,$E7,$41,$07,$42,$C5,$39,$43,$29,$22,$21,$25,$42
	.DB $66,$29,$04,$19,$04,$19,$04,$19,$E4,$18,$C4,$18,$C4,$18,$C3,$18
	.DB $A6,$31,$27,$42,$48,$42,$88,$4A,$E6,$39,$42,$21,$42,$21,$86,$4A
	.DB $45,$29,$04,$19,$04,$19,$04,$19,$E4,$18,$C4,$10,$A3,$10,$C2,$10
	.DB $A5,$29,$E6,$31,$26,$3A,$88,$4A,$E5,$39,$62,$29,$E4,$39,$A7,$52
	.DB $44,$29,$44,$21,$65,$21,$65,$21,$25,$21,$04,$19,$E4,$18,$E2,$10
	.DB $A5,$29,$C4,$29,$E5,$31,$46,$42,$C4,$31,$83,$29,$67,$4A,$C8,$52
	.DB $67,$29,$C7,$29,$66,$21,$66,$19,$66,$19,$05,$11,$46,$19,$04,$19
	.DB $85,$29,$A6,$31,$24,$21,$C5,$31,$47,$42,$26,$42,$E8,$5A,$E8,$62
	.DB $29,$42,$EC,$5A,$CB,$52,$A6,$29,$86,$29,$C7,$29,$04,$19,$E3,$10
	.DB $23,$19,$64,$21,$64,$29,$26,$42,$87,$52,$25,$42,$66,$52,$A6,$5A
	.DB $4D,$6B,$6D,$6B,$0B,$63,$27,$42,$27,$42,$AA,$52,$A9,$52,$23,$21
	.DB $43,$21,$C5,$31,$06,$42,$A7,$52,$A7,$52,$25,$42,$66,$52,$86,$52
	.DB $AD,$7B,$48,$52,$A5,$39,$68,$5A,$67,$52,$68,$5A,$8C,$7B,$06,$42
	.DB $26,$42,$C8,$52,$C8,$5A,$A7,$52,$86,$52,$45,$4A,$08,$63,$08,$63
	.DB $8C,$7B,$06,$4A,$E5,$49,$E9,$6A,$A8,$62,$67,$5A,$0A,$6B,$A8,$52
	.DB $67,$4A,$C8,$52,$A7,$52,$66,$4A,$86,$52,$A6,$52,$69,$6B,$E7,$5A
	.DB $0B,$6B,$67,$52,$E9,$62,$C8,$62,$A8,$5A,$4A,$73,$88,$5A,$67,$52
	.DB $C4,$31,$E5,$39,$05,$42,$25,$42,$C7,$5A,$E7,$5A,$28,$63,$44,$4A
	.DB $CA,$62,$88,$5A,$4B,$6B,$67,$52,$E5,$41,$0A,$63,$06,$4A,$C5,$39
	.DB $43,$29,$C4,$39,$E5,$39,$25,$42,$E8,$5A,$E7,$62,$85,$52,$A5,$52
	.DB $8D,$73,$6C,$73,$0E,$84,$6C,$73,$06,$42,$27,$4A,$E6,$41,$64,$29
	.DB $43,$29,$47,$4A,$67,$52,$46,$4A,$E7,$5A,$A6,$5A,$44,$4A,$C6,$52
	.DB $26,$4A,$26,$4A,$67,$52,$46,$52,$E5,$49,$E6,$49,$E6,$49,$07,$52
	.DB $E7,$49,$C7,$41,$66,$31,$05,$21,$26,$21,$26,$21,$C3,$10,$85,$29
	.DB $46,$4A,$66,$52,$87,$52,$05,$4A,$C4,$41,$E5,$41,$E5,$41,$47,$52
	.DB $27,$4A,$E7,$41,$65,$31,$C4,$18,$04,$19,$66,$29,$65,$21,$85,$29
	.DB $46,$4A,$A7,$52,$A7,$52,$45,$4A,$25,$42,$46,$4A,$46,$4A,$E5,$41
	.DB $E6,$41,$E6,$39,$65,$29,$C3,$10,$C3,$10,$65,$21,$C5,$29,$A4,$29
	.DB $25,$42,$E7,$5A,$E8,$5A,$86,$4A,$86,$4A,$A6,$52,$66,$4A,$C4,$39
	.DB $E5,$39,$47,$42,$47,$42,$A5,$29,$85,$29,$E6,$31,$06,$32,$46,$3A
	.DB $66,$4A,$08,$5B,$29,$63,$C7,$52,$A6,$52,$65,$4A,$04,$3A,$A4,$31
	.DB $C4,$31,$26,$42,$68,$4A,$06,$3A,$A5,$29,$A5,$29,$C4,$29,$A3,$29
	.DB $87,$4A,$08,$5B,$49,$63,$08,$5B,$C7,$52,$66,$4A,$E4,$39,$C4,$31
	.DB $83,$29,$C5,$31,$07,$42,$C5,$31,$64,$29,$84,$29,$A4,$31,$45,$42
	.DB $86,$4A,$C8,$5A,$29,$63,$C8,$5A,$67,$4A,$46,$4A,$05,$42,$E5,$39
	.DB $84,$31,$C5,$39,$48,$4A,$07,$42,$C6,$39,$26,$42,$87,$4A,$87,$52
	.DB $67,$4A,$87,$52,$C8,$5A,$46,$4A,$A4,$39,$84,$39,$A4,$39,$43,$29
	.DB $03,$21,$64,$31,$E7,$41,$C6,$39,$A5,$39,$27,$4A,$A8,$5A,$09,$63
	.DB $C7,$5A,$86,$52,$86,$52,$86,$52,$66,$4A,$E4,$39,$42,$21,$02,$19
	.DB $C5,$29,$47,$42,$87,$4A,$C7,$52,$E7,$52,$27,$5B,$A9,$6B,$47,$63
	.DB $86,$52,$86,$52,$A7,$5A,$E8,$5A,$A8,$52,$25,$42,$84,$31,$63,$21
	.DB $E5,$31,$25,$3A,$46,$42,$86,$4A,$A6,$4A,$C6,$52,$28,$63,$27,$63
	.DB $25,$42,$04,$42,$46,$4A,$66,$4A,$05,$42,$83,$29,$22,$21,$25,$3A
	.DB $86,$4A,$A6,$52,$A7,$52,$C7,$52,$A7,$52,$66,$52,$A7,$52,$A6,$52
	.DB $86,$52,$25,$42,$46,$4A,$46,$4A,$E5,$39,$A4,$31,$A4,$31,$65,$42
	.DB $C6,$52,$E7,$5A,$C7,$5A,$C7,$5A,$66,$4A,$05,$42,$05,$4A,$66,$52
	.DB $66,$52,$45,$4A,$86,$52,$87,$4A,$46,$42,$46,$42,$87,$4A,$A6,$4A
	.DB $07,$5B,$07,$5B,$C7,$5A,$C7,$5A,$67,$52,$46,$4A,$87,$52,$A8,$5A
	.DB $44,$4A,$85,$4A,$A6,$52,$86,$4A,$45,$42,$45,$42,$86,$4A,$A7,$52
	.DB $C8,$52,$87,$4A,$46,$4A,$66,$4A,$87,$52,$C7,$5A,$29,$6B,$A7,$5A
	.DB $E6,$5A,$27,$63,$07,$5B,$C6,$52,$85,$4A,$A6,$4A,$C7,$4A,$46,$42
	.DB $06,$3A,$A4,$29,$83,$29,$05,$3A,$46,$42,$66,$4A,$C7,$52,$A6,$52
	.DB $27,$63,$68,$63,$06,$5B,$A5,$4A,$85,$4A,$E6,$52,$29,$5B,$88,$42
	.DB $28,$3A,$A5,$29,$C5,$31,$67,$42,$C7,$52,$85,$4A,$85,$4A,$E7,$5A
	.DB $E6,$39,$A4,$31,$46,$4A,$4A,$6B,$6B,$73,$8B,$7B,$47,$52,$89,$52
	.DB $89,$52,$47,$4A,$46,$4A,$66,$4A,$65,$4A,$65,$4A,$E6,$5A,$6C,$8C
	.DB $C5,$31,$E5,$31,$46,$42,$A7,$52,$6A,$73,$4A,$6B,$29,$6B,$6B,$73
	.DB $4B,$6B,$E9,$5A,$A7,$52,$A7,$5A,$A6,$5A,$A6,$5A,$07,$63,$89,$6B
	.DB $C4,$31,$65,$42,$86,$4A,$04,$3A,$E7,$5A,$24,$42,$C7,$5A,$8A,$73
	.DB $8A,$73,$28,$63,$C7,$5A,$E7,$62,$E7,$5A,$A7,$5A,$C7,$5A,$E8,$5A
	.DB $C7,$4A,$28,$53,$CA,$73,$C6,$4A,$48,$63,$44,$42,$E7,$5A,$47,$6B
	.DB $68,$6B,$27,$63,$C6,$5A,$E7,$62,$C7,$5A,$66,$4A,$25,$42,$46,$4A
	.DB $A6,$4A,$64,$42,$C9,$6B,$07,$53,$07,$5B,$C6,$52,$E6,$52,$06,$5B
	.DB $67,$6B,$26,$63,$C6,$5A,$07,$5B,$E7,$5A,$86,$4A,$25,$42,$25,$42
	.DB $08,$5B,$65,$42,$89,$6B,$48,$63,$69,$63,$8A,$6B,$48,$63,$64,$4A
	.DB $E6,$5A,$E6,$5A,$A5,$52,$07,$5B,$69,$63,$48,$63,$07,$5B,$E7,$52
	.DB $C7,$52,$85,$4A,$A6,$52,$85,$4A,$C7,$52,$A7,$52,$65,$42,$E3,$31
	.DB $A6,$52,$07,$53,$E6,$52,$68,$5B,$C9,$6B,$A8,$63,$47,$5B,$E6,$4A
	.DB $C7,$5A,$28,$63,$65,$4A,$85,$4A,$E7,$5A,$25,$42,$E4,$39,$04,$3A
	.DB $08,$5B,$8A,$6B,$89,$63,$C9,$63,$C9,$6B,$46,$53,$A4,$3A,$E5,$4A
	.DB $06,$5B,$85,$4A,$E7,$5A,$66,$42,$66,$42,$67,$42,$29,$5B,$28,$63
	.DB $CA,$7B,$68,$6B,$A5,$52,$84,$52,$43,$4A,$83,$4A,$A8,$73,$47,$6B
	.DB $C6,$52,$A6,$52,$28,$63,$E7,$52,$86,$4A,$25,$3A,$66,$42,$A6,$52
	.DB $A9,$73,$27,$63,$24,$42,$44,$4A,$85,$4A,$A5,$52,$27,$63,$C9,$7B
	.DB $65,$4A,$A6,$52,$28,$63,$89,$6B,$28,$5B,$E7,$52,$A6,$52,$49,$63
	.DB $49,$63,$07,$5B,$28,$5B,$69,$63,$C6,$52,$44,$42,$07,$5B,$07,$63
	.DB $25,$4A,$A7,$5A,$E7,$5A,$AA,$73,$68,$63,$88,$6B,$48,$63,$08,$53
	.DB $66,$42,$E4,$31,$66,$42,$07,$53,$65,$4A,$04,$3A,$86,$4A,$69,$6B
	.DB $87,$52,$E8,$5A,$A6,$52,$28,$63,$C6,$52,$E6,$52,$A5,$4A,$A6,$42
	.DB $A7,$4A,$04,$32,$A2,$29,$65,$42,$AA,$6B,$CB,$73,$49,$63,$07,$5B
	.DB $A7,$5A,$C7,$5A,$66,$4A,$86,$4A,$24,$42,$03,$3A,$C2,$29,$03,$32
	.DB $45,$3A,$44,$3A,$44,$3A,$C6,$4A,$89,$63,$A9,$6B,$48,$5B,$64,$42
	.DB $A6,$52,$65,$4A,$66,$4A,$45,$42,$87,$4A,$26,$3A,$05,$32,$A6,$4A
	.DB $A5,$4A,$68,$63,$0A,$74,$A9,$6B,$06,$53,$26,$53,$67,$63,$C9,$6B
	.DB $E7,$52,$44,$42,$66,$42,$46,$42,$2A,$5B,$E9,$52,$A8,$4A,$A6,$4A
	.DB $27,$5B,$C9,$6B,$88,$63,$A4,$4A,$43,$3A,$C4,$4A,$25,$53,$09,$74
	.DB $28,$63,$25,$42,$63,$21,$E2,$10,$65,$21,$C8,$29,$46,$19,$84,$29
	.DB $E8,$52,$E9,$52,$E4,$29,$E4,$29,$C7,$42,$28,$53,$E7,$4A,$69,$63
	.DB $A7,$52,$25,$42,$E5,$31,$84,$29,$24,$19,$24,$19,$23,$19,$05,$3A
	.DB $49,$5B,$6A,$63,$C7,$4A,$E7,$4A,$AA,$63,$EA,$6B,$89,$63,$28,$5B
	.DB $87,$52,$25,$42,$46,$42,$46,$42,$E5,$39,$26,$3A,$E8,$52,$49,$63
	.DB $2B,$7C,$4C,$7C,$C9,$6B,$C9,$63,$EA,$6B,$88,$5B,$27,$53,$06,$5B
	.DB $66,$4A,$25,$42,$25,$42,$A6,$4A,$86,$4A,$C7,$52,$AA,$73,$67,$63
	.DB $09,$7C,$2A,$7C,$E9,$6B,$09,$6C,$C8,$6B,$67,$5B,$26,$53,$A9,$6B
	.DB $C8,$5A,$A7,$52,$66,$4A,$E7,$5A,$E7,$5A,$A5,$52,$C5,$52,$46,$63
	.DB $A8,$73,$C8,$6B,$87,$63,$A7,$63,$67,$5B,$26,$53,$06,$53,$43,$42
	.DB $48,$63,$07,$5B,$86,$4A,$C6,$52,$E7,$5A,$C6,$52,$64,$4A,$27,$63
	.DB $88,$73,$88,$6B,$26,$5B,$06,$53,$E6,$52,$E6,$52,$27,$5B,$07,$5B
	.DB $C5,$4A,$C5,$4A,$65,$42,$64,$42,$A6,$52,$C7,$52,$86,$52,$24,$4A
	.DB $C7,$5A,$E7,$5A,$85,$4A,$65,$42,$C6,$4A,$28,$5B,$CA,$73,$08,$5B
	.DB $A4,$42,$27,$53,$89,$63,$28,$5B,$08,$5B,$29,$63,$C8,$5A,$25,$4A
	.DB $E8,$62,$E8,$5A,$04,$42,$C3,$31,$04,$3A,$A6,$4A,$28,$63,$C7,$52
	.DB $E9,$7B,$26,$63,$84,$4A,$E6,$5A,$C5,$52,$85,$52,$68,$63,$07,$53
	.DB $28,$53,$48,$53,$A5,$42,$E6,$4A,$64,$42,$27,$5B,$EB,$7B,$AA,$73
	.DB $2B,$84,$A9,$73,$C6,$5A,$C6,$5A,$07,$63,$E7,$5A,$A6,$52,$C7,$52
	.DB $66,$4A,$65,$4A,$C6,$52,$A9,$6B,$27,$63,$85,$4A,$44,$42,$62,$29
	.DB $28,$63,$E7,$5A,$65,$4A,$86,$52,$08,$63,$49,$73,$09,$6B,$2A,$6B
	.DB $2A,$73,$49,$6B,$EB,$83,$0B,$84,$48,$6B,$44,$4A,$65,$4A,$82,$29
	.DB $49,$6B,$86,$52,$E4,$41,$04,$42,$25,$4A,$87,$52,$4A,$6B,$A8,$5A
	.DB $E9,$62,$E8,$62,$8A,$73,$E7,$5A,$A5,$52,$85,$4A,$AA,$73,$E7,$5A
	.DB $AA,$73,$45,$4A,$E0,$18,$21,$21,$62,$29,$A4,$39,$09,$63,$09,$5B
	.DB $E8,$52,$44,$3A,$C5,$4A,$64,$42,$64,$42,$44,$42,$48,$63,$C7,$52
	.DB $89,$6B,$65,$4A,$E0,$18,$81,$29,$65,$4A,$C7,$52,$69,$63,$69,$63
	.DB $48,$53,$C6,$4A,$06,$4B,$05,$53,$05,$53,$C4,$4A,$07,$5B,$07,$5B
	.DB $88,$63,$27,$5B,$E6,$52,$07,$5B,$A9,$6B,$A9,$6B,$07,$53,$65,$42
	.DB $C6,$52,$06,$53,$06,$5B,$C8,$6B,$C8,$6B,$E8,$73,$2A,$7C,$CA,$73
	.DB $01,$32,$83,$3A,$0A,$74,$E9,$6B,$A9,$63,$88,$63,$23,$3A,$28,$6B
	.DB $08,$6B,$47,$6B,$E5,$62,$09,$84,$29,$84,$6A,$84,$2A,$7C,$A8,$6B
	.DB $09,$5B,$C8,$4A,$09,$53,$26,$3A,$87,$42,$E9,$4A,$AD,$6B,$6A,$52
	.DB $E9,$41,$8B,$52,$EC,$5A,$72,$84,$BB,$C6,$5F,$15,$5F,$15,$5F,$15
	.DB $29,$5B,$4A,$5B,$2D,$7C,$6A,$63,$E8,$52,$2A,$5B,$B0,$8C,$EB,$5A
	.DB $C7,$39,$A7,$31,$49,$42,$50,$7C,$BA,$CE,$5F,$15,$BA,$C6,$5F,$15
	.DB $25,$3A,$82,$29,$04,$32,$66,$42,$4A,$63,$AB,$6B,$0D,$74,$2A,$5B
	.DB $47,$42,$27,$3A,$27,$3A,$4C,$63,$74,$A5,$99,$CE,$FB,$D6,$9A,$C6
	.DB $8B,$6B,$AB,$6B,$8A,$63,$29,$5B,$49,$63,$49,$63,$49,$5B,$29,$53
	.DB $E8,$4A,$C8,$4A,$46,$42,$06,$3A,$C9,$5A,$AD,$73,$2F,$84,$71,$8C
	.DB $66,$4A,$C3,$31,$25,$42,$28,$5B,$49,$63,$69,$6B,$8A,$6B,$C7,$4A
	.DB $87,$42,$87,$42,$A7,$4A,$87,$52,$67,$52,$47,$4A,$A5,$39,$47,$4A
	.DB $E8,$5A,$4A,$6B,$29,$63,$29,$63,$A7,$52,$86,$52,$86,$4A,$E8,$5A
	.DB $A7,$4A,$66,$4A,$E9,$5A,$09,$5B,$C8,$5A,$C8,$5A,$06,$42,$26,$42
	.DB $A8,$52,$E9,$5A,$26,$42,$C8,$5A,$4A,$6B,$6A,$6B,$87,$52,$09,$63
	.DB $C9,$62,$67,$4A,$68,$52,$E5,$39,$83,$29,$46,$42,$26,$3A,$87,$42
	.DB $06,$42,$C9,$5A,$87,$52,$A8,$52,$C4,$39,$E4,$41,$C4,$39,$C5,$41
	.DB $E6,$49,$C5,$41,$27,$4A,$E5,$39,$63,$21,$46,$3A,$46,$3A,$E4,$29
	.DB $89,$52,$48,$4A,$64,$29,$65,$29,$45,$29,$04,$21,$66,$29,$46,$21
	.DB $E9,$39,$28,$3A,$E6,$31,$E5,$31,$64,$21,$23,$21,$A5,$31,$24,$21
	.DB $E9,$52,$A8,$4A,$47,$42,$C5,$29,$64,$21,$65,$21,$65,$21,$66,$29
	.DB $08,$3A,$69,$4A,$C9,$52,$09,$53,$C8,$4A,$67,$42,$68,$42,$04,$19
	.DB $A3,$29,$04,$32,$C7,$4A,$66,$3A,$67,$3A,$E9,$4A,$EA,$52,$2F,$7C
	.DB $6C,$6B,$2B,$63,$4B,$63,$6B,$63,$2A,$5B,$88,$42,$85,$29,$24,$19
	.DB $C6,$4A,$68,$63,$2C,$7C,$EB,$73,$6A,$63,$8B,$6B,$AC,$6B,$AB,$73
	.DB $E8,$5A,$E8,$5A,$2A,$63,$4A,$63,$6B,$63,$4B,$5B,$88,$42,$07,$32
	.DB $6C,$8C,$EE,$9C,$EE,$9C,$AE,$94,$AA,$73,$E8,$5A,$09,$63,$8A,$73
	.DB $69,$6B,$AA,$73,$6A,$6B,$67,$4A,$25,$3A,$26,$3A,$C4,$29,$84,$21
	.DB $4B,$8C,$8C,$94,$4C,$8C,$4C,$8C,$49,$73,$45,$52,$66,$52,$48,$6B
	.DB $E7,$5A,$C6,$52,$86,$4A,$26,$42,$46,$42,$66,$42,$46,$3A,$86,$42
	.DB $29,$84,$2A,$84,$E9,$7B,$89,$73,$E7,$62,$45,$4A,$04,$42,$E6,$5A
	.DB $C6,$5A,$65,$4A,$46,$42,$A7,$52,$A8,$52,$A7,$4A,$A7,$4A,$68,$63
	.DB $87,$6B,$66,$63,$67,$63,$E6,$52,$C6,$52,$A6,$52,$E4,$39,$40,$21
	.DB $85,$52,$69,$6B,$8B,$6B,$6B,$6B,$A8,$52,$66,$4A,$E8,$5A,$27,$5B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $38,$AE,$79,$B6,$BA,$BE,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $8D,$63,$70,$7C,$38,$B6,$5F,$15,$5F,$15,$9B,$BE,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $C7,$31,$49,$3A,$6E,$63,$D3,$8C,$B7,$AD,$7A,$BE,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $28,$3A,$C7,$31,$6A,$42,$AB,$4A,$EC,$52,$52,$7C,$56,$9D,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $C9,$4A,$89,$42,$AE,$6B,$6D,$63,$C7,$29,$28,$32,$0C,$53,$F5,$9C
	.DB $78,$AD,$3A,$BE,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $4A,$5B,$67,$42,$EA,$52,$2A,$53,$88,$42,$EA,$4A,$CB,$52,$8B,$52
	.DB $CE,$62,$F2,$7B,$37,$9D,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $89,$52,$A6,$31,$07,$42,$48,$4A,$E6,$39,$EA,$5A,$EF,$73,$8E,$53
	.DB $72,$64,$ED,$3A,$B0,$5B,$73,$7C,$98,$A5,$5A,$BE,$7A,$C6,$1C,$DF
	.DB $89,$52,$28,$42,$27,$42,$07,$3A,$C6,$31,$68,$4A,$0B,$5B,$F3,$8C
	.DB $CF,$5B,$6E,$53,$F1,$63,$94,$7C,$90,$63,$12,$74,$32,$7C,$39,$BE
	.DB $47,$42,$A9,$52,$68,$4A,$E7,$39,$07,$3A,$28,$42,$0B,$5B,$2F,$7C
	.DB $8D,$63,$10,$74,$8F,$63,$D0,$63,$B0,$63,$F1,$6B,$B0,$63,$B3,$84
	.DB $27,$42,$E9,$5A,$48,$4A,$64,$29,$24,$21,$03,$21,$E6,$39,$C5,$39
	.DB $0A,$5B,$0E,$7C,$AE,$6B,$8F,$63,$16,$95,$33,$74,$94,$7C,$72,$7C
	.DB $06,$3A,$C9,$52,$88,$4A,$A5,$31,$64,$29,$44,$29,$27,$4A,$42,$29
	.DB $C5,$41,$EA,$5A,$CE,$73,$B3,$8C,$D4,$8C,$ED,$4A,$B4,$84,$52,$74
	.DB $43,$21,$C5,$39,$26,$42,$A5,$31,$A5,$31,$07,$42,$68,$4A,$C5,$39
	.DB $02,$21,$C6,$39,$28,$42,$72,$84,$AF,$6B,$AB,$42,$93,$84,$B6,$A5
	.DB $A8,$52,$C9,$52,$EA,$5A,$88,$4A,$A9,$52,$0A,$63,$07,$42,$A5,$31
	.DB $03,$21,$A6,$31,$45,$21,$6A,$42,$EB,$52,$8E,$63,$91,$84,$70,$84
	.DB $47,$4A,$26,$42,$47,$4A,$06,$42,$88,$4A,$EA,$5A,$E3,$18,$24,$21
	.DB $04,$19,$C7,$31,$49,$42,$04,$19,$28,$3A,$2B,$5B,$0E,$74,$4E,$84
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $F0,$73,$92,$8C,$FB,$DE,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $F0,$6B,$AF,$6B,$99,$CE,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $31,$74,$2C,$5B,$D6,$B5,$37,$CE,$9D,$FF,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $30,$7C,$68,$42,$6F,$8C,$4F,$8C,$74,$BD,$74,$BD,$37,$D6,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $2E,$7C,$11,$95,$F5,$BD,$8F,$94,$0A,$63,$67,$52,$6C,$73,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $CB,$73,$4D,$84,$08,$5B,$6A,$6B,$AC,$73,$E9,$5A,$A5,$39,$47,$4A
	.DB $4B,$6B,$AE,$73,$4D,$63,$B4,$8C,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$1D,$CF
	.DB $DB,$CE,$99,$CE,$56,$CE,$B2,$BD,$8D,$A4,$6C,$A4,$4F,$C5,$E9,$8B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$BA,$C6
	.DB $58,$BE,$57,$C6,$56,$CE,$F3,$CD,$6D,$A4,$2B,$9C,$CD,$B4,$6C,$A4
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$DA,$CE
	.DB $37,$C6,$56,$CE,$35,$D6,$13,$D6,$0A,$9C,$68,$83,$A9,$8B,$2B,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$78,$C6
	.DB $15,$C6,$34,$CE,$B2,$C5,$D1,$CD,$2B,$9C,$88,$8B,$47,$83,$A9,$83
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$9A,$C6,$93,$AD
	.DB $71,$B5,$70,$B5,$EE,$AC,$4F,$BD,$AC,$AC,$4B,$A4,$0A,$94,$69,$7B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$1D,$CF,$5F,$15,$58,$BE,$72,$AD
	.DB $70,$B5,$ED,$A4,$8C,$A4,$AC,$AC,$8C,$AC,$0A,$94,$EA,$93,$07,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$1C,$CF,$1C,$CF,$99,$C6,$F6,$B5,$91,$AD
	.DB $6F,$B5,$8C,$9C,$0E,$AD,$6B,$A4,$4B,$9C,$47,$7B,$89,$83,$48,$7B
	.DB $5F,$15,$5F,$15,$5F,$15,$FB,$CE,$FA,$CE,$78,$BE,$B3,$AD,$0F,$9D
	.DB $CD,$9C,$0A,$8C,$6F,$BD,$CD,$AC,$6C,$A4,$47,$7B,$0B,$94,$6D,$9C
	.DB $5F,$15,$3F,$CF,$5F,$15,$79,$C6,$97,$D6,$D3,$C5,$8D,$A4,$2C,$94
	.DB $AA,$83,$AD,$9C,$70,$B5,$2B,$8C,$AC,$9C,$0A,$8C,$67,$73,$09,$8C
	.DB $5F,$15,$5F,$15,$5F,$15,$16,$BE,$B3,$BD,$30,$B5,$CE,$AC,$71,$BD
	.DB $AE,$A4,$D2,$C5,$0F,$AD,$CE,$A4,$0A,$8C,$2A,$8C,$67,$73,$27,$73
	.DB $5F,$15,$5F,$15,$79,$C6,$53,$AD,$0D,$8C,$0B,$94,$8D,$A4,$0F,$AD
	.DB $2D,$94,$0C,$8C,$2C,$94,$6D,$94,$8D,$9C,$89,$7B,$07,$6B,$48,$73
	.DB $5F,$15,$BA,$CE,$36,$C6,$AF,$9C,$49,$73,$89,$83,$4C,$9C,$E8,$6A
	.DB $E8,$6A,$E8,$6A,$0C,$8C,$2C,$8C,$2C,$94,$E7,$62,$89,$7B,$89,$7B
	.DB $1B,$D7,$77,$CE,$B3,$BD,$2D,$94,$49,$7B,$2B,$94,$8D,$A4,$49,$7B
	.DB $C7,$62,$CB,$83,$4D,$94,$B2,$C5,$0C,$8C,$E7,$62,$28,$73,$E7,$6A
	.DB $B9,$D6,$B4,$BD,$F0,$AC,$AA,$83,$89,$7B,$CE,$A4,$4C,$94,$EB,$83
	.DB $CA,$83,$8E,$9C,$30,$AD,$71,$BD,$EB,$8B,$86,$5A,$E8,$6A,$E8,$6A
	.DB $D4,$BD,$CF,$A4,$AE,$A4,$CA,$83,$CB,$83,$30,$AD,$CB,$83,$48,$73
	.DB $8D,$9C,$30,$B5,$71,$B5,$8A,$7B,$69,$73,$A6,$62,$EB,$8B,$CB,$83
	.DB $D0,$9C,$0C,$8C,$EF,$A4,$4C,$94,$8E,$94,$D3,$B5,$EC,$7B,$8D,$9C
	.DB $4C,$94,$8D,$9C,$89,$7B,$2C,$94,$8E,$9C,$2C,$94,$69,$73,$0C,$94
	.DB $47,$7B,$E6,$6A,$C6,$6A,$68,$7B,$85,$5A,$C6,$6A,$28,$73,$44,$4A
	.DB $07,$63,$28,$63,$E7,$5A,$E7,$5A,$E7,$5A,$86,$4A,$24,$42,$25,$42
	.DB $4B,$9C,$69,$7B,$C6,$62,$07,$6B,$E7,$6A,$28,$73,$28,$73,$A6,$5A
	.DB $C6,$52,$27,$63,$68,$6B,$07,$5B,$85,$4A,$65,$4A,$86,$52,$A7,$52
	.DB $4C,$9C,$AA,$7B,$49,$73,$49,$73,$8A,$7B,$E7,$6A,$C7,$62,$48,$6B
	.DB $E7,$62,$48,$6B,$69,$6B,$85,$52,$04,$42,$65,$4A,$86,$52,$49,$6B
	.DB $89,$7B,$48,$73,$EB,$83,$CB,$83,$EB,$83,$E7,$62,$08,$6B,$89,$73
	.DB $27,$6B,$69,$73,$27,$63,$24,$42,$65,$4A,$E7,$5A,$86,$52,$45,$4A
	.DB $48,$73,$49,$6B,$CB,$7B,$EB,$83,$8E,$94,$69,$73,$CA,$83,$69,$73
	.DB $E6,$62,$27,$6B,$28,$6B,$C6,$5A,$E7,$62,$28,$6B,$86,$52,$66,$52
	.DB $69,$7B,$EB,$83,$AA,$7B,$4C,$94,$50,$B5,$0B,$8C,$89,$7B,$CA,$83
	.DB $A5,$5A,$A5,$5A,$89,$7B,$68,$73,$C6,$5A,$86,$52,$45,$4A,$83,$39
	.DB $8A,$83,$8D,$9C,$CB,$83,$CE,$A4,$91,$BD,$2B,$94,$47,$7B,$6C,$9C
	.DB $27,$73,$C6,$62,$68,$73,$89,$7B,$C7,$62,$65,$52,$A7,$5A,$86,$52
	.DB $CB,$8B,$8E,$9C,$EB,$8B,$EF,$AC,$4F,$BD,$2B,$9C,$0A,$9C,$CD,$A4
	.DB $0A,$8C,$07,$6B,$E7,$6A,$68,$73,$69,$73,$69,$73,$AB,$7B,$49,$6B
	.DB $2A,$94,$4B,$94,$6C,$9C,$A9,$8B,$0B,$94,$89,$8B,$0B,$9C,$C9,$8B
	.DB $6C,$9C,$68,$7B,$07,$6B,$07,$6B,$E7,$62,$29,$6B,$C8,$5A,$E5,$39
	.DB $47,$73,$47,$7B,$88,$83,$27,$7B,$EB,$93,$CA,$93,$6D,$A4,$4F,$BD
	.DB $AC,$A4,$67,$7B,$06,$6B,$C5,$62,$64,$52,$A5,$5A,$07,$63,$08,$63
	.DB $68,$7B,$A9,$83,$AA,$8B,$89,$83,$2C,$9C,$2C,$9C,$AE,$AC,$8B,$A4
	.DB $4A,$9C,$E8,$8B,$26,$73,$46,$73,$A8,$7B,$46,$6B,$C5,$5A,$C5,$5A
	.DB $EB,$8B,$AE,$A4,$AE,$A4,$6D,$A4,$8E,$AC,$4D,$A4,$AE,$AC,$4B,$A4
	.DB $6B,$A4,$0D,$B5,$4A,$9C,$2A,$94,$AC,$A4,$87,$7B,$63,$5A,$A4,$5A
	.DB $8A,$83,$CF,$AC,$CF,$AC,$8E,$A4,$8E,$A4,$4D,$A4,$8E,$AC,$4C,$A4
	.DB $69,$83,$0B,$94,$0B,$94,$CA,$8B,$A9,$83,$E6,$6A,$E6,$6A,$C6,$62
	.DB $8A,$83,$CF,$AC,$6D,$A4,$4D,$9C,$4C,$A4,$6D,$A4,$AE,$AC,$AA,$8B
	.DB $28,$73,$AA,$83,$AA,$83,$07,$73,$E7,$6A,$E7,$6A,$C7,$6A,$44,$52
	.DB $0C,$94,$0F,$AD,$4D,$9C,$6D,$A4,$6D,$A4,$AD,$AC,$CE,$AC,$6D,$9C
	.DB $AE,$A4,$30,$B5,$0F,$AD,$A9,$83,$E7,$6A,$27,$73,$64,$5A,$44,$52
	.DB $2C,$94,$0F,$B5,$4C,$9C,$8D,$A4,$8D,$AC,$CD,$B4,$CD,$AC,$AD,$A4
	.DB $8D,$9C,$2F,$AD,$12,$CE,$6C,$94,$84,$5A,$06,$6B,$06,$6B,$C5,$62
	.DB $E8,$62,$4A,$73,$11,$AD,$B4,$BD,$ED,$8B,$A8,$62,$B0,$A4,$6B,$73
	.DB $A4,$31,$6C,$73,$74,$B5,$48,$4A,$45,$31,$E4,$28,$28,$4A,$0F,$8C
	.DB $09,$63,$8B,$73,$F1,$A4,$73,$BD,$0A,$73,$47,$5A,$ED,$8B,$05,$42
	.DB $02,$21,$8C,$73,$75,$B5,$08,$42,$04,$21,$C3,$20,$07,$4A,$71,$9C
	.DB $09,$63,$4A,$6B,$D0,$A4,$94,$BD,$E9,$6A,$C9,$6A,$8C,$83,$48,$52
	.DB $85,$31,$10,$84,$76,$B5,$E8,$41,$25,$29,$25,$29,$48,$52,$CA,$62
	.DB $C8,$5A,$8B,$73,$52,$B5,$90,$9C,$67,$5A,$4B,$7B,$D2,$AC,$6C,$73
	.DB $28,$4A,$10,$8C,$D4,$A4,$87,$39,$04,$21,$E4,$28,$65,$31,$47,$52
	.DB $05,$42,$25,$4A,$B0,$A4,$CD,$83,$4B,$7B,$8C,$83,$50,$9C,$30,$94
	.DB $AB,$62,$31,$8C,$51,$94,$46,$31,$E4,$20,$03,$21,$23,$29,$A8,$52
	.DB $26,$4A,$AC,$7B,$53,$B5,$E6,$49,$06,$4A,$2B,$73,$54,$BD,$F2,$AC
	.DB $0B,$6B,$0F,$8C,$8E,$7B,$E3,$20,$C2,$18,$A5,$39,$47,$4A,$AB,$73
	.DB $46,$4A,$2E,$8C,$12,$AD,$E6,$49,$A9,$62,$AD,$83,$54,$B5,$53,$B5
	.DB $E9,$62,$8C,$7B,$2B,$6B,$23,$29,$43,$29,$67,$4A,$C8,$5A,$28,$63
	.DB $05,$42,$2E,$8C,$90,$9C,$68,$5A,$AD,$83,$50,$9C,$53,$B5,$B4,$BD
	.DB $09,$63,$0D,$84,$4F,$8C,$E9,$62,$E9,$62,$09,$63,$86,$52,$24,$42
	.DB $E6,$41,$70,$94,$B5,$BD,$6D,$7B,$13,$B5,$D2,$AC,$B1,$AC,$73,$C5
	.DB $29,$7B,$CF,$AC,$0C,$8C,$4A,$73,$66,$52,$66,$4A,$E4,$39,$08,$5B
	.DB $45,$4A,$CF,$9C,$93,$BD,$2D,$94,$6E,$9C,$D0,$AC,$72,$BD,$CF,$AC
	.DB $8A,$83,$AF,$A4,$CB,$83,$45,$52,$04,$42,$86,$52,$49,$6B,$86,$4A
	.DB $C5,$5A,$2F,$AD,$0E,$AD,$B1,$C5,$B1,$C5,$B1,$C5,$EE,$AC,$8D,$A4
	.DB $8A,$83,$CB,$83,$49,$73,$A6,$5A,$C7,$62,$86,$52,$08,$63,$08,$5B
	.DB $E5,$62,$0A,$8C,$06,$6B,$2B,$94,$CD,$AC,$EE,$AC,$2B,$94,$AD,$9C
	.DB $89,$7B,$E7,$62,$85,$5A,$48,$6B,$89,$73,$86,$52,$45,$4A,$C7,$5A
	.DB $27,$73,$48,$7B,$C6,$62,$A6,$62,$65,$5A,$65,$5A,$49,$7B,$CA,$7B
	.DB $E7,$62,$A6,$5A,$E2,$41,$28,$6B,$E7,$62,$07,$63,$08,$63,$45,$4A
	.DB $28,$73,$08,$6B,$08,$6B,$A7,$62,$87,$62,$E4,$49,$46,$52,$45,$4A
	.DB $C3,$39,$45,$4A,$E3,$41,$48,$6B,$C6,$5A,$69,$73,$89,$73,$69,$73
	.DB $E7,$6A,$45,$52,$66,$5A,$E8,$62,$6B,$7B,$C8,$62,$25,$4A,$C4,$39
	.DB $83,$31,$04,$42,$86,$52,$69,$73,$E7,$62,$48,$73,$69,$73,$EF,$A4
	.DB $E7,$6A,$28,$6B,$4D,$94,$CF,$9C,$AB,$7B,$A7,$5A,$26,$4A,$46,$4A
	.DB $25,$42,$46,$4A,$08,$63,$A7,$5A,$45,$52,$07,$6B,$CA,$83,$51,$B5
	.DB $6C,$73,$88,$5A,$C8,$62,$67,$5A,$6C,$7B,$EA,$6A,$06,$4A,$84,$39
	.DB $26,$4A,$46,$4A,$A8,$5A,$66,$4A,$66,$4A,$45,$42,$E7,$5A,$49,$63
	.DB $2F,$8C,$88,$5A,$05,$4A,$6F,$94,$8C,$7B,$88,$5A,$4B,$73,$46,$52
	.DB $C8,$5A,$A7,$5A,$E8,$62,$86,$52,$86,$52,$45,$42,$E7,$5A,$A6,$52
	.DB $A9,$5A,$26,$4A,$46,$4A,$ED,$83,$C8,$62,$E5,$41,$46,$4A,$46,$4A
	.DB $E8,$5A,$E8,$5A,$49,$6B,$E7,$5A,$A7,$52,$24,$42,$A6,$52,$48,$63
	.DB $09,$63,$49,$6B,$CB,$7B,$29,$6B,$29,$6B,$46,$4A,$05,$42,$E4,$39
	.DB $66,$4A,$66,$4A,$E7,$5A,$A6,$52,$A6,$52,$44,$4A,$C6,$5A,$68,$6B
	.DB $8B,$73,$08,$63,$A7,$5A,$04,$42,$25,$42,$83,$31,$E5,$39,$E4,$31
	.DB $45,$42,$24,$3A,$65,$4A,$45,$42,$86,$52,$85,$52,$48,$6B,$C6,$5A
	.DB $C7,$5A,$08,$5B,$86,$52,$45,$42,$E4,$39,$67,$4A,$E5,$39,$A3,$29
	.DB $66,$42,$86,$42,$E8,$5A,$86,$52,$86,$52,$65,$4A,$E7,$62,$08,$63
	.DB $24,$42,$07,$5B,$A6,$52,$24,$3A,$04,$3A,$C8,$52,$83,$29,$A3,$29
	.DB $87,$42,$C7,$4A,$29,$5B,$86,$52,$45,$4A,$C3,$39,$45,$4A,$29,$6B
	.DB $EA,$7B,$27,$5B,$24,$42,$E3,$39,$87,$4A,$A4,$31,$A4,$31,$25,$32
	.DB $A8,$4A,$87,$42,$87,$4A,$C3,$31,$83,$31,$82,$31,$46,$52,$08,$6B
	.DB $08,$5B,$65,$4A,$04,$3A,$E3,$39,$65,$4A,$82,$31,$E4,$39,$25,$42
	.DB $A8,$52,$46,$42,$26,$42,$83,$29,$21,$21,$65,$4A,$28,$63,$45,$4A
	.DB $86,$52,$A7,$52,$25,$42,$E4,$39,$C4,$39,$04,$42,$A7,$52,$C4,$39
	.DB $05,$3A,$A4,$31,$26,$42,$66,$4A,$66,$4A,$49,$6B,$69,$6B,$69,$6B
	.DB $66,$4A,$46,$4A,$E4,$39,$25,$42,$63,$31,$05,$42,$A4,$31,$C4,$31
	.DB $C4,$39,$C4,$31,$66,$4A,$E8,$5A,$E8,$5A,$08,$63,$E7,$5A,$C6,$5A
	.DB $25,$4A,$04,$42,$25,$42,$87,$52,$88,$52,$C9,$5A,$84,$31,$83,$31
	.DB $05,$42,$87,$52,$E8,$5A,$C7,$52,$65,$4A,$65,$4A,$85,$4A,$07,$63
	.DB $65,$52,$25,$4A,$87,$52,$47,$4A,$0A,$63,$88,$52,$43,$29,$42,$29
	.DB $05,$42,$E8,$62,$08,$63,$C7,$5A,$A6,$52,$C6,$5A,$27,$63,$68,$6B
	.DB $08,$6B,$C7,$5A,$E8,$62,$09,$63,$EE,$83,$47,$4A,$C1,$18,$A4,$39
	.DB $04,$42,$A7,$5A,$C7,$5A,$C7,$5A,$28,$63,$48,$63,$89,$73,$C5,$52
	.DB $8A,$7B,$8A,$7B,$8B,$7B,$B0,$9C,$73,$B5,$E9,$5A,$E5,$41,$C8,$5A
	.DB $86,$52,$C7,$5A,$65,$52,$85,$52,$27,$63,$07,$63,$68,$6B,$06,$5B
	.DB $AA,$83,$8D,$94,$AB,$7B,$D0,$9C,$6F,$94,$E5,$41,$05,$42,$AB,$7B
	.DB $29,$6B,$48,$6B,$65,$52,$44,$4A,$E6,$5A,$C6,$5A,$48,$6B,$06,$5B
	.DB $86,$4A,$86,$4A,$C7,$52,$86,$4A,$86,$4A,$C7,$52,$86,$4A,$66,$4A
	.DB $08,$5B,$C7,$52,$C7,$52,$C7,$5A,$A7,$52,$C7,$5A,$A7,$52,$66,$4A
	.DB $45,$42,$45,$42,$86,$4A,$65,$4A,$45,$42,$66,$42,$65,$42,$86,$4A
	.DB $48,$63,$08,$5B,$28,$63,$28,$63,$C7,$52,$A6,$52,$65,$4A,$86,$52
	.DB $28,$63,$E7,$52,$C6,$52,$C6,$52,$65,$42,$24,$3A,$65,$42,$85,$4A
	.DB $27,$5B,$07,$5B,$27,$63,$48,$63,$E7,$5A,$E7,$5A,$A6,$52,$C6,$5A
	.DB $48,$6B,$C6,$52,$A5,$52,$E7,$52,$65,$4A,$04,$3A,$85,$4A,$E6,$52
	.DB $47,$63,$E6,$52,$E6,$5A,$27,$5B,$C6,$5A,$E7,$5A,$E7,$5A,$C6,$5A
	.DB $C6,$5A,$85,$4A,$A6,$52,$28,$63,$E7,$5A,$65,$42,$C6,$52,$68,$63
	.DB $A9,$6B,$07,$5B,$07,$5B,$27,$5B,$A5,$52,$A6,$52,$A6,$52,$A5,$52
	.DB $E7,$62,$C7,$5A,$07,$63,$28,$63,$C6,$52,$44,$42,$44,$42,$47,$63
	.DB $88,$6B,$27,$5B,$48,$63,$68,$6B,$E7,$5A,$E6,$5A,$C6,$5A,$85,$52
	.DB $C7,$5A,$A7,$5A,$A7,$5A,$45,$4A,$04,$3A,$04,$3A,$04,$3A,$E6,$52
	.DB $27,$5B,$C6,$52,$27,$63,$89,$6B,$28,$63,$48,$6B,$48,$6B,$C6,$5A
	.DB $86,$5A,$87,$5A,$87,$52,$05,$42,$46,$4A,$49,$63,$AB,$73,$07,$5B
	.DB $27,$5B,$A5,$4A,$E7,$5A,$48,$63,$07,$63,$48,$6B,$69,$6B,$E8,$62
	.DB $08,$63,$A7,$5A,$A6,$5A,$45,$4A,$04,$42,$08,$63,$49,$73,$64,$4A
	.DB $E6,$5A,$48,$63,$28,$63,$E7,$5A,$C7,$5A,$08,$5B,$49,$63,$45,$42
	.DB $C6,$5A,$82,$31,$04,$42,$E7,$62,$08,$63,$08,$6B,$45,$4A,$85,$52
	.DB $C6,$5A,$E7,$5A,$E7,$5A,$C6,$52,$86,$52,$C6,$52,$E8,$5A,$45,$42
	.DB $C6,$5A,$85,$52,$85,$52,$65,$4A,$04,$42,$86,$52,$C7,$5A,$E7,$5A
	.DB $C6,$5A,$C6,$5A,$E7,$5A,$C6,$52,$85,$4A,$85,$4A,$C6,$52,$C7,$52
	.DB $E6,$5A,$C6,$5A,$C6,$5A,$A6,$52,$44,$4A,$85,$52,$E7,$5A,$07,$63
	.DB $C6,$5A,$C6,$5A,$E7,$5A,$C6,$5A,$65,$4A,$64,$4A,$A5,$52,$C6,$52
	.DB $E6,$5A,$C6,$5A,$07,$5B,$48,$6B,$48,$6B,$07,$5B,$07,$5B,$C6,$5A
	.DB $A6,$52,$E6,$5A,$07,$5B,$C6,$52,$44,$4A,$44,$42,$65,$4A,$44,$42
	.DB $E6,$5A,$68,$6B,$68,$63,$48,$63,$68,$6B,$27,$63,$48,$6B,$C6,$52
	.DB $E7,$5A,$27,$63,$07,$63,$E6,$5A,$A5,$52,$85,$4A,$65,$4A,$86,$4A
	.DB $06,$5B,$88,$6B,$47,$63,$88,$6B,$C9,$73,$06,$5B,$C6,$52,$07,$5B
	.DB $28,$63,$48,$63,$07,$63,$07,$5B,$07,$5B,$C7,$52,$65,$4A,$C7,$52
	.DB $88,$73,$8C,$8C,$C9,$73,$67,$63,$88,$6B,$E6,$52,$E6,$52,$27,$63
	.DB $48,$6B,$27,$63,$C6,$5A,$E7,$5A,$28,$63,$E7,$5A,$45,$42,$25,$42
	.DB $05,$42,$25,$42,$C8,$5A,$66,$4A,$A7,$52,$EC,$73,$E9,$5A,$E5,$41
	.DB $47,$4A,$47,$4A,$88,$52,$E4,$39,$25,$3A,$65,$3A,$28,$53,$27,$53
	.DB $E8,$5A,$29,$63,$08,$63,$46,$42,$86,$4A,$E8,$52,$C4,$31,$83,$31
	.DB $C5,$39,$E4,$39,$66,$4A,$45,$42,$E7,$52,$27,$5B,$A9,$63,$88,$63
	.DB $E7,$5A,$28,$63,$86,$52,$04,$3A,$86,$4A,$E7,$52,$C7,$52,$45,$42
	.DB $46,$4A,$E4,$39,$45,$42,$85,$4A,$89,$6B,$A9,$6B,$A8,$6B,$26,$5B
	.DB $C6,$5A,$AA,$73,$48,$63,$E7,$52,$C6,$52,$03,$3A,$24,$42,$A7,$52
	.DB $A7,$52,$25,$42,$86,$4A,$C7,$52,$AA,$73,$27,$63,$64,$4A,$84,$4A
	.DB $85,$52,$C6,$5A,$23,$42,$64,$42,$89,$6B,$89,$6B,$CA,$73,$04,$32
	.DB $26,$3A,$05,$3A,$C8,$52,$09,$5B,$8B,$6B,$45,$42,$21,$21,$45,$4A
	.DB $07,$63,$69,$6B,$A6,$52,$85,$4A,$AA,$6B,$48,$63,$03,$32,$46,$3A
	.DB $46,$3A,$E5,$31,$66,$42,$87,$42,$E9,$5A,$05,$3A,$42,$29,$E5,$39
	.DB $A6,$52,$86,$52,$65,$4A,$44,$42,$28,$5B,$CB,$73,$A6,$4A,$48,$53
	.DB $07,$4B,$44,$32,$45,$3A,$04,$32,$86,$42,$45,$3A,$46,$42,$26,$42
	.DB $24,$42,$24,$42,$69,$6B,$C7,$52,$45,$42,$28,$5B,$65,$42,$47,$53
	.DB $26,$4B,$A4,$3A,$C5,$42,$64,$3A,$85,$42,$65,$3A,$86,$42,$E8,$52
	.DB $C4,$31,$67,$4A,$A3,$31,$06,$42,$67,$4A,$A8,$52,$2A,$63,$0B,$74
	.DB $27,$5B,$03,$32,$CA,$6B,$27,$5B,$27,$5B,$85,$42,$85,$42,$02,$32
	.DB $25,$42,$86,$4A,$A3,$31,$25,$42,$A7,$52,$4A,$63,$6A,$6B,$28,$5B
	.DB $85,$42,$E2,$31,$CA,$6B,$CA,$6B,$89,$63,$C6,$4A,$64,$42,$64,$3A
	.DB $24,$42,$04,$3A,$65,$42,$86,$4A,$E3,$31,$86,$4A,$48,$63,$44,$42
	.DB $85,$42,$27,$5B,$2B,$7C,$89,$63,$64,$42,$44,$3A,$64,$42,$A5,$4A
	.DB $A6,$52,$44,$42,$07,$5B,$E7,$5A,$E2,$31,$65,$4A,$A5,$4A,$C5,$4A
	.DB $44,$3A,$85,$42,$27,$53,$27,$5B,$E2,$31,$E2,$31,$60,$19,$27,$5B
	.DB $CA,$73,$68,$63,$C6,$52,$85,$4A,$07,$5B,$69,$6B,$24,$42,$68,$63
	.DB $27,$5B,$88,$63,$68,$63,$68,$63,$64,$42,$68,$63,$68,$63,$CA,$73
	.DB $8A,$6B,$28,$5B,$86,$4A,$24,$3A,$C7,$52,$E7,$5A,$44,$42,$44,$42
	.DB $07,$5B,$0A,$74,$88,$6B,$A5,$4A,$A1,$29,$88,$6B,$8D,$8C,$A9,$6B
	.DB $45,$42,$25,$42,$87,$4A,$05,$3A,$E4,$39,$83,$29,$66,$4A,$E6,$52
	.DB $27,$5B,$88,$6B,$A9,$6B,$27,$5B,$84,$4A,$68,$63,$88,$6B,$68,$63
	.DB $A3,$31,$05,$3A,$87,$4A,$05,$3A,$26,$42,$83,$29,$87,$4A,$85,$4A
	.DB $06,$5B,$88,$6B,$E9,$73,$27,$5B,$64,$42,$C5,$4A,$A4,$4A,$C9,$6B
	.DB $07,$4B,$48,$53,$C6,$42,$27,$53,$E6,$4A,$69,$5B,$89,$63,$48,$63
	.DB $06,$5B,$27,$5B,$C5,$52,$26,$5B,$87,$63,$67,$63,$4A,$84,$C8,$6B
	.DB $88,$63,$E9,$6B,$06,$53,$88,$63,$06,$53,$C9,$73,$A9,$6B,$88,$63
	.DB $A9,$6B,$C9,$73,$06,$5B,$26,$5B,$88,$63,$05,$53,$26,$5B,$C8,$6B
	.DB $87,$6B,$4A,$84,$67,$6B,$87,$6B,$E5,$5A,$26,$63,$C5,$5A,$47,$63
	.DB $CA,$73,$C9,$73,$C5,$52,$C5,$52,$47,$63,$84,$4A,$02,$3A,$C5,$4A
	.DB $47,$6B,$88,$73,$07,$63,$84,$52,$84,$5A,$85,$5A,$64,$52,$68,$6B
	.DB $EA,$73,$A9,$6B,$E6,$52,$07,$53,$89,$6B,$27,$5B,$A5,$4A,$47,$5B
	.DB $E8,$62,$66,$52,$66,$52,$21,$29,$05,$4A,$05,$4A,$87,$5A,$A5,$4A
	.DB $E6,$52,$84,$42,$44,$3A,$A5,$4A,$C6,$4A,$85,$42,$44,$3A,$27,$5B
	.DB $88,$52,$E6,$41,$27,$4A,$A1,$18,$44,$29,$E3,$20,$A4,$31,$E7,$52
	.DB $26,$5B,$C5,$4A,$27,$5B,$68,$63,$06,$53,$C6,$4A,$85,$42,$A5,$4A
	.DB $47,$42,$06,$3A,$07,$3A,$24,$19,$E3,$18,$C3,$10,$43,$21,$E6,$4A
	.DB $67,$5B,$05,$53,$67,$5B,$A8,$63,$06,$53,$C5,$4A,$64,$3A,$88,$63
	.DB $67,$42,$67,$3A,$A5,$29,$A6,$29,$44,$19,$28,$32,$E9,$4A,$47,$53
	.DB $E8,$6B,$67,$5B,$A8,$63,$E9,$6B,$68,$5B,$47,$5B,$C5,$4A,$06,$53
	.DB $44,$3A,$06,$53,$48,$5B,$27,$53,$27,$5B,$48,$5B,$27,$5B,$C6,$52
	.DB $C6,$52,$C6,$52,$C7,$52,$08,$5B,$86,$4A,$04,$3A,$C3,$31,$65,$42
	.DB $27,$53,$CA,$6B,$68,$5B,$84,$42,$64,$3A,$C5,$4A,$A5,$4A,$A6,$4A
	.DB $44,$42,$24,$42,$A6,$52,$C7,$52,$45,$42,$45,$42,$28,$5B,$A6,$4A
	.DB $48,$5B,$A9,$6B,$27,$5B,$64,$42,$A5,$4A,$07,$53,$C6,$4A,$A5,$4A
	.DB $65,$4A,$65,$4A,$C6,$52,$C6,$52,$24,$42,$86,$4A,$AA,$6B,$66,$4A
	.DB $E6,$52,$C6,$4A,$C6,$4A,$06,$53,$68,$63,$68,$5B,$A5,$4A,$85,$4A
	.DB $07,$5B,$28,$63,$07,$5B,$E7,$52,$A6,$52,$C6,$52,$28,$5B,$86,$4A
	.DB $C5,$4A,$44,$3A,$C6,$4A,$47,$5B,$47,$5B,$C6,$4A,$23,$3A,$84,$42
	.DB $07,$5B,$47,$63,$28,$5B,$28,$5B,$E7,$5A,$C6,$52,$C7,$52,$C7,$52
	.DB $A5,$4A,$84,$42,$27,$5B,$68,$63,$27,$5B,$E6,$52,$C5,$52,$E6,$52
	.DB $E6,$52,$E6,$52,$27,$5B,$27,$5B,$85,$4A,$44,$42,$C7,$52,$E4,$31
	.DB $E6,$52,$E6,$52,$68,$63,$68,$63,$68,$63,$88,$6B,$A9,$6B,$27,$5B
	.DB $27,$5B,$06,$5B,$27,$5B,$07,$5B,$45,$42,$04,$3A,$86,$4A,$EB,$73
	.DB $68,$63,$67,$63,$68,$63,$47,$5B,$47,$63,$68,$63,$47,$63,$E5,$52
	.DB $88,$6B,$88,$6B,$27,$5B,$E7,$5A,$C6,$52,$85,$4A,$86,$4A,$A6,$4A
	.DB $67,$63,$A4,$4A,$06,$5B,$0A,$7C,$CA,$73,$E6,$52,$A2,$29,$C7,$52
	.DB $2A,$5B,$A7,$52,$26,$42,$C8,$52,$69,$63,$49,$63,$28,$63,$C9,$73
	.DB $C9,$73,$E5,$52,$84,$4A,$C5,$52,$A5,$4A,$C6,$52,$A6,$4A,$65,$42
	.DB $25,$3A,$25,$42,$25,$3A,$45,$42,$28,$5B,$CA,$73,$C9,$73,$68,$6B
	.DB $88,$6B,$68,$63,$47,$5B,$47,$63,$48,$63,$CA,$73,$EB,$73,$EB,$73
	.DB $08,$53,$E7,$5A,$08,$5B,$C6,$52,$89,$6B,$6D,$8C,$4C,$84,$A9,$6B
	.DB $A9,$6B,$06,$53,$85,$42,$C6,$4A,$27,$5B,$A9,$6B,$A9,$6B,$CA,$73
	.DB $E7,$52,$65,$42,$85,$4A,$85,$4A,$C6,$52,$47,$63,$68,$63,$A9,$6B
	.DB $89,$63,$C6,$4A,$85,$42,$E6,$4A,$06,$53,$27,$5B,$C5,$4A,$48,$5B
	.DB $68,$63,$E6,$52,$C6,$52,$27,$5B,$C6,$52,$C6,$52,$2B,$7C,$89,$6B
	.DB $48,$5B,$47,$5B,$68,$63,$88,$63,$27,$5B,$06,$53,$C5,$4A,$C6,$4A
	.DB $CA,$6B,$48,$63,$C6,$52,$E6,$52,$03,$3A,$03,$3A,$0B,$7C,$2B,$7C
	.DB $A9,$63,$06,$53,$E6,$4A,$C5,$4A,$63,$3A,$E5,$4A,$47,$5B,$48,$5B
	.DB $CA,$6B,$48,$63,$C6,$52,$C6,$52,$24,$3A,$24,$42,$A9,$6B,$4B,$84
	.DB $27,$53,$C5,$4A,$26,$53,$87,$63,$26,$53,$87,$63,$E9,$73,$48,$5B
	.DB $C7,$4A,$45,$42,$45,$42,$86,$4A,$C7,$52,$27,$5B,$89,$6B,$47,$63
	.DB $A6,$4A,$C6,$52,$85,$4A,$65,$42,$85,$42,$C5,$4A,$C6,$4A,$47,$5B
	.DB $27,$5B,$27,$5B,$84,$4A,$68,$63,$68,$63,$68,$63,$85,$4A,$48,$63
	.DB $24,$3A,$04,$3A,$65,$42,$65,$42,$44,$3A,$85,$42,$27,$5B,$CA,$73
	.DB $CA,$73,$E6,$52,$89,$6B,$EB,$7B,$AA,$6B,$65,$42,$C7,$52,$C7,$52
	.DB $65,$42,$A6,$4A,$E7,$52,$A6,$4A,$85,$42,$27,$5B,$EB,$73,$48,$5B
	.DB $E7,$52,$C7,$52,$8A,$6B,$49,$63,$2C,$7C,$6A,$6B,$66,$4A,$86,$4A
	.DB $86,$4A,$86,$4A,$86,$4A,$07,$53,$89,$63,$89,$63,$27,$5B,$A5,$4A
	.DB $68,$63,$06,$5B,$44,$42,$0B,$7C,$0B,$7C,$48,$63,$C7,$52,$A6,$4A
	.DB $29,$63,$28,$5B,$65,$42,$C3,$31,$C2,$29,$E3,$31,$C2,$31,$67,$63
	.DB $63,$42,$05,$5B,$88,$6B,$0A,$7C,$C9,$73,$0A,$7C,$48,$63,$C6,$52
	.DB $25,$3A,$66,$42,$45,$42,$C3,$31,$A2,$29,$85,$42,$89,$63,$A4,$4A
	.DB $87,$6B,$E5,$52,$47,$63,$C9,$73,$CA,$73,$C5,$4A,$48,$63,$48,$63
	.DB $E8,$52,$04,$3A,$04,$3A,$86,$42,$A7,$4A,$86,$42,$44,$3A,$47,$5B
	.DB $C9,$73,$88,$63,$07,$5B,$E3,$31,$86,$4A,$44,$42,$A2,$29,$C3,$31
	.DB $E8,$5A,$C7,$52,$25,$3A,$04,$32,$86,$42,$C7,$4A,$A6,$4A,$28,$5B
	.DB $08,$5B,$E7,$52,$04,$3A,$66,$4A,$C4,$31,$46,$42,$0A,$5B,$E5,$39
	.DB $69,$6B,$44,$3A,$45,$42,$49,$5B,$E4,$29,$66,$42,$A7,$4A,$A7,$52
	.DB $86,$52,$C3,$39,$29,$6B,$E8,$62,$24,$42,$A6,$52,$CB,$7B,$49,$6B
	.DB $A6,$4A,$A6,$4A,$86,$4A,$C7,$4A,$04,$32,$65,$42,$66,$42,$04,$42
	.DB $04,$42,$82,$29,$45,$4A,$65,$4A,$65,$4A,$65,$4A,$E7,$5A,$85,$4A
	.DB $44,$42,$EB,$7B,$AA,$6B,$07,$5B,$85,$4A,$C6,$52,$C6,$52,$65,$4A
	.DB $86,$4A,$24,$42,$86,$4A,$C7,$52,$89,$6B,$C6,$52,$E7,$5A,$03,$3A
	.DB $C7,$52,$48,$63,$C6,$52,$C6,$52,$85,$4A,$06,$5B,$E6,$52,$85,$4A
	.DB $C6,$4A,$89,$63,$0B,$74,$EB,$73,$EB,$73,$64,$42,$A5,$4A,$A5,$4A
	.DB $A6,$52,$C2,$31,$03,$3A,$CA,$73,$88,$6B,$A9,$73,$E6,$5A,$02,$32
	.DB $64,$42,$EA,$73,$EE,$94,$2B,$7C,$06,$53,$40,$19,$02,$32,$64,$42
	.DB $CB,$7B,$65,$4A,$65,$4A,$C9,$73,$84,$4A,$A5,$4A,$43,$42,$05,$53
	.DB $67,$63,$E9,$73,$8B,$84,$A8,$6B,$A4,$4A,$02,$32,$27,$5B,$8C,$84
	.DB $86,$4A,$C3,$31,$81,$29,$44,$3A,$23,$3A,$88,$63,$4A,$7C,$29,$7C
	.DB $8A,$8C,$C7,$6B,$87,$6B,$25,$5B,$87,$6B,$E9,$73,$6B,$84,$0A,$7C
	.DB $E4,$39,$65,$42,$85,$42,$27,$53,$07,$53,$27,$53,$05,$4B,$86,$63
	.DB $AA,$8C,$45,$63,$E3,$52,$24,$5B,$6A,$84,$CC,$94,$E9,$73,$06,$5B
	.DB $C5,$52,$0A,$74,$2A,$7C,$AB,$84,$E7,$6B,$24,$53,$24,$53,$26,$63
	.DB $06,$63,$E5,$5A,$83,$4A,$C7,$73,$AB,$8C,$88,$6B,$28,$63,$09,$5B
	.DB $C9,$73,$8B,$84,$4A,$7C,$C8,$6B,$C7,$6B,$25,$53,$46,$5B,$85,$4A
	.DB $44,$42,$23,$42,$42,$42,$C8,$6B,$8B,$8C,$67,$63,$E6,$52,$A6,$4A
	.DB $E8,$73,$6A,$7C,$E9,$73,$A4,$42,$05,$53,$42,$3A,$23,$3A,$E7,$52
	.DB $C7,$52,$C6,$52,$64,$42,$A4,$4A,$E5,$52,$C5,$52,$67,$63,$88,$6B
	.DB $E8,$73,$6A,$84,$E9,$73,$C5,$4A,$E6,$52,$A5,$4A,$86,$4A,$28,$5B
	.DB $69,$5B,$89,$63,$07,$5B,$44,$42,$C2,$31,$23,$3A,$27,$63,$84,$4A
	.DB $25,$5B,$A8,$6B,$26,$5B,$C6,$52,$65,$42,$28,$5B,$89,$63,$2A,$74
	.DB $27,$53,$64,$3A,$64,$42,$44,$3A,$24,$42,$A6,$52,$28,$63,$A6,$52
	.DB $06,$5B,$A9,$73,$A9,$6B,$CA,$73,$27,$5B,$2B,$84,$8C,$8C,$4A,$74
	.DB $25,$53,$22,$32,$84,$42,$A6,$52,$A6,$52,$49,$63,$28,$63,$CA,$7B
	.DB $A6,$52,$28,$63,$89,$6B,$68,$6B,$06,$5B,$C8,$73,$09,$74,$45,$5B
	.DB $66,$5B,$88,$63,$2C,$84,$8A,$73,$A7,$5A,$E7,$62,$65,$52,$64,$4A
	.DB $2A,$63,$E8,$5A,$69,$6B,$C5,$52,$05,$5B,$C7,$73,$29,$7C,$09,$74
	.DB $09,$74,$C9,$73,$EA,$7B,$C7,$5A,$25,$4A,$28,$6B,$06,$63,$A3,$52
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$38,$CE,$36,$CE
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$32,$AD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$31,$AD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5C,$E7,$1A,$DF,$36,$C6,$B2,$BD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$B8,$D6,$B3,$BD,$30,$AD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$1A,$DF,$56,$CE,$10,$AD,$2B,$94
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$FB,$D6,$D9,$D6,$F4,$BD,$8D,$94,$0A,$8C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$BA,$CE,$77,$CE,$72,$B5,$0B,$8C,$6B,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$D9,$D6,$51,$AD,$AE,$9C,$CE,$A4,$6C,$9C,$E9,$83
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$B8,$CE,$10,$A5,$8D,$9C,$4B,$94,$EA,$8B,$67,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$3B,$DF,$F5,$BD,$8E,$94,$8D,$9C,$2B,$94,$C9,$83,$88,$7B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $FA,$D6,$16,$BE,$31,$A5,$6D,$94,$6C,$9C,$2B,$94,$06,$73,$47,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$B9,$CE
	.DB $B8,$D6,$F0,$9C,$AE,$9C,$AD,$A4,$4C,$9C,$4B,$9C,$23,$5A,$CD,$A4
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$3A,$DF
	.DB $D7,$DE,$AE,$9C,$2C,$94,$8C,$9C,$6C,$A4,$8C,$A4,$E6,$6A,$EB,$8B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$D8,$D6
	.DB $13,$C6,$AE,$9C,$A9,$83,$CA,$8B,$4B,$A4,$0A,$94,$88,$83,$A9,$83
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$F0,$9C
	.DB $4C,$8C,$8D,$9C,$67,$7B,$47,$83,$EA,$93,$84,$6A,$E6,$72,$07,$73
	.DB $11,$A5,$2C,$8C,$2C,$94,$6D,$9C,$8E,$A4,$14,$D6,$EB,$8B,$EE,$AC
	.DB $AD,$9C,$A9,$7B,$2C,$8C,$4D,$9C,$F4,$CD,$CB,$8B,$4A,$7B,$6E,$9C
	.DB $92,$BD,$B2,$BD,$51,$BD,$30,$B5,$CF,$AC,$10,$B5,$AE,$A4,$50,$B5
	.DB $8D,$9C,$A6,$62,$AA,$7B,$71,$BD,$8E,$9C,$AB,$83,$72,$BD,$0D,$94
	.DB $14,$CE,$F3,$C5,$50,$B5,$30,$B5,$0F,$AD,$4D,$94,$72,$BD,$30,$B5
	.DB $0B,$8C,$E7,$6A,$EF,$AC,$31,$B5,$CC,$8B,$CC,$8B,$8F,$A4,$09,$73
	.DB $D2,$C5,$CD,$A4,$AD,$A4,$0F,$AD,$50,$B5,$8E,$9C,$72,$BD,$D3,$CD
	.DB $51,$BD,$EB,$8B,$F0,$AC,$29,$73,$0D,$94,$0D,$94,$8C,$83,$09,$73
	.DB $50,$B5,$4B,$94,$4F,$B5,$2F,$B5,$50,$B5,$92,$BD,$72,$BD,$D3,$CD
	.DB $D3,$CD,$8E,$A4,$CB,$8B,$4A,$7B,$B0,$A4,$CD,$8B,$ED,$93,$8C,$83
	.DB $ED,$AC,$AC,$A4,$90,$BD,$CD,$A4,$8E,$9C,$35,$CE,$D4,$C5,$92,$BD
	.DB $8B,$83,$6E,$9C,$6A,$83,$0D,$94,$4F,$9C,$D1,$AC,$90,$A4,$4B,$7B
	.DB $AC,$A4,$ED,$AC,$CD,$A4,$6C,$94,$4C,$94,$B3,$BD,$B3,$C5,$93,$C5
	.DB $C8,$6A,$0C,$94,$09,$73,$6B,$83,$AD,$8B,$57,$DE,$D1,$AC,$0A,$73
	.DB $8B,$9C,$0D,$AD,$2A,$94,$CE,$A4,$EF,$A4,$D0,$A4,$8F,$9C,$4E,$9C
	.DB $4E,$A4,$8F,$A4,$4B,$83,$0D,$94,$70,$A4,$16,$D6,$EE,$93,$C9,$6A
	.DB $47,$73,$50,$AD,$50,$AD,$71,$B5,$4D,$94,$AF,$9C,$2D,$94,$CF,$AC
	.DB $AE,$AC,$6D,$A4,$0C,$94,$31,$B5,$92,$BD,$F4,$C5,$0C,$8C,$C8,$62
	.DB $68,$73,$91,$B5,$50,$AD,$10,$A5,$4D,$8C,$F0,$A4,$AE,$A4,$CF,$AC
	.DB $4D,$9C,$0C,$94,$8A,$83,$CF,$A4,$B3,$C5,$F0,$AC,$8B,$7B,$09,$6B
	.DB $2B,$8C,$F3,$C5,$0F,$A5,$8D,$94,$6D,$94,$51,$B5,$30,$B5,$4E,$9C
	.DB $8F,$A4,$8F,$A4,$0D,$94,$4E,$9C,$72,$BD,$4E,$94,$8F,$9C,$69,$73
	.DB $8D,$9C,$75,$D6,$91,$BD,$10,$AD,$EF,$AC,$CF,$AC,$EC,$8B,$4E,$9C
	.DB $4E,$9C,$09,$73,$8B,$83,$8F,$A4,$D4,$CD,$2D,$94,$EC,$8B,$49,$73
	.DB $30,$B5,$91,$BD,$0B,$8C,$CB,$83,$8E,$A4,$EF,$AC,$4D,$9C,$4E,$9C
	.DB $6F,$9C,$05,$52,$AC,$8B,$F0,$AC,$93,$C5,$4D,$9C,$6A,$7B,$C7,$62
	.DB $AE,$A4,$50,$B5,$6D,$9C,$6D,$9C,$EF,$AC,$4C,$9C,$AA,$8B,$4E,$9C
	.DB $D0,$AC,$87,$62,$8F,$A4,$F0,$AC,$AF,$A4,$2C,$94,$29,$7B,$65,$5A
	.DB $8D,$9C,$0F,$AD,$2C,$94,$EB,$8B,$4C,$9C,$4B,$9C,$8D,$A4,$CF,$A4
	.DB $6E,$9C,$C7,$62,$8E,$A4,$CF,$AC,$4D,$9C,$69,$83,$45,$5A,$28,$73
	.DB $2C,$94,$EF,$AC,$2C,$94,$69,$7B,$AA,$8B,$EA,$93,$0F,$BD,$8E,$A4
	.DB $8F,$9C,$8E,$9C,$72,$BD,$8D,$9C,$AA,$83,$08,$73,$08,$73,$0C,$94
	.DB $8F,$A4,$6E,$A4,$0C,$94,$CF,$AC,$48,$7B,$EE,$B4,$EE,$AC,$0F,$AD
	.DB $0F,$A5,$CE,$9C,$6C,$8C,$88,$73,$C9,$7B,$E5,$62,$C9,$7B,$26,$6B
	.DB $AF,$A4,$2D,$94,$EC,$8B,$EB,$93,$CA,$8B,$70,$BD,$70,$BD,$2F,$B5
	.DB $91,$BD,$91,$BD,$EE,$A4,$ED,$A4,$ED,$A4,$EA,$83,$88,$7B,$27,$6B
	.DB $8B,$83,$4A,$7B,$49,$7B,$E8,$72,$8A,$83,$30,$BD,$EF,$AC,$EE,$B4
	.DB $B1,$CD,$50,$BD,$91,$C5,$2F,$BD,$CE,$AC,$0B,$94,$69,$7B,$2B,$94
	.DB $C8,$6A,$4A,$7B,$EC,$8B,$EC,$8B,$EC,$93,$30,$B5,$CE,$AC,$AE,$AC
	.DB $CE,$B4,$8A,$8B,$50,$C5,$EF,$B4,$AE,$AC,$EB,$93,$EB,$93,$8A,$83
	.DB $A8,$62,$A8,$62,$51,$BD,$AF,$A4,$EF,$AC,$AE,$A4,$EF,$B4,$B2,$CD
	.DB $CE,$B4,$69,$8B,$4D,$A4,$0F,$BD,$4D,$A4,$CB,$8B,$49,$7B,$8E,$A4
	.DB $68,$62,$83,$41,$32,$B5,$0C,$8C,$51,$B5,$0C,$8C,$10,$B5,$50,$BD
	.DB $CE,$B4,$CF,$B4,$69,$83,$CF,$AC,$AA,$8B,$4D,$9C,$69,$7B,$0C,$94
	.DB $A9,$62,$C5,$49,$4F,$9C,$0D,$8C,$B3,$C5,$AF,$A4,$92,$BD,$2C,$94
	.DB $EB,$8B,$4C,$9C,$C7,$6A,$EB,$8B,$CB,$8B,$4D,$9C,$0C,$94,$49,$7B
	.DB $88,$62,$A8,$62,$2A,$73,$8F,$9C,$B3,$C5,$B3,$C5,$B2,$C5,$D2,$C5
	.DB $6C,$94,$E7,$6A,$A6,$62,$08,$73,$0B,$8C,$A6,$62,$A6,$62,$65,$5A
	.DB $E4,$41,$2D,$8C,$52,$B5,$6D,$94,$72,$BD,$10,$B5,$50,$BD,$B1,$CD
	.DB $EB,$93,$AA,$8B,$C7,$6A,$C8,$6A,$C8,$6A,$E9,$6A,$88,$5A,$67,$5A
	.DB $C8,$62,$CF,$A4,$CF,$A4,$0C,$8C,$92,$BD,$71,$BD,$EF,$AC,$0B,$94
	.DB $69,$83,$65,$5A,$86,$62,$A6,$62,$8F,$A4,$EC,$8B,$E8,$6A,$29,$73
	.DB $49,$73,$10,$AD,$10,$AD,$0F,$AD,$D2,$C5,$50,$B5,$0C,$94,$0C,$94
	.DB $AB,$8B,$25,$5A,$C7,$6A,$65,$5A,$49,$7B,$07,$6B,$E7,$6A,$4C,$94
	.DB $A7,$62,$2C,$94,$30,$B5,$33,$CE,$B1,$BD,$2F,$AD,$4C,$94,$8F,$A4
	.DB $AC,$8B,$C8,$6A,$C7,$6A,$49,$7B,$07,$73,$C9,$8B,$0A,$94,$6B,$9C
	.DB $86,$5A,$69,$7B,$CE,$A4,$13,$CE,$50,$B5,$B1,$BD,$B3,$C5,$90,$9C
	.DB $4C,$7B,$88,$62,$25,$52,$EB,$8B,$AD,$A4,$CD,$AC,$67,$7B,$A8,$83
	.DB $AA,$83,$2C,$94,$30,$B5,$F3,$C5,$71,$B5,$34,$CE,$35,$CE,$F1,$A4
	.DB $EA,$6A,$E9,$62,$4A,$73,$8E,$9C,$30,$B5,$4C,$94,$85,$62,$88,$83
	.DB $0F,$B5,$AE,$A4,$72,$BD,$14,$CE,$D3,$C5,$14,$CE,$D3,$BD,$8F,$94
	.DB $26,$4A,$6F,$94,$F1,$A4,$B3,$BD,$31,$B5,$8F,$A4,$AB,$83,$8D,$A4
	.DB $51,$BD,$CB,$8B,$F0,$AC,$D3,$C5,$14,$C6,$15,$C6,$B3,$BD,$AB,$73
	.DB $83,$31,$D4,$B5,$4E,$8C,$B4,$BD,$11,$AD,$32,$B5,$6B,$7B,$32,$B5
	.DB $C5,$5A,$26,$6B,$4C,$94,$8D,$9C,$AA,$83,$A6,$62,$86,$52,$46,$42
	.DB $C5,$29,$05,$32,$C8,$52,$C7,$5A,$65,$4A,$E7,$62,$4C,$94,$6C,$9C
	.DB $C6,$62,$C6,$62,$89,$7B,$CB,$83,$69,$73,$86,$5A,$04,$4A,$A3,$31
	.DB $A7,$52,$E8,$5A,$A6,$5A,$48,$73,$4C,$94,$CE,$AC,$CE,$AC,$CD,$AC
	.DB $EB,$8B,$89,$7B,$C7,$62,$45,$52,$05,$4A,$25,$52,$46,$52,$45,$52
	.DB $69,$7B,$68,$7B,$E6,$6A,$89,$83,$0A,$94,$88,$8B,$06,$7B,$A9,$8B
	.DB $AB,$83,$EC,$8B,$6A,$7B,$09,$6B,$05,$4A,$C4,$41,$A3,$41,$48,$73
	.DB $89,$83,$07,$73,$07,$73,$EA,$8B,$EA,$93,$27,$7B,$E6,$72,$C6,$6A
	.DB $6A,$83,$08,$73,$E8,$6A,$EC,$8B,$6A,$7B,$8B,$7B,$6A,$7B,$CB,$8B
	.DB $8A,$83,$86,$62,$04,$52,$A6,$62,$08,$6B,$86,$5A,$25,$52,$04,$4A
	.DB $08,$73,$49,$7B,$8A,$83,$CF,$A4,$0C,$8C,$EB,$8B,$69,$7B,$28,$73
	.DB $66,$5A,$45,$52,$08,$6B,$8A,$7B,$6A,$73,$6A,$73,$AC,$7B,$46,$4A
	.DB $A6,$62,$E7,$6A,$64,$5A,$68,$7B,$06,$6B,$C9,$83,$A9,$83,$28,$73
	.DB $45,$5A,$E8,$6A,$CB,$8B,$E8,$6A,$04,$4A,$25,$52,$87,$5A,$8B,$7B
	.DB $24,$52,$C6,$6A,$84,$5A,$2A,$94,$2A,$94,$AC,$A4,$E9,$8B,$48,$83
	.DB $69,$83,$2C,$9C,$69,$83,$C2,$49,$E4,$51,$C7,$6A,$45,$5A,$4A,$7B
	.DB $A8,$5A,$6A,$7B,$CB,$8B,$CF,$AC,$AD,$A4,$2B,$94,$07,$73,$A9,$8B
	.DB $69,$83,$AA,$8B,$65,$5A,$C7,$62,$C7,$5A,$26,$4A,$05,$42,$87,$52
	.DB $49,$73,$CB,$83,$2C,$94,$AE,$A4,$6D,$9C,$4C,$9C,$2B,$94,$AA,$8B
	.DB $69,$83,$E7,$6A,$24,$52,$86,$5A,$8B,$7B,$86,$52,$62,$31,$67,$52
	.DB $2B,$94,$4C,$94,$AE,$AC,$AE,$AC,$6D,$A4,$2C,$9C,$4D,$9C,$CA,$8B
	.DB $8A,$83,$49,$7B,$28,$6B,$E4,$41,$46,$52,$25,$4A,$C8,$5A,$A7,$5A
	.DB $6B,$9C,$6B,$9C,$CE,$AC,$CE,$B4,$AF,$AC,$2D,$9C,$CB,$8B,$0C,$94
	.DB $69,$7B,$E7,$6A,$A7,$5A,$A2,$39,$E4,$41,$87,$5A,$4A,$73,$2D,$8C
	.DB $0A,$94,$4B,$9C,$2B,$9C,$6D,$A4,$CF,$B4,$8F,$A4,$CC,$8B,$AA,$83
	.DB $4D,$94,$0C,$8C,$45,$52,$04,$4A,$08,$6B,$2D,$8C,$AB,$83,$E8,$6A
	.DB $EA,$93,$2B,$9C,$48,$7B,$CB,$8B,$4D,$9C,$AF,$AC,$EC,$93,$AB,$83
	.DB $F0,$AC,$D0,$AC,$29,$73,$29,$73,$29,$73,$2D,$94,$EC,$8B,$8A,$83
	.DB $EB,$8B,$CB,$8B,$85,$62,$EB,$8B,$0C,$94,$CE,$A4,$CB,$83,$AB,$83
	.DB $6F,$A4,$6B,$83,$09,$73,$8B,$83,$E8,$72,$C8,$6A,$E8,$72,$8A,$83
	.DB $AB,$83,$29,$73,$04,$52,$6D,$9C,$6D,$9C,$0F,$AD,$8A,$83,$26,$5A
	.DB $D1,$B4,$0D,$94,$4A,$83,$C8,$6A,$87,$62,$A7,$6A,$AB,$8B,$A7,$6A
	.DB $2F,$B5,$68,$7B,$0B,$94,$0F,$AD,$4C,$94,$07,$6B,$08,$6B,$AA,$83
	.DB $0B,$8C,$08,$73,$C2,$41,$65,$5A,$28,$6B,$85,$5A,$65,$52,$07,$63
	.DB $90,$C5,$2B,$9C,$27,$73,$A9,$83,$AD,$A4,$07,$6B,$04,$4A,$C7,$62
	.DB $E8,$6A,$E3,$41,$45,$52,$49,$73,$08,$6B,$C6,$5A,$85,$5A,$07,$63
	.DB $48,$83,$06,$73,$85,$62,$85,$62,$69,$7B,$44,$52,$86,$5A,$66,$52
	.DB $E7,$62,$A3,$39,$C7,$5A,$CB,$83,$08,$63,$28,$6B,$A6,$52,$85,$52
	.DB $03,$52,$24,$52,$C7,$62,$E3,$41,$82,$31,$21,$29,$87,$52,$66,$4A
	.DB $29,$63,$66,$52,$C7,$5A,$49,$6B,$E7,$5A,$48,$6B,$E7,$5A,$E2,$39
	.DB $87,$52,$66,$52,$E9,$5A,$C4,$39,$84,$29,$E5,$39,$E5,$39,$25,$42
	.DB $A7,$52,$E8,$5A,$A7,$52,$86,$52,$86,$52,$E7,$5A,$89,$73,$07,$63
	.DB $E9,$5A,$47,$4A,$E9,$5A,$47,$42,$27,$42,$68,$4A,$A4,$31,$A7,$52
	.DB $65,$4A,$08,$5B,$E7,$5A,$65,$4A,$04,$42,$44,$4A,$89,$73,$0B,$84
	.DB $4A,$73,$A8,$5A,$E9,$62,$47,$4A,$26,$42,$A4,$39,$C4,$39,$6A,$6B
	.DB $E7,$5A,$85,$52,$A6,$52,$44,$4A,$E3,$41,$85,$52,$C6,$5A,$48,$6B
	.DB $29,$73,$29,$73,$A7,$62,$25,$52,$C8,$6A,$46,$52,$08,$6B,$8A,$73
	.DB $69,$73,$E2,$39,$C2,$39,$E3,$41,$44,$4A,$48,$6B,$24,$4A,$C6,$62
	.DB $C8,$5A,$2D,$84,$E8,$62,$A7,$5A,$8A,$7B,$AA,$83,$A6,$5A,$EC,$83
	.DB $E8,$62,$C4,$41,$83,$39,$25,$4A,$C8,$5A,$C7,$5A,$46,$4A,$C7,$5A
	.DB $52,$AD,$2D,$8C,$25,$4A,$AB,$83,$A7,$62,$45,$5A,$8E,$A4,$AB,$7B
	.DB $86,$5A,$25,$4A,$E7,$62,$08,$63,$86,$52,$86,$52,$08,$63,$08,$63
	.DB $D0,$A4,$6B,$73,$87,$5A,$0D,$94,$A7,$6A,$E7,$6A,$8E,$A4,$49,$73
	.DB $28,$6B,$49,$6B,$69,$73,$28,$6B,$A6,$5A,$85,$52,$C6,$5A,$48,$6B
	.DB $0D,$8C,$EC,$8B,$AB,$83,$8B,$83,$E8,$72,$AB,$8B,$69,$83,$A9,$7B
	.DB $CA,$7B,$EA,$83,$A9,$73,$47,$6B,$06,$63,$26,$63,$47,$6B,$E5,$5A
	.DB $E8,$6A,$4A,$7B,$87,$62,$49,$7B,$CB,$8B,$0C,$94,$8E,$A4,$28,$6B
	.DB $A5,$5A,$85,$52,$27,$6B,$47,$6B,$C5,$5A,$05,$63,$C9,$7B,$47,$6B
	.DB $4A,$7B,$6A,$7B,$25,$52,$49,$7B,$CB,$8B,$08,$73,$08,$73,$85,$52
	.DB $64,$52,$64,$52,$A5,$5A,$05,$63,$26,$63,$46,$6B,$67,$6B,$2A,$84
	.DB $08,$73,$28,$73,$C7,$62,$CB,$83,$2D,$94,$EC,$8B,$28,$6B,$E7,$62
	.DB $A9,$7B,$A9,$7B,$06,$63,$26,$63,$09,$84,$2A,$84,$A7,$73,$E9,$7B
	.DB $49,$7B,$86,$5A,$24,$52,$28,$6B,$C7,$62,$AB,$83,$0C,$8C,$28,$6B
	.DB $69,$73,$89,$73,$47,$6B,$26,$6B,$87,$73,$E8,$7B,$29,$84,$C8,$7B
	.DB $EB,$83,$28,$6B,$48,$6B,$85,$52,$AA,$7B,$07,$63,$07,$63,$26,$6B
	.DB $E5,$62,$88,$73,$47,$6B,$47,$6B,$47,$6B,$E6,$5A,$48,$6B,$E6,$52
	.DB $E6,$62,$03,$42,$A6,$5A,$E6,$5A,$89,$73,$06,$63,$47,$6B,$88,$73
	.DB $27,$6B,$68,$73,$C5,$5A,$43,$4A,$44,$4A,$44,$4A,$07,$63,$A5,$52
	.DB $64,$4A,$64,$4A,$47,$6B,$4B,$84,$4B,$84,$88,$73,$68,$6B,$88,$73
	.DB $48,$63,$48,$63,$E6,$5A,$85,$4A,$A5,$52,$85,$4A,$07,$5B,$24,$42
	.DB $A5,$52,$67,$6B,$88,$6B,$8B,$8C,$4A,$84,$C8,$73,$26,$63,$68,$63
	.DB $07,$5B,$A6,$52,$E7,$5A,$E7,$5A,$48,$63,$C7,$52,$E7,$5A,$E7,$5A
	.DB $27,$63,$68,$6B,$06,$63,$E9,$7B,$88,$73,$A8,$73,$88,$6B,$27,$5B
	.DB $E6,$5A,$03,$3A,$85,$4A,$44,$42,$A6,$52,$86,$4A,$A6,$52,$45,$42
	.DB $48,$6B,$06,$63,$06,$63,$C9,$7B,$E6,$62,$06,$63,$88,$73,$64,$4A
	.DB $C6,$5A,$24,$42,$E7,$5A,$44,$42,$86,$4A,$65,$4A,$A7,$52,$08,$5B
	.DB $C6,$5A,$C6,$62,$07,$6B,$28,$6B,$44,$52,$23,$4A,$C6,$62,$84,$52
	.DB $06,$63,$64,$4A,$89,$73,$E7,$62,$28,$6B,$C7,$5A,$A7,$52,$86,$4A
	.DB $48,$73,$69,$7B,$28,$6B,$A6,$62,$24,$52,$65,$5A,$07,$6B,$88,$73
	.DB $67,$73,$02,$42,$27,$6B,$07,$63,$8A,$73,$E8,$62,$86,$52,$29,$63
	.DB $C7,$5A,$8A,$73,$49,$6B,$07,$5B,$24,$42,$A5,$52,$A6,$52,$45,$4A
	.DB $08,$63,$29,$6B,$A7,$5A,$A7,$52,$45,$4A,$A6,$52,$69,$6B,$27,$5B
	.DB $E7,$5A,$07,$63,$48,$63,$07,$5B,$07,$63,$07,$63,$E7,$5A,$A6,$52
	.DB $86,$52,$65,$4A,$A7,$52,$49,$6B,$08,$63,$A5,$52,$E6,$5A,$A8,$6B
	.DB $68,$6B,$27,$63,$CA,$73,$48,$63,$CA,$7B,$68,$6B,$28,$63,$65,$4A
	.DB $45,$4A,$24,$42,$C6,$5A,$C9,$73,$0A,$7C,$26,$5B,$A4,$4A,$C3,$4A
	.DB $47,$63,$88,$6B,$C9,$73,$07,$63,$47,$63,$06,$5B,$A6,$52,$44,$4A
	.DB $C7,$52,$C6,$52,$48,$63,$E9,$73,$AB,$8C,$C7,$73,$E4,$52,$82,$4A
	.DB $47,$6B,$C8,$73,$67,$6B,$27,$63,$C6,$5A,$E6,$5A,$85,$4A,$65,$4A
	.DB $85,$4A,$85,$4A,$47,$63,$67,$63,$E9,$7B,$26,$5B,$A3,$4A,$62,$42
	.DB $47,$6B,$67,$6B,$E5,$5A,$27,$63,$C6,$52,$27,$63,$C6,$52,$0B,$7C
	.DB $27,$5B,$A5,$4A,$2B,$7C,$C9,$73,$EA,$73,$27,$5B,$06,$5B,$A5,$4A
	.DB $E5,$5A,$A4,$52,$84,$4A,$64,$4A,$24,$42,$86,$52,$C6,$52,$68,$63
	.DB $C5,$52,$23,$3A,$48,$63,$85,$4A,$E8,$5A,$65,$42,$45,$42,$E3,$31
	.DB $46,$63,$E5,$5A,$47,$63,$44,$42,$24,$42,$65,$4A,$28,$63,$E6,$52
	.DB $67,$63,$07,$5B,$28,$63,$E4,$31,$29,$5B,$E9,$5A,$68,$4A,$26,$42
	.DB $8A,$63,$07,$53,$02,$32,$84,$42,$C8,$6B,$C7,$6B,$A4,$4A,$44,$42
	.DB $86,$4A,$48,$63,$44,$42,$C5,$52,$C4,$52,$05,$5B,$29,$7C,$A8,$6B
	.DB $C6,$4A,$44,$42,$44,$3A,$27,$5B,$C9,$6B,$A8,$6B,$47,$5B,$07,$5B
	.DB $07,$5B,$A9,$6B,$89,$6B,$EA,$73,$68,$63,$E5,$52,$A8,$6B,$68,$63
	.DB $E6,$52,$85,$4A,$85,$4A,$07,$5B,$08,$5B,$C6,$52,$C6,$52,$07,$5B
	.DB $C5,$4A,$E6,$52,$CA,$73,$EA,$73,$07,$5B,$C2,$31,$24,$3A,$85,$4A
	.DB $07,$5B,$E6,$52,$85,$4A,$86,$4A,$86,$4A,$46,$42,$25,$42,$88,$63
	.DB $47,$5B,$27,$5B,$2B,$7C,$CA,$6B,$28,$5B,$E4,$31,$25,$3A,$04,$3A
	.DB $06,$53,$A5,$4A,$65,$42,$66,$4A,$26,$42,$C4,$31,$A4,$31,$64,$42
	.DB $84,$42,$44,$3A,$C6,$4A,$44,$3A,$65,$42,$04,$32,$04,$32,$A7,$4A
	.DB $64,$42,$C2,$31,$65,$42,$08,$53,$A7,$4A,$04,$3A,$25,$3A,$04,$32
	.DB $65,$42,$65,$42,$A6,$4A,$65,$42,$E7,$52,$69,$63,$28,$5B,$69,$63
	.DB $23,$32,$A2,$29,$64,$3A,$69,$5B,$69,$5B,$28,$5B,$89,$63,$E4,$29
	.DB $25,$3A,$87,$42,$08,$53,$48,$5B,$89,$63,$6C,$7C,$68,$63,$27,$53
	.DB $AA,$63,$28,$53,$C6,$42,$E6,$4A,$27,$53,$06,$53,$07,$53,$C4,$29
	.DB $E5,$31,$87,$42,$49,$5B,$0B,$74,$CA,$6B,$6C,$7C,$06,$53,$63,$3A
	.DB $07,$5B,$E9,$52,$28,$3A,$AA,$52,$27,$3A,$24,$32,$A5,$4A,$22,$3A
	.DB $64,$42,$26,$5B,$AD,$8C,$0B,$7C,$C9,$6B,$A8,$63,$66,$5B,$A6,$63
	.DB $44,$3A,$47,$42,$E7,$39,$44,$21,$E5,$31,$C2,$29,$84,$42,$06,$53
	.DB $06,$5B,$88,$6B,$27,$5B,$A5,$4A,$47,$5B,$67,$63,$E9,$73,$8A,$84
	.DB $04,$3A,$05,$3A,$07,$3A,$C5,$31,$C8,$52,$A6,$4A,$06,$53,$E9,$73
	.DB $67,$63,$2A,$7C,$C6,$52,$85,$4A,$C5,$52,$43,$42,$46,$63,$49,$7C
	.DB $46,$42,$67,$42,$26,$42,$E8,$52,$C7,$52,$E7,$52,$07,$5B,$26,$5B
	.DB $83,$4A,$67,$63,$07,$5B,$A9,$73,$E6,$52,$23,$3A,$27,$63,$87,$6B
	.DB $83,$29,$04,$3A,$24,$3A,$E7,$52,$07,$5B,$68,$63,$A9,$6B,$C9,$73
	.DB $A8,$6B,$68,$63,$E6,$52,$07,$5B,$82,$29,$61,$29,$02,$3A,$88,$6B
	.DB $E4,$31,$24,$42,$06,$5B,$46,$63,$AC,$8C,$4B,$84,$A9,$6B,$84,$42
	.DB $68,$63,$27,$5B,$65,$42,$E4,$31,$A3,$29,$66,$4A,$85,$4A,$85,$4A
	.DB $29,$63,$88,$6B,$4A,$7C,$E7,$73,$AB,$8C,$A8,$6B,$A4,$4A,$47,$5B
	.DB $A9,$63,$89,$63,$86,$42,$A3,$29,$66,$42,$87,$42,$46,$42,$C6,$4A
	.DB $45,$42,$E9,$73,$28,$7C,$C6,$6B,$E3,$52,$05,$5B,$C5,$52,$6B,$7C
	.DB $48,$53,$48,$53,$A7,$42,$05,$32,$6B,$63,$C8,$4A,$E9,$52,$E8,$52
	.DB $08,$74,$C7,$6B,$E4,$4A,$C7,$6B,$08,$74,$29,$7C,$09,$7C,$E8,$6B
	.DB $E8,$6B,$25,$53,$86,$63,$09,$7C,$29,$7C,$6B,$8C,$C9,$7B,$48,$6B
	.DB $09,$74,$0A,$74,$26,$53,$C8,$6B,$E9,$73,$2A,$74,$C8,$6B,$C8,$6B
	.DB $29,$74,$E8,$73,$49,$7C,$6B,$84,$2A,$84,$4B,$84,$A9,$73,$64,$4A
	.DB $E6,$52,$27,$53,$48,$5B,$CA,$6B,$A5,$4A,$A5,$42,$06,$53,$A4,$4A
	.DB $66,$5B,$A7,$6B,$09,$74,$E8,$73,$67,$63,$46,$63,$C5,$52,$A4,$4A
	.DB $A9,$6B,$C9,$6B,$E6,$4A,$88,$63,$C9,$6B,$68,$5B,$23,$32,$26,$5B
	.DB $A8,$6B,$09,$74,$2A,$7C,$E9,$73,$87,$6B,$67,$63,$06,$5B,$C5,$52
	.DB $A8,$6B,$09,$74,$47,$5B,$C8,$6B,$2A,$74,$4B,$7C,$46,$5B,$C9,$6B
	.DB $C9,$73,$E9,$73,$A8,$6B,$87,$63,$88,$63,$67,$63,$47,$5B,$47,$5B
	.DB $C4,$4A,$87,$63,$09,$74,$09,$74,$26,$5B,$C8,$6B,$6B,$84,$6B,$84
	.DB $EA,$7B,$E9,$73,$68,$63,$88,$63,$C9,$6B,$A8,$63,$A8,$63,$A8,$63
	.DB $06,$5B,$CA,$73,$88,$6B,$89,$6B,$CA,$73,$6C,$84,$C9,$73,$27,$5B
	.DB $A5,$52,$E6,$52,$A5,$4A,$E5,$4A,$47,$5B,$E5,$4A,$05,$4B,$46,$53
	.DB $04,$3A,$C7,$52,$08,$63,$AB,$73,$EB,$7B,$2D,$84,$49,$6B,$85,$4A
	.DB $64,$42,$E6,$52,$E6,$52,$47,$5B,$E9,$6B,$67,$5B,$67,$5B,$C8,$63
	.DB $45,$3A,$A6,$42,$CA,$6B,$2B,$7C,$4B,$7C,$4A,$7C,$A8,$6B,$06,$5B
	.DB $C6,$52,$C6,$5A,$A4,$52,$A8,$73,$46,$63,$66,$63,$66,$63,$46,$63
	.DB $4B,$7C,$8C,$84,$EE,$94,$CD,$8C,$A9,$6B,$67,$63,$4B,$84,$A9,$73
	.DB $07,$5B,$27,$63,$48,$63,$88,$6B,$88,$6B,$E9,$73,$A8,$73,$26,$5B
	.DB $0D,$95,$6B,$84,$4A,$84,$CD,$94,$CA,$73,$07,$5B,$2C,$84,$48,$63
	.DB $85,$4A,$E6,$5A,$CB,$73,$69,$63,$49,$63,$E7,$52,$04,$3A,$E3,$31
	.DB $69,$84,$46,$63,$A4,$4A,$48,$6B,$28,$63,$45,$4A,$86,$52,$24,$42
	.DB $E2,$39,$24,$42,$69,$6B,$08,$5B,$C8,$52,$C4,$31,$42,$21,$66,$42
	.DB $29,$84,$C8,$7B,$47,$6B,$65,$4A,$C3,$39,$83,$31,$A4,$39,$81,$31
	.DB $64,$52,$03,$42,$24,$42,$04,$3A,$E4,$39,$42,$21,$66,$42,$A7,$4A
	.DB $47,$63,$A9,$73,$0C,$84,$A7,$52,$62,$29,$84,$29,$A4,$31,$86,$5A
	.DB $2C,$94,$CA,$83,$A6,$5A,$44,$4A,$82,$29,$61,$21,$28,$5B,$66,$42
	.DB $A3,$39,$05,$42,$A8,$52,$05,$3A,$84,$29,$84,$29,$43,$29,$C8,$6A
	.DB $6D,$A4,$CE,$AC,$A9,$83,$68,$73,$A5,$52,$84,$4A,$47,$5B,$47,$5B
	.DB $26,$42,$27,$42,$C5,$31,$84,$29,$E6,$39,$07,$3A,$63,$29,$01,$39
	.DB $E4,$59,$E6,$72,$84,$5A,$27,$6B,$47,$6B,$A7,$6B,$25,$53,$E4,$42
	.DB $E7,$5A,$45,$4A,$45,$42,$25,$42,$46,$42,$E9,$52,$87,$4A,$67,$42
	.DB $A7,$4A,$A7,$52,$6A,$63,$E7,$52,$86,$4A,$69,$6B,$08,$63,$85,$52
	.DB $85,$52,$A6,$52,$E8,$5A,$C7,$52,$A7,$4A,$09,$5B,$87,$4A,$66,$42
	.DB $25,$42,$45,$42,$C3,$31,$41,$21,$45,$42,$E7,$5A,$A6,$52,$65,$4A
	.DB $27,$5B,$89,$6B,$CA,$73,$28,$5B,$66,$4A,$A7,$4A,$46,$42,$C6,$52
	.DB $65,$42,$28,$63,$A6,$52,$85,$4A,$89,$6B,$68,$63,$28,$5B,$A6,$4A
	.DB $27,$5B,$68,$63,$89,$6B,$A6,$4A,$04,$3A,$A6,$4A,$C7,$52,$C6,$52
	.DB $23,$42,$48,$63,$EA,$7B,$EA,$73,$0B,$7C,$A9,$6B,$69,$63,$85,$42
	.DB $68,$63,$47,$5B,$27,$5B,$44,$42,$C2,$31,$A6,$4A,$E7,$5A,$27,$5B
	.DB $A5,$4A,$85,$4A,$68,$6B,$68,$63,$27,$5B,$CA,$73,$68,$63,$0B,$74
	.DB $88,$63,$67,$5B,$68,$63,$E6,$52,$A5,$4A,$48,$63,$89,$6B,$A9,$6B
	.DB $89,$6B,$E6,$52,$E6,$5A,$85,$4A,$A6,$52,$CA,$73,$07,$5B,$65,$42
	.DB $06,$53,$E6,$4A,$88,$63,$A9,$6B,$89,$6B,$2B,$7C,$0B,$7C,$69,$6B
	.DB $48,$63,$28,$63,$E7,$5A,$44,$42,$85,$4A,$08,$5B,$C6,$52,$44,$42
	.DB $26,$53,$C5,$4A,$47,$5B,$47,$5B,$E6,$52,$E6,$52,$64,$42,$A9,$6B
	.DB $85,$4A,$48,$63,$28,$5B,$85,$4A,$85,$4A,$A6,$52,$49,$6B,$28,$63
	.DB $C8,$73,$46,$5B,$47,$63,$CA,$73,$48,$5B,$E7,$52,$E7,$52,$C9,$63
	.DB $A8,$63,$A8,$63,$06,$53,$88,$63,$88,$63,$A5,$4A,$69,$6B,$CB,$73
	.DB $A4,$4A,$63,$42,$06,$5B,$E6,$52,$44,$42,$A5,$4A,$E7,$52,$27,$5B
	.DB $88,$63,$C9,$6B,$06,$53,$E6,$52,$68,$6B,$26,$63,$68,$6B,$AD,$94
	.DB $40,$21,$81,$29,$65,$4A,$E2,$31,$A1,$31,$26,$5B,$89,$6B,$AA,$6B
	.DB $AA,$6B,$6C,$84,$0B,$7C,$88,$6B,$E9,$7B,$2A,$84,$2A,$84,$4A,$8C
	.DB $A7,$4A,$C7,$52,$E7,$52,$23,$42,$23,$42,$67,$6B,$A9,$73,$C7,$5A
	.DB $05,$42,$08,$63,$EA,$83,$67,$73,$66,$6B,$C7,$7B,$A7,$73,$66,$6B
	.DB $49,$5B,$C7,$4A,$C7,$52,$27,$5B,$C6,$52,$06,$5B,$27,$63,$07,$63
	.DB $65,$52,$07,$6B,$4C,$94,$6B,$94,$2A,$8C,$29,$84,$86,$73,$86,$73
	.DB $45,$3A,$82,$21,$65,$42,$89,$63,$C6,$4A,$24,$3A,$65,$4A,$E6,$62
	.DB $68,$7B,$48,$73,$A9,$83,$6B,$94,$6B,$94,$C9,$7B,$84,$52,$47,$6B
	.DB $E6,$4A,$84,$3A,$68,$53,$8A,$63,$86,$42,$25,$32,$04,$3A,$C5,$5A
	.DB $4A,$94,$87,$7B,$67,$73,$8C,$94,$6D,$94,$AA,$73,$07,$63,$08,$63
	.DB $06,$4B,$A8,$5B,$EB,$63,$08,$4B,$87,$3A,$0A,$53,$86,$4A,$21,$42
	.DB $E7,$83,$E4,$62,$06,$6B,$CE,$9C,$6D,$94,$EC,$7B,$6E,$8C,$6B,$6B
	.DB $25,$39,$86,$41,$A7,$41,$08,$42,$08,$42,$45,$29,$A7,$31,$25,$21
	.DB $45,$29,$C7,$39,$28,$42,$28,$42,$47,$42,$A8,$4A,$2A,$63,$C8,$52
	.DB $28,$4A,$CA,$5A,$07,$42,$45,$21,$45,$21,$24,$19,$86,$29,$86,$29
	.DB $08,$42,$27,$42,$A5,$31,$E6,$39,$E9,$5A,$6B,$6B,$4B,$63,$6A,$63
	.DB $68,$42,$A9,$4A,$27,$3A,$E6,$31,$E7,$31,$86,$29,$85,$29,$07,$3A
	.DB $89,$52,$4B,$63,$6C,$6B,$0A,$5B,$A8,$52,$A7,$4A,$A7,$4A,$C8,$52
	.DB $4A,$53,$8B,$63,$6C,$63,$A9,$4A,$A6,$31,$A5,$31,$68,$4A,$0A,$5B
	.DB $C8,$52,$C8,$52,$09,$5B,$09,$5B,$C8,$52,$08,$5B,$6A,$6B,$29,$5B
	.DB $A7,$4A,$A7,$4A,$E9,$5A,$88,$52,$A5,$39,$A5,$39,$27,$4A,$67,$4A
	.DB $86,$4A,$45,$42,$05,$3A,$E8,$5A,$2D,$7C,$6E,$8C,$EC,$7B,$CB,$73
	.DB $66,$4A,$C4,$39,$05,$42,$46,$4A,$26,$42,$26,$42,$A4,$31,$66,$4A
	.DB $28,$5B,$AB,$6B,$CB,$73,$AB,$73,$AB,$6B,$6A,$63,$09,$5B,$A7,$52
	.DB $0C,$84,$AA,$73,$69,$63,$A6,$4A,$04,$32,$E8,$4A,$69,$5B,$8E,$84
	.DB $6D,$84,$6E,$84,$2D,$7C,$29,$63,$25,$3A,$05,$42,$A8,$52,$67,$4A
	.DB $8A,$6B,$28,$5B,$CA,$6B,$89,$5B,$07,$43,$4C,$64,$30,$8D,$0F,$95
	.DB $8E,$8C,$CB,$6B,$C7,$52,$86,$4A,$A8,$52,$87,$4A,$06,$42,$E5,$39
	.DB $88,$6B,$47,$63,$88,$6B,$8A,$84,$2B,$9D,$29,$95,$E9,$8C,$09,$7C
	.DB $6D,$94,$0C,$84,$4A,$6B,$A9,$5A,$07,$42,$C6,$39,$86,$31,$85,$31
	.DB $4A,$84,$CB,$8C,$AA,$8C,$69,$84,$49,$7C,$4A,$84,$CD,$8C,$4B,$84
	.DB $0A,$7C,$E7,$5A,$05,$3A,$E5,$39,$E6,$39,$26,$3A,$47,$42,$2A,$5B
	.DB $E5,$52,$46,$63,$46,$63,$67,$63,$68,$63,$89,$6B,$CA,$73,$4B,$7C
	.DB $0A,$74,$48,$5B,$C7,$4A,$C7,$4A,$C8,$4A,$09,$53,$AB,$6B,$C7,$4A
	.DB $05,$42,$45,$42,$E8,$5A,$2B,$7C,$AC,$8C,$8A,$8C,$69,$84,$CB,$8C
	.DB $4A,$7C,$EA,$73,$A9,$6B,$CB,$73,$AA,$6B,$49,$63,$CA,$6B,$48,$5B
	.DB $E6,$5A,$48,$63,$0B,$7C,$AC,$94,$CC,$94,$AA,$8C,$AA,$8C,$0C,$9D
	.DB $E9,$7B,$47,$63,$C6,$5A,$27,$63,$48,$63,$65,$4A,$A6,$52,$A9,$73
	.DB $EB,$94,$A9,$8C,$A9,$8C,$AA,$94,$8A,$8C,$4A,$84,$8B,$8C,$88,$73
	.DB $26,$63,$47,$6B,$A5,$52,$85,$52,$A6,$5A,$23,$4A,$06,$63,$E6,$62
	.DB $69,$8C,$C5,$73,$65,$6B,$A6,$73,$87,$6B,$67,$6B,$A8,$73,$05,$5B
	.DB $26,$5B,$09,$7C,$47,$63,$06,$5B,$68,$6B,$E6,$5A,$C9,$73,$C9,$7B
	.DB $AA,$73,$8A,$73,$CA,$7B,$2B,$84,$09,$7C,$E8,$7B,$29,$84,$4A,$7C
	.DB $A8,$63,$A8,$63,$06,$53,$88,$63,$8C,$84,$88,$63,$A8,$6B,$8B,$8C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$1B,$D7,$3A,$E7,$F7,$E6
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$DA,$CE,$57,$CE,$10,$AD
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$98,$CE,$73,$AD,$8E,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$B9,$D6,$D4,$B5,$AF,$9C,$AE,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$5F,$15,$5F,$15,$B8,$D6,$B3,$B5,$AF,$9C,$4C,$94,$CA,$83
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $5F,$15,$3B,$DF,$19,$DF,$55,$CE,$0F,$AD,$4B,$94,$4B,$94,$A8,$7B
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $3B,$D7,$D8,$CE,$75,$CE,$D1,$BD,$0D,$AD,$AB,$9C,$8B,$9C,$6A,$94
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15
	.DB $D9,$C6,$35,$BE,$91,$B5,$2E,$AD,$0D,$AD,$EB,$AC,$AB,$A4,$8A,$94
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$98,$CE,$30,$AD
	.DB $EA,$93,$4B,$9C,$8C,$9C,$A9,$7B,$EA,$7B,$06,$63,$2B,$84,$A9,$73
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$5F,$15,$F5,$BD,$AD,$9C
	.DB $A9,$83,$8C,$9C,$C9,$83,$0A,$8C,$C9,$7B,$88,$73,$4C,$8C,$CE,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$5F,$15,$5C,$E7,$B8,$CE,$51,$AD,$0A,$8C
	.DB $A8,$83,$4A,$94,$A8,$7B,$6B,$94,$0A,$84,$C9,$7B,$AD,$94,$EE,$9C
	.DB $5F,$15,$5F,$15,$5F,$15,$1A,$DF,$56,$CE,$92,$B5,$0F,$A5,$29,$8C
	.DB $29,$94,$E8,$83,$4B,$94,$6B,$94,$8C,$94,$A9,$7B,$30,$AD,$69,$73
	.DB $5F,$15,$5F,$15,$D9,$D6,$35,$C6,$30,$AD,$CD,$A4,$ED,$A4,$EC,$AC
	.DB $AB,$9C,$C8,$83,$4A,$94,$4B,$94,$0B,$8C,$AA,$7B,$71,$B5,$4D,$94
	.DB $9E,$E7,$FA,$D6,$D4,$BD,$10,$AD,$CD,$A4,$AB,$A4,$CB,$A4,$6E,$B5
	.DB $6A,$94,$C8,$83,$47,$73,$EA,$83,$07,$6B,$CB,$83,$31,$B5,$F0,$AC
	.DB $5F,$15,$57,$CE,$31,$AD,$EE,$A4,$AC,$A4,$69,$A4,$8A,$A4,$CB,$A4
	.DB $2A,$8C,$C9,$83,$47,$73,$AA,$7B,$49,$73,$6E,$9C,$F0,$AC,$CC,$83
	.DB $78,$C6,$73,$AD,$10,$AD,$2F,$B5,$6A,$A4,$C6,$8B,$48,$9C,$E8,$8B
	.DB $4B,$94,$EA,$83,$2C,$8C,$AA,$83,$AF,$9C,$11,$B5,$31,$B5,$31,$B5
	.DB $2F,$B5,$ED,$B4,$10,$D6,$CB,$A4,$8B,$94,$4A,$84,$68,$6B,$A9,$7B
	.DB $EB,$8B,$08,$73,$AE,$A4,$6D,$9C,$50,$B5,$6C,$9C,$B1,$C5,$2E,$B5
	.DB $2B,$94,$AC,$AC,$AC,$AC,$E8,$8B,$A8,$7B,$09,$84,$47,$6B,$69,$73
	.DB $8E,$9C,$CA,$83,$CE,$A4,$8D,$9C,$AE,$A4,$CE,$A4,$70,$BD,$6C,$9C
	.DB $6C,$9C,$CD,$AC,$E9,$8B,$E9,$8B,$A9,$7B,$6B,$8C,$CA,$7B,$69,$73
	.DB $AE,$9C,$AA,$7B,$2C,$8C,$2C,$8C,$EB,$83,$8D,$9C,$0B,$8C,$C6,$62
	.DB $AD,$A4,$6C,$9C,$A9,$83,$0A,$8C,$2B,$8C,$8C,$9C,$6C,$94,$AE,$9C
	.DB $CF,$A4,$8A,$7B,$2C,$8C,$2C,$8C,$6D,$94,$EB,$8B,$49,$73,$A6,$5A
	.DB $0A,$8C,$88,$7B,$88,$7B,$89,$7B,$EA,$83,$EA,$8B,$AD,$A4,$91,$BD
	.DB $8D,$9C,$AA,$7B,$8D,$9C,$EB,$83,$0F,$AD,$69,$73,$AA,$83,$6D,$94
	.DB $0A,$84,$C9,$7B,$2B,$8C,$88,$7B,$88,$7B,$A8,$83,$CD,$AC,$2F,$B5
	.DB $EA,$8B,$CA,$83,$0F,$AD,$CA,$83,$EF,$AC,$49,$7B,$AE,$A4,$92,$BD
	.DB $4B,$8C,$4B,$8C,$2A,$84,$88,$7B,$68,$7B,$EA,$93,$ED,$AC,$8C,$A4
	.DB $C9,$83,$0B,$94,$50,$BD,$2B,$94,$AD,$A4,$0B,$94,$30,$B5,$0F,$B5
	.DB $C8,$7B,$88,$73,$E5,$5A,$06,$6B,$27,$73,$4B,$9C,$AC,$AC,$C9,$8B
	.DB $47,$7B,$88,$83,$AD,$AC,$0B,$94,$EA,$93,$2B,$9C,$8D,$A4,$8D,$A4
	.DB $EE,$A4,$8D,$94,$8D,$9C,$EB,$8B,$08,$73,$4D,$9C,$AE,$AC,$88,$83
	.DB $AD,$A4,$27,$73,$2F,$B5,$4C,$9C,$28,$73,$28,$73,$A7,$6A,$AF,$A4
	.DB $8D,$94,$CA,$7B,$EF,$A4,$2C,$94,$E7,$6A,$CB,$8B,$CF,$AC,$88,$83
	.DB $4B,$9C,$06,$6B,$EB,$8B,$48,$73,$A6,$62,$08,$73,$49,$7B,$10,$B5
	.DB $6D,$94,$2C,$8C,$14,$CE,$4D,$94,$A7,$62,$69,$7B,$AE,$AC,$2B,$94
	.DB $6C,$9C,$A9,$83,$C9,$83,$89,$7B,$49,$7B,$AA,$83,$0C,$94,$51,$BD
	.DB $8A,$7B,$AB,$7B,$72,$BD,$AA,$7B,$49,$73,$EB,$8B,$AE,$A4,$EA,$8B
	.DB $E9,$8B,$E9,$83,$A8,$7B,$EA,$8B,$EA,$8B,$8A,$7B,$AA,$83,$72,$BD
	.DB $EC,$83,$AA,$7B,$F0,$AC,$AA,$83,$2C,$94,$2C,$94,$2B,$94,$2A,$8C
	.DB $2A,$94,$8B,$9C,$2A,$94,$4B,$94,$2C,$94,$AA,$83,$0C,$8C,$51,$BD
	.DB $6E,$9C,$8E,$9C,$B2,$C5,$8D,$9C,$0B,$8C,$89,$7B,$E9,$83,$E8,$83
	.DB $2A,$8C,$2A,$8C,$C9,$83,$47,$73,$27,$6B,$07,$6B,$0C,$8C,$F0,$AC
	.DB $0C,$8C,$CF,$A4,$71,$B5,$6C,$9C,$88,$7B,$46,$6B,$29,$8C,$8A,$94
	.DB $CC,$A4,$29,$8C,$4B,$94,$67,$73,$27,$6B,$28,$6B,$2C,$94,$AF,$A4
	.DB $11,$B5,$10,$AD,$8D,$9C,$0A,$8C,$67,$73,$45,$6B,$E8,$83,$8B,$94
	.DB $AB,$9C,$87,$73,$6B,$94,$A8,$7B,$68,$73,$C6,$62,$48,$73,$CF,$A4
	.DB $EE,$AC,$09,$8C,$6B,$9C,$0D,$AD,$09,$8C,$66,$73,$6B,$94,$4A,$8C
	.DB $09,$84,$E9,$83,$4B,$94,$89,$7B,$A6,$5A,$4A,$73,$26,$52,$2E,$94
	.DB $89,$7B,$47,$73,$C9,$83,$E9,$8B,$A4,$5A,$2A,$94,$09,$8C,$67,$73
	.DB $68,$73,$47,$73,$EA,$83,$48,$73,$8A,$7B,$2D,$94,$4A,$73,$8F,$9C
	.DB $68,$73,$E6,$62,$A9,$7B,$C9,$83,$06,$6B,$EE,$A4,$EA,$83,$C9,$83
	.DB $4C,$9C,$0B,$8C,$4C,$9C,$28,$73,$2D,$94,$AF,$A4,$4E,$9C,$87,$62
	.DB $AE,$9C,$07,$6B,$89,$7B,$2C,$8C,$CE,$A4,$CE,$A4,$4C,$94,$4C,$9C
	.DB $CE,$AC,$0C,$94,$0C,$94,$E8,$6A,$B0,$AC,$4E,$9C,$0D,$94,$2A,$7B
	.DB $CF,$A4,$28,$6B,$69,$73,$8E,$9C,$31,$B5,$8E,$9C,$AF,$A4,$2D,$94
	.DB $2D,$9C,$6A,$7B,$AB,$8B,$09,$73,$93,$C5,$4E,$9C,$2E,$94,$53,$BD
	.DB $28,$73,$8A,$7B,$2C,$94,$72,$BD,$2D,$94,$AF,$A4,$AF,$A4,$EC,$8B
	.DB $2E,$94,$2E,$94,$0D,$94,$C8,$6A,$B0,$A4,$6B,$83,$8F,$A4,$ED,$8B
	.DB $28,$73,$0C,$94,$F0,$AC,$F4,$CD,$08,$73,$2D,$9C,$4E,$9C,$AC,$83
	.DB $0E,$8C,$D0,$A4,$2E,$94,$26,$52,$09,$6B,$05,$4A,$4E,$94,$09,$6B
	.DB $10,$B5,$4D,$9C,$AF,$AC,$51,$BD,$05,$52,$09,$7B,$ED,$93,$4F,$94
	.DB $ED,$8B,$90,$9C,$0E,$8C,$C8,$62,$6B,$7B,$05,$4A,$4E,$94,$8F,$9C
	.DB $92,$C5,$8F,$A4,$F0,$AC,$76,$DE,$92,$C5,$F4,$D5,$F4,$CD,$6B,$7B
	.DB $07,$4A,$D1,$A4,$74,$BD,$93,$BD,$32,$B5,$4E,$94,$CB,$83,$EF,$A4
	.DB $31,$B5,$B3,$C5,$11,$B5,$35,$D6,$35,$D6,$B2,$C5,$14,$D6,$8C,$83
	.DB $8C,$83,$D1,$A4,$11,$AD,$32,$B5,$52,$B5,$0C,$8C,$EC,$8B,$30,$AD
	.DB $10,$B5,$F4,$CD,$51,$BD,$D3,$CD,$14,$D6,$B3,$C5,$D4,$CD,$C9,$6A
	.DB $B0,$A4,$6F,$9C,$11,$AD,$93,$BD,$D4,$C5,$EC,$8B,$0C,$8C,$AE,$A4
	.DB $10,$AD,$31,$B5,$92,$BD,$D4,$CD,$52,$BD,$35,$D6,$D4,$C5,$4A,$7B
	.DB $93,$BD,$0D,$94,$31,$B5,$72,$BD,$D4,$CD,$CB,$83,$8A,$7B,$EC,$8B
	.DB $4E,$9C,$CF,$A4,$B4,$C5,$15,$D6,$72,$BD,$15,$D6,$D4,$CD,$4E,$9C
	.DB $72,$BD,$8A,$7B,$D0,$AC,$CF,$AC,$D3,$C5,$AE,$A4,$2C,$94,$6D,$9C
	.DB $29,$73,$31,$B5,$93,$BD,$35,$D6,$35,$D6,$52,$B5,$B4,$C5,$F0,$AC
	.DB $52,$BD,$AC,$83,$6E,$9C,$AF,$A4,$B3,$C5,$30,$B5,$2D,$94,$51,$BD
	.DB $8A,$7B,$93,$BD,$72,$BD,$15,$CE,$77,$DE,$52,$B5,$D4,$C5,$73,$BD
	.DB $F5,$CD,$B0,$A4,$AC,$83,$6E,$9C,$10,$B5,$8E,$9C,$69,$7B,$50,$BD
	.DB $D0,$A4,$B3,$BD,$93,$BD,$36,$D6,$56,$D6,$56,$D6,$15,$CE,$F1,$AC
	.DB $D5,$CD,$90,$A4,$26,$5A,$ED,$93,$F0,$B4,$F0,$AC,$6D,$9C,$CE,$AC
	.DB $32,$B5,$11,$AD,$93,$BD,$56,$D6,$F3,$C5,$18,$EF,$D3,$C5,$11,$B5
	.DB $73,$C5,$0D,$94,$05,$52,$AB,$83,$CF,$AC,$0F,$AD,$0B,$94,$28,$73
	.DB $F1,$AC,$D0,$A4,$B3,$C5,$55,$D6,$D3,$C5,$34,$D6,$52,$BD,$D4,$CD
	.DB $32,$B5,$67,$62,$05,$52,$2D,$94,$10,$B5,$2C,$94,$A6,$62,$45,$5A
	.DB $4F,$9C,$73,$BD,$B4,$CD,$D4,$CD,$97,$E6,$55,$DE,$52,$BD,$11,$AD
	.DB $53,$BD,$A7,$62,$6A,$7B,$0C,$8C,$EB,$8B,$E7,$6A,$A6,$62,$65,$5A
	.DB $F1,$AC,$15,$D6,$D4,$CD,$52,$BD,$35,$DE,$10,$B5,$2D,$94,$72,$B5
	.DB $15,$CE,$E8,$62,$8E,$9C,$4D,$94,$0C,$8C,$08,$6B,$E7,$6A,$08,$6B
	.DB $90,$A4,$F1,$B4,$93,$C5,$72,$BD,$73,$C5,$AB,$83,$AB,$83,$15,$CE
	.DB $56,$D6,$09,$6B,$8F,$9C,$6E,$9C,$CF,$AC,$4D,$9C,$EC,$93,$EC,$8B
	.DB $6B,$83,$6F,$9C,$73,$C5,$D4,$CD,$B4,$C5,$4E,$9C,$6F,$9C,$F4,$CD
	.DB $51,$B5,$A7,$62,$8B,$83,$EC,$8B,$31,$BD,$93,$CD,$D0,$B4,$0D,$9C
	.DB $EC,$8B,$73,$BD,$F4,$CD,$15,$CE,$93,$BD,$32,$B5,$11,$AD,$56,$D6
	.DB $2D,$94,$66,$5A,$C7,$6A,$AB,$8B,$11,$BD,$72,$CD,$6B,$8B,$09,$83
	.DB $6E,$9C,$72,$B5,$F5,$CD,$56,$D6,$52,$B5,$93,$BD,$F4,$CD,$55,$D6
	.DB $AF,$A4,$8F,$A4,$CC,$8B,$09,$7B,$E9,$7A,$ED,$9B,$A8,$7A,$A8,$7A
	.DB $C7,$62,$07,$6B,$45,$52,$AA,$7B,$0B,$8C,$8E,$9C,$6E,$9C,$67,$5A
	.DB $F2,$B4,$90,$A4,$68,$5A,$88,$62,$2A,$73,$E9,$6A,$4A,$73,$EC,$83
	.DB $E7,$62,$AA,$83,$49,$73,$28,$6B,$A6,$5A,$8A,$7B,$AB,$83,$25,$52
	.DB $F1,$AC,$4F,$9C,$47,$5A,$C8,$62,$09,$73,$A7,$62,$4A,$73,$EC,$8B
	.DB $86,$5A,$CB,$83,$8A,$7B,$86,$5A,$04,$4A,$4A,$73,$29,$73,$E8,$6A
	.DB $F0,$AC,$ED,$8B,$25,$52,$E8,$6A,$29,$73,$86,$5A,$28,$73,$CE,$A4
	.DB $86,$5A,$08,$6B,$A7,$62,$46,$5A,$25,$52,$4A,$7B,$08,$6B,$2C,$94
	.DB $8E,$9C,$8A,$7B,$86,$5A,$28,$6B,$CB,$83,$AA,$7B,$89,$7B,$C6,$62
	.DB $69,$7B,$E7,$6A,$E8,$6A,$09,$73,$46,$5A,$87,$62,$E8,$6A,$08,$6B
	.DB $64,$52,$24,$4A,$03,$4A,$03,$4A,$E6,$62,$89,$7B,$E7,$62,$C6,$62
	.DB $4D,$9C,$69,$7B,$0C,$8C,$2D,$94,$86,$62,$66,$5A,$AA,$83,$A9,$7B
	.DB $C5,$62,$07,$6B,$48,$73,$85,$5A,$E7,$62,$CA,$7B,$89,$73,$CA,$7B
	.DB $6D,$9C,$49,$7B,$EB,$8B,$0B,$94,$28,$73,$89,$7B,$0B,$8C,$8D,$9C
	.DB $C9,$83,$68,$73,$68,$73,$E6,$62,$A5,$5A,$E7,$62,$28,$6B,$E3,$41
	.DB $EB,$8B,$A5,$62,$C6,$6A,$C6,$6A,$AA,$83,$8D,$A4,$89,$7B,$2F,$B5
	.DB $4B,$8C,$C6,$62,$65,$52,$E7,$62,$85,$52,$24,$4A,$E7,$62,$65,$52
	.DB $A6,$62,$65,$5A,$C7,$6A,$4A,$7B,$2A,$73,$E9,$72,$6F,$9C,$D1,$AC
	.DB $ED,$8B,$26,$52,$86,$52,$A6,$52,$48,$63,$47,$63,$05,$5B,$C4,$52
	.DB $A6,$62,$A6,$62,$04,$4A,$66,$5A,$09,$73,$4A,$7B,$4F,$9C,$52,$BD
	.DB $8B,$83,$25,$4A,$C7,$5A,$CA,$7B,$C9,$73,$0A,$7C,$09,$7C,$87,$6B
	.DB $E7,$6A,$E7,$6A,$66,$5A,$4A,$73,$6F,$9C,$8F,$9C,$F1,$AC,$11,$B5
	.DB $8A,$83,$A6,$5A,$A6,$5A,$AA,$7B,$E6,$62,$67,$6B,$26,$63,$26,$63
	.DB $C7,$62,$86,$5A,$E8,$62,$6E,$94,$31,$AD,$6E,$94,$2D,$94,$4D,$94
	.DB $2C,$94,$CA,$83,$49,$7B,$68,$73,$C6,$62,$26,$63,$A4,$52,$43,$4A
	.DB $49,$7B,$E8,$6A,$09,$73,$6E,$9C,$AF,$A4,$49,$6B,$A7,$5A,$86,$5A
	.DB $08,$73,$C6,$62,$07,$6B,$85,$5A,$A5,$5A,$C5,$5A,$C5,$5A,$C5,$5A
	.DB $0D,$9C,$CB,$8B,$49,$7B,$EC,$8B,$0C,$94,$E8,$62,$66,$5A,$E7,$62
	.DB $C7,$62,$24,$4A,$07,$6B,$A5,$5A,$C6,$5A,$06,$63,$88,$73,$68,$73
	.DB $09,$7B,$09,$7B,$E8,$72,$6A,$7B,$29,$73,$66,$5A,$C7,$62,$E8,$62
	.DB $A7,$5A,$24,$4A,$86,$52,$65,$52,$24,$4A,$C5,$5A,$47,$6B,$68,$73
	.DB $E5,$59,$05,$62,$E9,$7A,$8B,$83,$A7,$62,$C3,$49,$08,$6B,$86,$52
	.DB $29,$63,$69,$6B,$C7,$5A,$C6,$5A,$85,$52,$A9,$7B,$A9,$7B,$68,$6B
	.DB $C7,$62,$85,$5A,$88,$73,$26,$6B,$87,$73,$A7,$7B,$C8,$7B,$E5,$5A
	.DB $68,$6B,$48,$6B,$C6,$5A,$07,$63,$0A,$84,$0B,$84,$48,$6B,$68,$6B
	.DB $49,$73,$07,$6B,$47,$73,$26,$6B,$C8,$83,$4A,$8C,$4A,$8C,$E5,$5A
	.DB $27,$63,$27,$6B,$47,$6B,$68,$73,$A9,$7B,$68,$6B,$C6,$5A,$27,$63
	.DB $4C,$94,$C9,$83,$E5,$62,$05,$63,$C8,$7B,$8A,$94,$29,$8C,$A8,$73
	.DB $67,$6B,$67,$73,$C9,$7B,$C9,$7B,$48,$73,$C6,$62,$85,$5A,$64,$52
	.DB $C6,$62,$47,$6B,$E5,$62,$A7,$73,$E8,$7B,$09,$84,$46,$6B,$09,$84
	.DB $A8,$7B,$67,$73,$A8,$7B,$88,$7B,$27,$6B,$E6,$62,$C6,$62,$48,$73
	.DB $85,$52,$A5,$5A,$84,$52,$A8,$73,$88,$73,$C8,$7B,$46,$6B,$87,$73
	.DB $67,$73,$27,$6B,$06,$6B,$E6,$62,$E6,$62,$C6,$62,$C6,$62,$CA,$83
	.DB $48,$6B,$E6,$5A,$C5,$5A,$88,$73,$47,$6B,$46,$6B,$67,$6B,$26,$6B
	.DB $68,$73,$47,$6B,$C5,$5A,$64,$52,$65,$52,$64,$52,$24,$4A,$E7,$62
	.DB $85,$52,$A6,$5A,$69,$73,$EA,$83,$A9,$7B,$E5,$5A,$E6,$62,$27,$63
	.DB $48,$6B,$48,$6B,$C6,$5A,$44,$4A,$24,$4A,$45,$4A,$65,$52,$07,$63
	.DB $28,$6B,$07,$63,$68,$6B,$68,$6B,$A9,$7B,$27,$6B,$A9,$7B,$C6,$5A
	.DB $C6,$5A,$C6,$5A,$A6,$52,$44,$4A,$44,$4A,$C6,$5A,$69,$73,$28,$63
	.DB $09,$74,$67,$63,$05,$5B,$26,$6B,$68,$73,$C9,$83,$E7,$62,$A7,$5A
	.DB $46,$52,$E7,$62,$27,$6B,$64,$4A,$C5,$5A,$E9,$7B,$2B,$8C,$CA,$7B
	.DB $09,$7C,$C8,$73,$88,$73,$88,$73,$28,$73,$E6,$62,$45,$52,$87,$5A
	.DB $87,$52,$C7,$5A,$E7,$62,$64,$52,$E6,$5A,$A9,$73,$CA,$7B,$EA,$7B
	.DB $47,$6B,$A9,$7B,$A9,$7B,$68,$73,$A6,$62,$04,$52,$67,$5A,$C8,$62
	.DB $C8,$5A,$86,$52,$24,$4A,$44,$4A,$A5,$5A,$27,$63,$27,$6B,$4C,$8C
	.DB $A5,$5A,$A9,$7B,$69,$73,$E7,$62,$66,$5A,$E5,$49,$0A,$73,$47,$52
	.DB $67,$52,$E4,$41,$A2,$39,$64,$52,$68,$6B,$CA,$7B,$EB,$83,$48,$6B
	.DB $07,$63,$89,$7B,$08,$6B,$A7,$62,$A8,$62,$46,$5A,$0A,$73,$A8,$5A
	.DB $A8,$5A,$45,$4A,$24,$4A,$07,$63,$0A,$84,$2B,$8C,$0B,$84,$C6,$5A
	.DB $48,$73,$07,$63,$86,$52,$C7,$62,$29,$73,$A8,$62,$A8,$62,$E8,$62
	.DB $C7,$5A,$C6,$5A,$E7,$62,$88,$73,$0A,$84,$0A,$84,$69,$73,$07,$63
	.DB $68,$73,$85,$52,$85,$52,$08,$63,$09,$6B,$08,$63,$A7,$5A,$86,$52
	.DB $24,$4A,$A5,$52,$47,$6B,$A8,$73,$2A,$84,$4B,$8C,$A9,$73,$A7,$5A
	.DB $89,$73,$C6,$5A,$E7,$62,$28,$63,$A7,$52,$E8,$5A,$A7,$5A,$89,$73
	.DB $C6,$5A,$06,$63,$88,$73,$67,$6B,$88,$73,$C9,$7B,$27,$6B,$08,$63
	.DB $EA,$7B,$47,$63,$C5,$52,$84,$42,$64,$42,$67,$5B,$0A,$74,$E5,$52
	.DB $06,$5B,$C6,$52,$24,$42,$65,$4A,$89,$6B,$EA,$7B,$47,$63,$E9,$73
	.DB $CA,$7B,$89,$73,$68,$63,$27,$5B,$C6,$52,$E6,$52,$E6,$52,$E6,$52
	.DB $A5,$52,$A6,$52,$A6,$52,$86,$4A,$E7,$5A,$28,$63,$E6,$5A,$67,$6B
	.DB $E7,$62,$C6,$5A,$08,$63,$48,$63,$C7,$52,$C6,$52,$86,$4A,$64,$42
	.DB $08,$5B,$48,$63,$24,$42,$61,$29,$24,$42,$07,$5B,$06,$5B,$83,$4A
	.DB $08,$6B,$65,$52,$86,$52,$A6,$52,$04,$42,$45,$42,$A7,$52,$69,$6B
	.DB $07,$5B,$24,$42,$82,$29,$E4,$39,$08,$5B,$28,$63,$03,$3A,$E5,$5A
	.DB $28,$6B,$65,$52,$C7,$5A,$C7,$5A,$04,$42,$A7,$52,$AB,$73,$28,$63
	.DB $86,$4A,$A6,$52,$08,$63,$A7,$52,$25,$42,$65,$4A,$C6,$52,$63,$4A
	.DB $C7,$5A,$45,$4A,$C7,$5A,$08,$5B,$65,$4A,$08,$5B,$EB,$7B,$68,$63
	.DB $85,$4A,$65,$4A,$C7,$52,$C7,$52,$A7,$52,$A6,$52,$64,$4A,$63,$42
	.DB $49,$6B,$C7,$5A,$A6,$52,$A6,$4A,$45,$42,$A6,$4A,$C6,$52,$64,$42
	.DB $27,$63,$68,$6B,$A6,$52,$04,$3A,$C7,$52,$89,$6B,$48,$63,$C9,$73
	.DB $EA,$7B,$28,$63,$65,$42,$65,$42,$C6,$52,$48,$5B,$E6,$52,$A4,$4A
	.DB $E5,$52,$47,$63,$27,$63,$28,$63,$CB,$73,$0B,$7C,$A9,$73,$E6,$52
	.DB $CA,$7B,$65,$4A,$A6,$52,$24,$42,$A6,$52,$27,$63,$A5,$52,$85,$52
	.DB $C2,$41,$44,$52,$27,$6B,$88,$7B,$0A,$84,$C9,$7B,$64,$52,$E2,$41
	.DB $69,$6B,$C6,$5A,$08,$5B,$07,$5B,$89,$6B,$EA,$7B,$89,$73,$A5,$52
	.DB $44,$4A,$C6,$5A,$88,$6B,$A9,$73,$0A,$84,$E9,$7B,$C5,$52,$E5,$5A
	.DB $48,$6B,$28,$63,$08,$5B,$28,$63,$07,$63,$07,$63,$06,$5B,$C5,$52
	.DB $85,$4A,$E6,$52,$27,$5B,$47,$63,$C9,$73,$E9,$73,$05,$53,$05,$53
	.DB $04,$42,$A6,$52,$86,$52,$E7,$5A,$A6,$52,$A5,$52,$07,$63,$E6,$5A
	.DB $C6,$52,$E6,$52,$C5,$52,$E6,$5A,$A8,$6B,$E9,$7B,$67,$63,$E5,$52
	.DB $82,$31,$66,$4A,$46,$4A,$A6,$52,$65,$4A,$86,$52,$48,$6B,$07,$63
	.DB $07,$6B,$07,$6B,$07,$63,$48,$6B,$EA,$83,$0A,$84,$A8,$7B,$06,$63
	.DB $E4,$41,$25,$42,$26,$42,$05,$42,$A3,$31,$E4,$39,$86,$52,$85,$52
	.DB $C6,$62,$27,$6B,$69,$7B,$CA,$83,$CA,$83,$88,$7B,$27,$6B,$A5,$5A
	.DB $E4,$41,$42,$29,$C5,$39,$84,$31,$84,$31,$26,$4A,$A7,$52,$E3,$39
	.DB $85,$52,$27,$6B,$CA,$7B,$0A,$84,$88,$73,$E5,$5A,$C5,$5A,$A4,$52
	.DB $66,$4A,$02,$21,$E6,$39,$85,$31,$C6,$39,$88,$52,$A8,$52,$03,$3A
	.DB $A5,$52,$88,$6B,$2B,$7C,$2B,$7C,$67,$63,$C5,$52,$06,$5B,$C8,$73
	.DB $C8,$73,$C7,$73,$C7,$73,$C7,$73,$C8,$73,$2A,$7C,$6A,$8C,$AA,$94
	.DB $E7,$83,$2A,$8C,$EB,$83,$05,$42,$47,$4A,$26,$4A,$A7,$5A,$48,$6B
	.DB $66,$63,$A7,$6B,$29,$7C,$8A,$8C,$6A,$84,$2A,$84,$29,$7C,$6A,$8C
	.DB $87,$73,$88,$73,$A7,$52,$42,$29,$C5,$39,$46,$4A,$A7,$5A,$26,$63
	.DB $86,$6B,$69,$84,$6A,$84,$C7,$73,$66,$63,$C7,$73,$4A,$84,$27,$63
	.DB $44,$42,$45,$42,$C3,$31,$C4,$39,$C7,$52,$8A,$73,$A9,$73,$26,$63
	.DB $E8,$73,$8A,$8C,$69,$84,$29,$7C,$29,$7C,$09,$7C,$A8,$6B,$07,$5B
	.DB $04,$32,$45,$42,$86,$4A,$AA,$73,$0B,$7C,$4B,$84,$A8,$6B,$04,$5B
	.DB $A8,$73,$4A,$84,$E8,$73,$E8,$73,$6A,$84,$09,$7C,$06,$5B,$A6,$4A
	.DB $24,$3A,$E7,$52,$EA,$73,$0F,$9D,$8C,$8C,$8B,$84,$E8,$73,$A3,$4A
	.DB $87,$6B,$E8,$73,$05,$5B,$A4,$52,$26,$63,$67,$63,$06,$5B,$44,$42
	.DB $E6,$52,$A9,$6B,$8C,$8C,$CC,$94,$09,$7C,$49,$7C,$4A,$84,$6A,$84
	.DB $88,$6B,$46,$63,$26,$5B,$67,$63,$C9,$73,$68,$6B,$E6,$5A,$46,$63
	.DB $8B,$8C,$AA,$8C,$CB,$94,$08,$7C,$86,$6B,$C7,$6B,$C8,$73,$A0,$29
	.DB $06,$5B,$A8,$6B,$4B,$84,$2B,$84,$89,$6B,$28,$63,$27,$63,$A7,$6B
	.DB $4C,$A5,$CA,$94,$A9,$8C,$E7,$73,$48,$7C,$6A,$84,$09,$7C,$47,$63
	.DB $68,$73,$0A,$8C,$2B,$8C,$48,$73,$89,$7B,$E7,$62,$A5,$5A,$A7,$73
	.DB $CB,$9C,$AA,$94,$C7,$7B,$E4,$5A,$E8,$7B,$4A,$84,$4B,$84,$C8,$73
	.DB $88,$73,$EA,$7B,$C9,$73,$E6,$5A,$85,$52,$64,$4A,$84,$52,$C8,$7B
	.DB $49,$8C,$E8,$7B,$A7,$73,$87,$73,$8C,$94,$4A,$84,$A8,$73,$88,$6B
	.DB $26,$5B,$A8,$6B,$88,$6B,$27,$5B,$64,$42,$84,$4A,$E6,$52,$C9,$7B
	.DB $6B,$8C,$4B,$8C,$6B,$8C,$C9,$7B,$E9,$7B,$E5,$5A,$22,$42,$C8,$73
	.DB $C5,$52,$67,$63,$47,$63,$88,$6B,$E6,$5A,$68,$63,$89,$73,$AD,$94
	.DB $AD,$94,$0A,$7C,$68,$6B,$84,$4A,$C5,$52,$84,$4A,$84,$52,$87,$6B
	.DB $06,$63,$48,$73,$E6,$62,$68,$73,$68,$73,$CA,$7B,$89,$73,$EA,$7B
	.DB $EA,$7B,$67,$6B,$47,$63,$E6,$5A,$68,$6B,$47,$63,$47,$63,$87,$6B
	.DB $27,$6B,$88,$7B,$27,$6B,$89,$7B,$AA,$83,$89,$7B,$C6,$5A,$E5,$5A
	.DB $E9,$7B,$8B,$8C,$ED,$94,$6B,$84,$6B,$84,$A8,$6B,$47,$63,$A8,$73
	.DB $67,$6B,$88,$73,$88,$73,$68,$6B,$CA,$7B,$68,$6B,$C5,$5A,$29,$7C
	.DB $0C,$95,$EC,$94,$CB,$94,$E8,$73,$09,$7C,$A8,$6B,$87,$6B,$C5,$52
	.DB $0A,$7C,$88,$6B,$47,$63,$A5,$4A,$68,$63,$68,$63,$88,$6B,$CB,$8C
	.DB $EA,$94,$27,$7C,$E7,$73,$86,$6B,$29,$7C,$67,$63,$A4,$4A,$A0,$31
	.DB $25,$63,$07,$7C,$68,$84,$CA,$8C,$49,$7C,$09,$7C,$46,$63,$A5,$73
	.DB $84,$73,$85,$73,$05,$63,$EA,$83,$A7,$5A,$6A,$73,$87,$5A,$04,$42
	.DB $A7,$6B,$2C,$9D,$0B,$9D,$49,$84,$66,$63,$06,$5B,$C1,$31,$86,$6B
	.DB $45,$5B,$83,$4A,$43,$42,$A6,$4A,$45,$42,$86,$4A,$E4,$39,$E4,$31
	.DB $08,$7C,$E7,$7B,$65,$6B,$8B,$8C,$EA,$7B,$C6,$5A,$45,$42,$27,$53
	.DB $48,$53,$85,$3A,$85,$3A,$44,$32,$A6,$42,$E7,$4A,$C6,$4A,$E2,$29
	.DB $86,$6B,$AB,$94,$4A,$84,$67,$6B,$04,$42,$A3,$39,$C0,$18,$A6,$42
	.DB $49,$53,$69,$5B,$88,$5B,$67,$5B,$09,$6C,$6B,$7C,$AC,$84,$2E,$95
	.DB $08,$7C,$4A,$84,$A9,$73,$EB,$7B,$C3,$39,$E1,$20,$05,$42,$45,$42
	.DB $E6,$52,$EA,$73,$2A,$7C,$CB,$8C,$0C,$95,$2D,$9D,$0D,$9D,$0D,$9D
	.DB $67,$63,$EA,$7B,$CA,$73,$08,$5B,$C3,$31,$26,$42,$66,$4A,$C6,$5A
	.DB $E5,$5A,$E8,$7B,$69,$84,$2C,$A5,$2C,$A5,$6A,$8C,$E9,$7B,$27,$63
	.DB $44,$42,$48,$63,$28,$5B,$E7,$52,$A3,$31,$66,$42,$AA,$6B,$C9,$7B
	.DB $87,$73,$28,$84,$AA,$94,$AA,$94,$8A,$94,$47,$63,$28,$63,$C9,$5A
	.DB $85,$4A,$E7,$52,$86,$42,$04,$32,$C3,$31,$E7,$52,$E6,$52,$6B,$8C
	.DB $6A,$8C,$8A,$8C,$2C,$A5,$E7,$73,$E9,$7B,$85,$4A,$4B,$63,$CB,$5A
	.DB $46,$63,$84,$4A,$27,$5B,$47,$5B,$C9,$6B,$A8,$6B,$2A,$7C,$8C,$94
	.DB $6B,$94,$09,$84,$46,$6B,$A8,$73,$C9,$7B,$6C,$8C,$48,$63,$45,$4A
	.DB $46,$63,$05,$53,$67,$63,$87,$63,$E9,$73,$29,$74,$8B,$84,$A7,$63
	.DB $25,$53,$04,$53,$46,$5B,$4A,$84,$EA,$73,$68,$6B,$A2,$31,$A3,$31
	.DB $C8,$73,$09,$74,$E8,$73,$E8,$73,$C7,$6B,$07,$74,$08,$74,$28,$6C
	.DB $07,$64,$E7,$63,$C7,$63,$46,$5B,$84,$42,$C6,$52,$65,$42,$62,$29
	.DB $E8,$73,$AB,$8C,$6A,$84,$CA,$8C,$69,$84,$89,$84,$48,$7C,$A9,$84
	.DB $69,$7C,$E8,$6B,$A8,$63,$47,$5B,$E7,$52,$AA,$6B,$69,$63,$E3,$39
	.DB $A8,$6B,$4A,$84,$08,$7C,$69,$84,$07,$74,$07,$74,$A5,$6B,$65,$6B
	.DB $C8,$7B,$C9,$7B,$CA,$7B,$29,$63,$C8,$5A,$C8,$52,$E5,$39,$82,$29
	.DB $A8,$6B,$C8,$73,$A7,$6B,$A7,$6B,$25,$5B,$04,$5B,$A3,$52,$E4,$62
	.DB $67,$73,$07,$6B,$C8,$62,$A4,$39,$64,$31,$63,$29,$C0,$10,$A2,$29
	.DB $E5,$5A,$46,$63,$E9,$7B,$C8,$73,$87,$6B,$26,$63,$26,$63,$67,$6B
	.DB $C6,$5A,$82,$31,$84,$31,$23,$21,$86,$31,$C6,$31,$63,$29,$28,$5B
	.DB $22,$3A,$84,$52,$C9,$73,$47,$63,$E6,$5A,$22,$42,$43,$42,$05,$53
	.DB $27,$53,$A6,$42,$E9,$4A,$44,$19,$C3,$10,$65,$21,$26,$3A,$EB,$73
	.DB $A6,$52,$2B,$84,$27,$63,$05,$5B,$05,$5B,$6B,$8C,$88,$6B,$EB,$73
	.DB $29,$53,$69,$5B,$45,$3A,$69,$63,$CE,$94,$EE,$94,$2B,$7C,$68,$6B
	.DB $20,$19,$68,$63,$A9,$6B,$2E,$9D,$ED,$94,$ED,$8C,$88,$63,$C5,$4A
	.DB $47,$5B,$2A,$74,$E9,$73,$C9,$6B,$68,$63,$07,$5B,$07,$5B,$2C,$84
	.DB $C2,$21,$C9,$6B,$4B,$7C,$EE,$8C,$A8,$63,$8C,$84,$6B,$7C,$EB,$8C
	.DB $8E,$9D,$CB,$8C,$49,$7C,$67,$63,$C4,$4A,$C5,$52,$69,$6B,$AB,$73
	.DB $ED,$8C,$2A,$7C,$CD,$8C,$0E,$95,$E9,$73,$2D,$95,$4A,$7C,$6A,$84
	.DB $0C,$95,$46,$5B,$67,$63,$47,$63,$E6,$52,$85,$4A,$A7,$52,$26,$42
	.DB $6B,$84,$C6,$5A,$E6,$5A,$06,$5B,$E9,$7B,$87,$73,$83,$52,$CA,$73
	.DB $0B,$7C,$85,$4A,$49,$63,$69,$63,$86,$4A,$C4,$31,$26,$42,$67,$4A
	.DB $65,$52,$66,$52,$61,$31,$40,$29,$A9,$7B,$C4,$5A,$88,$73,$28,$63
	.DB $E8,$5A,$46,$4A,$09,$5B,$A8,$52,$42,$21,$43,$21,$C9,$52,$2A,$63
	.DB $65,$31,$04,$21,$A2,$18,$E5,$39,$6A,$6B,$47,$63,$E9,$7B,$45,$4A
	.DB $46,$4A,$6A,$6B,$ED,$7B,$ED,$7B,$2A,$63,$0A,$5B,$CD,$73,$67,$42
	.DB $0A,$42,$E6,$18,$C9,$31,$8E,$6B,$47,$42,$EB,$73,$89,$63,$85,$4A
	.DB $A6,$52,$EB,$7B,$29,$63,$CB,$73,$0D,$84,$8B,$6B,$87,$4A,$05,$3A
	.DB $86,$4A,$83,$29,$66,$42,$AA,$6B,$4C,$84,$CD,$8C,$47,$63,$E6,$5A
	.DB $07,$63,$EB,$83,$CC,$7B,$8C,$73,$C9,$5A,$64,$29,$A5,$31,$C1,$10
	.DB $87,$52,$87,$52,$8B,$73,$EC,$83,$6A,$73,$69,$73,$23,$4A,$44,$52
	.DB $85,$5A,$8A,$73,$8A,$73,$88,$52,$84,$31,$E6,$39,$64,$29,$E5,$39
	.DB $87,$52,$E8,$62,$4A,$73,$C8,$62,$A3,$41,$A3,$41,$21,$31,$A1,$39
	.DB $EB,$83,$69,$6B,$E4,$39,$A4,$31,$42,$21,$05,$3A,$A4,$31,$69,$63
	.DB $A7,$52,$66,$4A,$04,$42,$82,$31,$42,$31,$05,$4A,$05,$4A,$E7,$5A
	.DB $27,$63,$04,$3A,$04,$42,$E4,$39,$42,$21,$04,$32,$AA,$6B,$CA,$6B
	.DB $24,$3A,$E2,$31,$61,$21,$A2,$21,$85,$42,$29,$53,$C7,$4A,$A6,$52
	.DB $E3,$39,$04,$3A,$04,$3A,$80,$08,$6A,$63,$4D,$7C,$48,$5B,$A5,$4A
	.DB $23,$32,$64,$3A,$84,$3A,$06,$4B,$89,$5B,$08,$4B,$E4,$29,$65,$42
	.DB $41,$21,$C3,$31,$46,$42,$A3,$29,$EC,$73,$AA,$6B,$07,$53,$85,$42
	.DB $27,$5B,$67,$63,$A8,$63,$C9,$6B,$68,$5B,$45,$3A,$82,$21,$21,$19
	.DB $09,$5B,$E8,$52,$83,$29,$C8,$4A,$EC,$73,$29,$53,$25,$3A,$8A,$63
	.DB $27,$5B,$05,$5B,$05,$5B,$C5,$52,$03,$42,$41,$29,$42,$29,$83,$29
	.DB $E8,$52,$2A,$5B,$87,$4A,$CC,$6B,$C8,$4A,$29,$5B,$66,$42,$49,$63
	.DB $EE,$9C,$0A,$84,$CA,$83,$0B,$84,$28,$6B,$08,$6B,$65,$52,$AA,$73
	.DB $A9,$6B,$67,$63,$C8,$73,$45,$63,$65,$63,$45,$63,$A8,$6B,$A5,$52
	.DB $8E,$94,$08,$63,$C8,$5A,$88,$5A,$23,$31,$03,$29,$02,$29,$26,$42
	.DB $8A,$6B,$2C,$7C,$2B,$7C,$47,$63,$C8,$73,$A9,$6B,$89,$6B,$08,$5B
	.DB $AC,$7B,$47,$4A,$07,$4A,$C8,$41,$A4,$20,$A4,$20,$C4,$20,$C1,$18
	.DB $87,$4A,$EC,$73,$CF,$94,$4C,$7C,$2C,$7C,$08,$5B,$05,$3A,$83,$29
	.DB $47,$4A,$85,$31,$86,$31,$66,$31,$A4,$18,$E5,$20,$25,$29,$2B,$63
	.DB $4A,$63,$CC,$73,$AF,$8C,$8A,$6B,$C7,$52,$25,$3A,$05,$3A,$46,$42
	.DB $06,$3A,$64,$29,$85,$31,$44,$29,$A1,$10,$24,$21,$27,$42,$2D,$7C
	.DB $CB,$73,$0C,$7C,$6E,$84,$86,$4A,$E3,$31,$45,$42,$28,$5B,$89,$63
	.DB $C5,$31,$22,$19,$A3,$29,$62,$21,$01,$19,$25,$3A,$EB,$73,$8D,$8C
	.DB $6D,$8C,$AE,$94,$AF,$94,$29,$63,$AB,$73,$2C,$7C,$EB,$73,$85,$4A
	.DB $43,$21,$01,$11,$21,$19,$62,$21,$E4,$31,$08,$53,$2D,$7C,$6D,$8C
	.DB $EB,$7B,$29,$63,$67,$52,$84,$31,$2A,$63,$6B,$6B,$04,$3A,$65,$4A
	.DB $E5,$39,$E5,$31,$63,$21,$C4,$29,$4A,$5B,$EC,$73,$8B,$6B,$E7,$5A
	.DB $C7,$5A,$A8,$5A,$68,$52,$85,$31,$0B,$6B,$2B,$6B,$A4,$31,$66,$4A
	.DB $83,$29,$87,$4A,$29,$5B,$25,$42,$AB,$73,$CC,$7B,$C8,$5A,$68,$4A
	.DB $68,$52,$A9,$5A,$07,$42,$2B,$6B,$A9,$5A,$2B,$6B,$E5,$41,$C4,$39
	.DB $A3,$29,$A3,$31,$CB,$73,$69,$6B,$28,$63,$CB,$73,$CB,$7B,$08,$63
	.DB $66,$52,$A4,$39,$0D,$84,$AC,$7B,$09,$63,$83,$31,$43,$29,$02,$21
	.DB $2C,$7C,$27,$5B,$CA,$73,$0B,$7C,$4C,$84,$4C,$84,$A5,$52,$27,$63
	.DB $48,$6B,$C6,$5A,$89,$73,$E8,$5A,$25,$42,$42,$29,$C1,$18,$60,$08
	.DB $0A,$74,$68,$63,$2B,$7C,$2B,$7C,$C6,$52,$24,$42,$24,$42,$C6,$5A
	.DB $49,$6B,$08,$63,$04,$42,$05,$42,$E4,$39,$26,$42,$02,$21,$23,$21
	.DB $A5,$4A,$A8,$6B,$A9,$6B,$27,$5B,$61,$29,$C0,$18,$C8,$5A,$06,$42
	.DB $C5,$39,$03,$21,$E6,$41,$27,$42,$68,$4A,$47,$42,$C6,$39,$27,$3A
	.DB $48,$5B,$EB,$73,$82,$29,$82,$29,$26,$4A,$64,$31,$85,$31,$65,$31
	.DB $E4,$20,$E3,$20,$24,$29,$E7,$39,$65,$29,$06,$42,$06,$3A,$A5,$31
	.DB $A7,$4A,$45,$3A,$80,$10,$03,$21,$45,$29,$C3,$18,$63,$10,$24,$21
	.DB $86,$31,$E7,$39,$A2,$10,$03,$21,$80,$10,$06,$3A,$C5,$39,$02,$21
	.DB $25,$3A,$83,$29,$C1,$18,$04,$21,$42,$10,$A5,$18,$E5,$20,$65,$29
	.DB $27,$42,$64,$29,$06,$3A,$E1,$18,$A4,$31,$84,$31,$C5,$39,$63,$31
	.DB $45,$4A,$A7,$52,$05,$42,$E3,$41,$23,$4A,$88,$73,$88,$73,$27,$5B
	.DB $A5,$42,$26,$53,$E5,$4A,$26,$5B,$66,$5B,$E9,$73,$46,$5B,$C7,$6B
	.DB $A8,$52,$C8,$5A,$C5,$39,$83,$31,$E4,$39,$07,$63,$08,$63,$04,$3A
	.DB $65,$42,$07,$53,$85,$42,$E3,$31,$43,$3A,$68,$63,$A9,$6B,$26,$53
	.DB $06,$3A,$A9,$52,$C5,$39,$84,$29,$A4,$31,$A7,$52,$A7,$52,$67,$4A
	.DB $A8,$52,$A7,$52,$66,$4A,$E4,$39,$E7,$5A,$8A,$6B,$AA,$6B,$48,$5B
	.DB $C8,$52,$6A,$63,$87,$4A,$67,$42,$46,$42,$87,$4A,$25,$42,$67,$52
	.DB $67,$52,$E4,$39,$E4,$39,$E4,$39,$28,$63,$08,$5B,$A6,$4A,$27,$5B
	.DB $69,$63,$48,$63,$A6,$4A,$28,$5B,$28,$5B,$08,$5B,$45,$42,$C7,$5A
	.DB $29,$6B,$A6,$52,$66,$52,$24,$42,$07,$5B,$06,$5B,$06,$5B,$46,$5B
	.DB $48,$63,$0A,$74,$88,$63,$A9,$6B,$27,$5B,$07,$53,$E7,$52,$A7,$52
	.DB $E8,$62,$65,$4A,$E3,$39,$C2,$31,$23,$42,$64,$42,$C5,$52,$67,$63
	.DB $EA,$73,$4F,$A5,$ED,$94,$6C,$84,$C7,$52,$86,$4A,$E8,$5A,$29,$63
	.DB $E9,$5A,$46,$42,$E4,$39,$86,$4A,$86,$4A,$85,$4A,$A6,$52,$C6,$52
	.DB $68,$6B,$6C,$84,$6C,$84,$8D,$8C,$29,$63,$A8,$52,$C9,$52,$68,$4A
	.DB $47,$42,$06,$3A,$C4,$31,$A8,$52,$05,$42,$05,$3A,$45,$42,$66,$4A
	.DB $66,$52,$08,$63,$A6,$5A,$07,$6B,$C5,$5A,$26,$6B,$88,$73,$8A,$6B
	.DB $4A,$63,$67,$42,$A8,$4A,$26,$3A,$A4,$29,$83,$21,$09,$53,$49,$5B
	.DB $23,$29,$63,$29,$02,$21,$C4,$39,$C4,$39,$25,$4A,$46,$4A,$E5,$39
	.DB $47,$42,$66,$42,$26,$42,$86,$42,$66,$42,$C7,$4A,$A9,$63,$2B,$74
	.DB $E3,$18,$65,$29,$C3,$18,$04,$21,$82,$10,$E4,$18,$24,$21,$23,$21
	.DB $A5,$31,$46,$4A,$C4,$31,$65,$42,$44,$42,$88,$63,$6B,$84,$6B,$84
	.DB $C6,$31,$49,$42,$A7,$31,$86,$29,$A3,$08,$C4,$10,$E4,$10,$C3,$18
	.DB $85,$31,$4B,$6B,$2D,$84,$4D,$84,$67,$63,$C7,$73,$6A,$84,$08,$7C
	.DB $48,$42,$89,$42,$07,$32,$48,$42,$A5,$29,$44,$19,$C2,$10,$C3,$18
	.DB $03,$21,$C5,$39,$09,$63,$49,$6B,$A9,$73,$C8,$7B,$6A,$8C,$8A,$8C
	.DB $68,$4A,$4B,$63,$C9,$52,$C9,$52,$A4,$29,$02,$19,$81,$08,$61,$08
	.DB $E3,$18,$64,$29,$C9,$52,$29,$63,$4D,$8C,$CA,$73,$E6,$5A,$6B,$8C
	.DB $88,$4A,$CD,$73,$2A,$63,$68,$4A,$02,$19,$E2,$18,$03,$21,$E3,$18
	.DB $03,$19,$24,$21,$44,$21,$43,$29,$67,$4A,$C8,$5A,$25,$42,$C7,$5A
	.DB $47,$4A,$0A,$63,$27,$42,$A6,$39,$E3,$20,$65,$31,$08,$42,$04,$19
	.DB $24,$19,$48,$42,$07,$3A,$85,$31,$03,$21,$06,$42,$06,$42,$06,$42
	.DB $4B,$BD,$C9,$AC,$2B,$B5,$2A,$AD,$C9,$A4,$2A,$AD,$C9,$9C,$A8,$94
	.DB $A8,$9C,$6B,$AD,$ED,$BD,$6B,$B5,$0A,$A5,$4B,$AD,$8C,$B5,$4B,$AD
	.DB $2B,$BD,$89,$A4,$C9,$A4,$EA,$AC,$EA,$A4,$8C,$B5,$4B,$AD,$2A,$A5
	.DB $2A,$A5,$8C,$B5,$CC,$BD,$4B,$AD,$2A,$AD,$6B,$B5,$8C,$B5,$6C,$B5
	.DB $EA,$AC,$68,$9C,$88,$9C,$E9,$A4,$0A,$AD,$8C,$B5,$6C,$B5,$6B,$AD
	.DB $6B,$AD,$8B,$B5,$6B,$B5,$0A,$A5,$4A,$AD,$8C,$B5,$6B,$B5,$8C,$B5
	.DB $A9,$A4,$A8,$9C,$C9,$A4,$0A,$A5,$0A,$AD,$2A,$AD,$0A,$A5,$4A,$AD
	.DB $6B,$AD,$8B,$B5,$4A,$AD,$0A,$A5,$6B,$B5,$AC,$B5,$4A,$AD,$2A,$AD
	.DB $A9,$9C,$E9,$A4,$EA,$A4,$0A,$AD,$0A,$AD,$C9,$A4,$C9,$A4,$2A,$AD
	.DB $6B,$B5,$AC,$BD,$8B,$B5,$2A,$AD,$8B,$B5,$8C,$B5,$E9,$A4,$C9,$9C
	.DB $C9,$9C,$0A,$AD,$EA,$A4,$EA,$A4,$0A,$AD,$C9,$A4,$0A,$A5,$0A,$A5
	.DB $4B,$AD,$AC,$B5,$8B,$B5,$2A,$A5,$2A,$AD,$0A,$A5,$88,$94,$C9,$9C
	.DB $C9,$9C,$6B,$AD,$0A,$A5,$E9,$A4,$0A,$A5,$A8,$9C,$0A,$AD,$E9,$A4
	.DB $0A,$A5,$6B,$AD,$4A,$AD,$C9,$9C,$C9,$9C,$C9,$9C,$67,$94,$2A,$A5
	.DB $C9,$9C,$8C,$B5,$4B,$AD,$E9,$A4,$E9,$A4,$47,$94,$A9,$9C,$E9,$A4
	.DB $09,$A5,$2A,$AD,$2A,$A5,$A8,$9C,$A8,$9C,$E9,$9C,$C9,$9C,$2A,$AD
	.DB $0A,$A5,$4B,$AD,$2A,$A5,$0A,$A5,$0A,$A5,$EA,$9C,$A9,$94,$0A,$A5
	.DB $AD,$B5,$AD,$BD,$4B,$AD,$0A,$A5,$E9,$A4,$C9,$9C,$C9,$9C,$E9,$A4
	.DB $2B,$AD,$0A,$A5,$C9,$9C,$A9,$9C,$EA,$A4,$2B,$AD,$4B,$AD,$8C,$B5
	.DB $AD,$BD,$4B,$AD,$A9,$9C,$A8,$9C,$C9,$9C,$88,$9C,$88,$9C,$E9,$A4
	.DB $4B,$AD,$2A,$AD,$C9,$A4,$A9,$9C,$EA,$A4,$4B,$B5,$8C,$BD,$6C,$B5
	.DB $6C,$B5,$C9,$9C,$68,$94,$E9,$9C,$2A,$AD,$0A,$AD,$EA,$A4,$4B,$B5
	.DB $4B,$B5,$6C,$B5,$4B,$B5,$2B,$AD,$2B,$AD,$2B,$B5,$4B,$B5,$EA,$A4
	.DB $0A,$A5,$C9,$9C,$C9,$9C,$4B,$AD,$8C,$B5,$4B,$AD,$0A,$AD,$88,$9C
	.DB $6C,$B5,$AD,$C5,$CD,$C5,$AD,$BD,$4C,$B5,$0B,$AD,$0A,$AD,$C9,$A4
	.DB $0A,$A5,$EA,$A4,$EA,$A4,$2A,$AD,$0A,$AD,$A9,$9C,$88,$9C,$68,$9C
	.DB $8C,$B5,$AD,$BD,$CD,$C5,$AD,$BD,$4B,$B5,$0A,$AD,$0A,$AD,$EA,$A4
	.DB $2B,$AD,$0A,$AD,$C9,$A4,$C9,$A4,$A8,$9C,$88,$9C,$A9,$A4,$0A,$AD
	.DB $6C,$B5,$6B,$B5,$6C,$B5,$6C,$B5,$2B,$AD,$0A,$A5,$0A,$A5,$2B,$AD
	.DB $4B,$B5,$0A,$AD,$C9,$A4,$A9,$9C,$88,$9C,$A9,$A4,$2B,$AD,$88,$9C
	.DB $4B,$AD,$2A,$A5,$4B,$AD,$4B,$AD,$0A,$A5,$EA,$9C,$EA,$A4,$4B,$B5
	.DB $6C,$B5,$2B,$AD,$C9,$A4,$A9,$9C,$88,$9C,$88,$9C,$0A,$AD,$0A,$AD



; --------------------------------------
; SYSTEM VECTORS

        .ORG $FFFA

        .DB $0A, $00 ; /NMI - Non-Maskable Int ; Emulator opposite endian from 6502
        .DB $0A, $00 ; /RES - System reset ; Emulator opposite endian from 6502
;       .DB $00, $0A ; /RES - System reset
        .DB $0A, $00 ; /IRQ - Interrupt Request ; Emulator opposite endian from 6502

        .END
